package com.xzchaoo.commons.basic.config.bind;

import com.xzchaoo.commons.basic.unsafe.UnsafeUtils;

import java.lang.reflect.*;
import java.time.Duration;
import java.util.*;

/**
 * @author xzchaoo
 * <p> created at 2021/12/16
 */
public class Binder {

    public static Map<String, Object> convertToMapObject(Map<String, String> p) {
        Map<String, Object> map = new HashMap<>();
        for (Map.Entry<String, String> e : p.entrySet()) {
            set(map, e.getKey(), e.getValue(), 0);
        }
        return map;
    }

    private static void set(Map<String, Object> map, String key, String value, int offset) {
        int index = nextOffset(key, offset);

        int nextIndex;
        String subKey;

        if (index < 0) {
            nextIndex = -1;
            subKey = key.substring(offset);
        } else {
            subKey = key.substring(offset, index);
            nextIndex = index + 1;
            while (nextIndex < key.length()) {
                char c = key.charAt(nextIndex);
                if (c == '.' || c == '[' || c == ']') {
                    ++nextIndex;
                } else {
                    break;
                }
            }
            if (nextIndex == key.length()) {
                nextIndex = -1;
            }
        }
        subKey = format(subKey);
        if (nextIndex < 0) {
            map.put(subKey, value);
        } else {
            Object temp = map.get(subKey);
            if (temp == null) {
                temp = new HashMap<>();
                map.put(subKey, temp);
            } else {
                if (!(temp instanceof Map)) {
                    String invalidKey = key.substring(0, index);
                    throw new IllegalStateException("key [" + invalidKey + "] is prefix of other key");
                }
            }
            Map<String, Object> sub = (Map<String, Object>) temp;
            set(sub, key, value, nextIndex);
        }
    }

    static String format(String key) {
        StringBuilder sb = null;

        int offset = 0;
        while (offset < key.length()) {
            int index = key.indexOf('-', offset);
            if (index < 0) {
                if (offset == 0) {
                    return key;
                }
                break;
            }
            if (sb == null) {
                sb = new StringBuilder();
            }
            sb.append(key, offset, index);
            if (index + 2 < key.length() && Character.isLetter(key.charAt(index + 1))) {
                sb.append(Character.toUpperCase(key.charAt(index + 1)));
                offset = index + 2;
            } else {
                offset = index;
                break;
            }
        }
        sb.append(key, offset, key.length());

        return sb.toString();
    }

    private Object convertToSimple(String value, Type t) {
        if (!(t instanceof Class)) {
            throw new IllegalStateException("t must be a class instance");
        }

        if (t == String.class || t == Object.class) {
            return value;
        }
        if (t == Integer.class || t == int.class) {
            return Integer.parseInt(value);
        }
        if (t == Long.class || t == long.class) {
            return Long.parseLong(value);
        }
        if (t == Float.class || t == float.class) {
            return Float.parseFloat(value);
        }
        if (t == Double.class || t == double.class) {
            return Double.parseDouble(value);
        }
        if (t == Boolean.class || t == boolean.class) {
            return Boolean.parseBoolean(value);
        }
        if (t == Duration.class) {
            return parseDuration(value);
        }

        try {
            Class<?> c = (Class<?>) t;
            OfValueData ofValueData = OF_VALUE_DATA.get(c);
            if (ofValueData.ofValueMethod != null) {
                return ofValueData.ofValueMethod.invoke(null, value);
            }
        } catch (Throwable e) {
        }

        // convert map
        throw new RuntimeException("convert error");
    }

    private static Duration parseDuration(String str) {
        char u = str.charAt(str.length() - 1);
        long value = Long.parseLong(str.substring(0, str.length() - 1));
        switch (u) {
            case 's':
                return Duration.ofSeconds(value);
            case 'm':
                return Duration.ofMinutes(value);
            case 'h':
                return Duration.ofHours(value);
            default:
                throw new IllegalArgumentException("fail to parse " + str + " to Duration");
        }
    }

    private Object convertToKey(String value, Type type) {
        if (!(type instanceof Class)) {
            throw new IllegalStateException("Map key type must be a class instance");
        }

        Class c = (Class) type;

        // 99% case
        if (type == String.class) {
            return value;
        }

        OfValueData ovd = OF_VALUE_DATA.get(c);
        if (ovd.ofValueMethod != null) {
            try {
                return ovd.ofValueMethod.invoke(null, value);
            } catch (IllegalAccessException | InvocationTargetException e) {
                throw new IllegalStateException("fail to call ofValue to construct a key", e);
            }
        }

        throw new IllegalStateException("fail to create a key for type " + type);
    }

    private void bindSub(Map<String, ?> sub, Object obj, Type type) {
        if (obj instanceof Map) {
            if (!(type instanceof ParameterizedType)) {
                throw new RuntimeException("type must be ParameterizedType");
            }
            Type[] args = ((ParameterizedType) type).getActualTypeArguments();
            if (args[0] != String.class) {
                throw new IllegalStateException("Map key must be String");
            }
            Map mapObj = (Map) obj;
            for (Map.Entry<String, ?> e : sub.entrySet()) {
                Object mapKey = convertToKey(e.getKey(), args[0]);
                if (e.getValue() instanceof Map) {
                    try {
                        Object mapValue = ((Class) args[1]).newInstance();
                        bindSub((Map) e.getValue(), mapValue, args[1]);
                        mapObj.put(mapKey, mapValue);
                    } catch (Throwable ex) {
                        ex.printStackTrace();
                    }
                } else {
                    Object mapValue = convertToSimple((String) e.getValue(), args[1]);
                    mapObj.put(mapKey, mapValue);
                }
            }
            return;
        }

        // TODO cache
        // 有没有办法lazy resolver sub
        ClassData cd = CV.get(obj.getClass());
        for (FieldData field : cd.fields) {
            Object value = sub.get(field.field.getName());
            if (value == null) {
                // 这里不要将字段set成null
                continue;
            }

            Object fieldValue = convertToFieldValue(value, obj, field);
            field.set(obj, fieldValue);
        }
    }

    private Object convertToFieldValue(Object value, Object obj, FieldData field) {
        Object fieldValue = field.get(obj);

        if (field.simple) {
            if (value instanceof String) {
                fieldValue = convertToSimple((String) value, field.field.getType());
            } else {
                throw new IllegalStateException("value must be String " + value);
            }
        } else if (field.collection) {
            Type t = field.field.getGenericType();
            if (!(t instanceof ParameterizedType)) {
                throw new RuntimeException("field must be a Collection<T>");
            }
            Type valueType = ((ParameterizedType) t).getActualTypeArguments()[0];

            if (!(value instanceof String)) {
                throw new IllegalStateException("value must be a String");
            }

            if (fieldValue == null) {
                fieldValue = newCollectionInstance(field);
            }
            Collection<Object> coll = (Collection) fieldValue;
            coll.clear();

            String separator = ",";
            ListValue listValueAn = field.field.getAnnotation(ListValue.class);
            if (listValueAn != null) {
                separator = listValueAn.separator();
            }
            String[] ss = ((String) value).split(separator);
            for (String s : ss) {
                coll.add(convertToSimple(s, valueType));
            }
        } else if (field.array) {
            fieldValue = convertToArray(value, field);
        } else {
            if (fieldValue == null) {
                // create if null
                fieldValue = newFieldValue(field);
            }

            if (value instanceof String) {
                String sep1 = "=";
                String sep2 = "^";
                MapValue mv = field.field.getAnnotation(MapValue.class);
                if (mv != null) {
                    sep1 = mv.sep1();
                    sep2 = mv.sep2();
                }
                Map<String, String> parsedMap = parseToMap((String) value, sep1, sep2);
                bindSub(parsedMap, fieldValue, field.field.getGenericType());
            } else if (value instanceof Map) {
                bindSub((Map<String, Object>) (value), fieldValue, field.field.getGenericType());
            } else {
                throw new IllegalStateException("value must be map " + value);
            }

            try {
                Method afterPropertiesSetMethod = fieldValue.getClass().getDeclaredMethod("afterPropertiesSet");
                afterPropertiesSetMethod.invoke(fieldValue);
            } catch (NoSuchMethodException e) {
                // ignored
            } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
                throw new IllegalStateException(e);
            }

            return fieldValue;
        }

        // fieldValue
        return fieldValue;
    }

    private static Map<String, String> parseToMap(String str, String sep1, String sep2) {
        Map<String, String> map = new HashMap<>();
        int offset = 0;

        while (true) {
            int index1 = str.indexOf(sep1, offset);
            if (index1 < 0) {
                break;
            }
            int index2 = str.indexOf(sep2, index1 + sep1.length());
            if (index2 < 0) {
                index2 = str.length();
            }
            String key = str.substring(offset, index1);
            String value = str.substring(index1 + sep1.length(), index2);
            map.put(key, value);
            offset = index2 + sep2.length();
            if (index2 == str.length()) {
                break;
            }
        }

        return map;
    }

    private Object convertToArray(Object value, FieldData field) {
        Object fieldValue;
        Class<?> componentType = field.componentType;
        String separator = ",";
        ListValue listValueAn = field.field.getAnnotation(ListValue.class);
        if (listValueAn != null) {
            separator = listValueAn.separator();
        }
        String[] ss = ((String) value).split(separator);
        if (componentType == int.class) {
            int[] array = new int[ss.length];
            for (int i = 0; i < ss.length; i++) {
                array[i] = (Integer) convertToSimple(ss[i], componentType);
            }
            fieldValue = array;
        } else if (componentType == long.class) {
            long[] array = new long[ss.length];
            for (int i = 0; i < ss.length; i++) {
                array[i] = (Long) convertToSimple(ss[i], componentType);
            }
            fieldValue = array;
        } else if (componentType == short.class) {
            short[] array = new short[ss.length];
            for (int i = 0; i < ss.length; i++) {
                array[i] = (Short) convertToSimple(ss[i], componentType);
            }
            fieldValue = array;
        } else if (componentType == boolean.class) {
            boolean[] array = new boolean[ss.length];
            for (int i = 0; i < ss.length; i++) {
                array[i] = (Boolean) convertToSimple(ss[i], componentType);
            }
            fieldValue = array;
        } else if (componentType == float.class) {
            float[] array = new float[ss.length];
            for (int i = 0; i < ss.length; i++) {
                array[i] = (Float) convertToSimple(ss[i], componentType);
            }
            fieldValue = array;
        } else if (componentType == double.class) {
            double[] array = new double[ss.length];
            for (int i = 0; i < ss.length; i++) {
                array[i] = (Double) convertToSimple(ss[i], componentType);
            }
            fieldValue = array;
        } else {
            Object[] array = (Object[]) Array.newInstance(componentType, ss.length);
            for (int i = 0; i < ss.length; i++) {
                array[i] = convertToSimple(ss[i], componentType);
            }
            fieldValue = array;
        }
        return fieldValue;
    }

    private Object newFieldValue(FieldData fd) {
        Class<?> t = fd.field.getType();
        if (Modifier.isAbstract(t.getModifiers())) {
            throw new IllegalArgumentException("type [" + t.getName() + "] is abstract");
        }

        try {
            return t.newInstance();
        } catch (InstantiationException | IllegalAccessException e) {
            throw new IllegalStateException("fail to create field value", e);
        }
    }

    private Object newCollectionInstance(FieldData fd) {
        Class<?> t = fd.field.getType();
        // 具体类型直接new
        if (!t.isInterface() && !t.isAnnotation() && !Modifier.isAbstract(t.getModifiers())) {
            try {
                return t.newInstance();
            } catch (InstantiationException | IllegalAccessException e) {
                throw new IllegalStateException("fail to new collection instance", e);
            }
        }
        if (t == Collection.class || List.class.isAssignableFrom(t)) {
            return new ArrayList<>();
        }
        if (Set.class.isAssignableFrom(t)) {
            return new HashSet<>();
        }
        throw new IllegalStateException("fail to new collection instance");
    }

    /**
     * Find next offset of '[' or ']' or '.'
     *
     * @param str
     * @param offset starting offset
     * @return
     */
    private static int nextOffset(String str, int offset) {
        for (int i = offset; i < str.length(); i++) {
            char c = str.charAt(i);
            if (c == '[' || c == ']' || c == '.') {
                return i;
            }
        }
        return -1;
    }

    /**
     * Bind sub map to obj
     *
     * @param map    property map
     * @param prefix sub map path
     * @param obj    object
     */
    public void bind(Map<String, Object> map, String prefix, Object obj) {
        int offset = 0;
        if (prefix.startsWith("[") || prefix.startsWith(".")) {
            offset = 1;
        }
        while (true) {
            int index = nextOffset(prefix, offset);
            if (index < 0) {
                index = prefix.length();
            }
            String key = prefix.substring(offset, index);
            Object sub = map.get(key);
            if (sub == null) {
                return;
            }
            if (!(sub instanceof Map)) {
                throw new RuntimeException("invalid path " + prefix + " is not a map");
            }
            offset = index + 1;
            while (offset < prefix.length()) {
                char c = prefix.charAt(offset);
                if (c == '[' || c == ']' || c == '.') {
                    ++offset;
                } else {
                    break;
                }
            }
            if (offset == prefix.length()) {
                Map<String, Object> subMap = (Map<String, Object>) sub;
                if (!subMap.isEmpty()) {
                    bindSub(subMap, obj, obj.getClass());
                }
                return;
            }
        }
    }

    private static final ClassValue<ClassData> CV = new ClassValue<ClassData>() {
        @Override
        protected ClassData computeValue(Class<?> type) {
            ClassData cd = new ClassData();

            List<FieldData> fds = new ArrayList<>();
            Class<?> t = type;
            while (t != null && t != Object.class) {
                for (Field f : t.getDeclaredFields()) {
                    if (!f.isAccessible()) {
                        f.setAccessible(true);
                    }
                    FieldData fd = new FieldData();
                    fd.field = f;
                    fd.init();
                    fds.add(fd);
                }
                t = t.getSuperclass();
            }
            cd.fields = fds.toArray(new FieldData[fds.size()]);
            return cd;
        }
    };

    private static class ClassData {
        FieldData[] fields;
    }

    private static class FieldData {
        Field field;
        long fieldOffset;
        boolean simple;
        boolean collection;
        boolean primitive;
        boolean array;
        Class<?> componentType;

        void init() {
            fieldOffset = UnsafeUtils.getUnsafe().objectFieldOffset(field);
            Class<?> t = field.getType();
            simple = SIMPLE_TYPES.contains(t);
            collection = Collection.class.isAssignableFrom(t);
            primitive = PRIMITIVE_TYPES.contains(t);
            array = t.isArray();
            if (array) {
                componentType = t.getComponentType();
            }
        }

        Object get(Object obj) {
            if (primitive) {
                try {
                    return field.get(obj);
                } catch (IllegalAccessException e) {
                    throw new IllegalStateException("fail to get field", e);
                }
            } else {
                return UnsafeUtils.getUnsafe().getObject(obj, fieldOffset);
            }
        }

        void set(Object obj, Object value) {
            if (primitive) {
                try {
                    field.set(obj, value);
                } catch (IllegalAccessException e) {
                    throw new IllegalStateException("fail to set field", e);
                }
            } else {
                UnsafeUtils.getUnsafe().putObject(obj, fieldOffset, value);
            }
        }
    }

    private static final Set<Class<?>> SIMPLE_TYPES = new HashSet<>(Arrays.asList(
            String.class, //
            int.class, Integer.class, //
            long.class, Long.class, //
            float.class, Float.class, //
            double.class, Double.class, //
            boolean.class, Boolean.class, //
            byte.class, Byte.class, //
            char.class, Character.class, //
            short.class, Short.class, //
            Duration.class
    ));

    private static final Set<Class<?>> PRIMITIVE_TYPES = new HashSet<>(Arrays.asList(
            int.class, //
            long.class,//
            float.class, //
            double.class, //
            boolean.class, //
            char.class, //
            byte.class, //
            short.class
    ));

    private static final ClassValue<OfValueData> OF_VALUE_DATA = new ClassValue<OfValueData>() {
        @Override
        protected OfValueData computeValue(Class<?> type) {
            OfValueData d = new OfValueData();
            try {
                Method method = type.getDeclaredMethod("ofValue", String.class);
                Class<?> returnType = method.getReturnType();
                if (type.isAssignableFrom(returnType)) {
                    d.ofValueMethod = method;
                }
            } catch (NoSuchMethodException e) {
            }
            return d;
        }
    };

    private static class OfValueData {
        Method ofValueMethod;
    }
}
