package com.xzchaoo.commons.basic.diff;

import com.xzchaoo.commons.basic.function.TriConsumer;

import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Predicate;

/**
 * Diff utilities
 *
 * @author xzchaoo
 * <p> created at 2020/9/23
 */
public final class DiffUtils {
    private DiffUtils() {
    }

    /**
     * Diff two {@link Iterable}s.
     *
     * @param left         left iterable
     * @param right        right iterable
     * @param leftContain  left contain predicate
     * @param rightContain right contain predicate
     * @param leftOnly     left-only consumer
     * @param rightOnly    right-only consumer
     * @param intersection intersection consumer
     * @param <T>          element type
     */
    public static <T> void diff(Iterable<T> left, //
                                Iterable<T> right, //
                                Predicate<T> leftContain, //
                                Predicate<T> rightContain, //
                                Consumer<T> leftOnly, //
                                Consumer<T> rightOnly, //
                                Consumer<T> intersection) {
        for (T t : left) {
            if (rightContain.test(t)) {
                intersection.accept(t);
            } else {
                leftOnly.accept(t);
            }
        }
        for (T t : right) {
            if (!leftContain.test(t)) {
                rightOnly.accept(t);
            }
        }
    }

    /**
     * Diff two map
     *
     * @param left         left map
     * @param right        right map
     * @param leftOnly     left-only consumer
     * @param rightOnly    right-only consumer
     * @param intersection intersection consumer
     * @param <K>          key type
     * @param <V>          value type
     */
    public static <K, V> void diff2(Map<K, V> left, //
                                    Map<K, V> right, //
                                    BiConsumer<K, V> leftOnly, //
                                    BiConsumer<K, V> rightOnly, //
                                    TriConsumer<K, V, V> intersection) {
        for (Map.Entry<K, V> e : left.entrySet()) {
            K k = e.getKey();
            V v = e.getValue();
            if (right.containsKey(k)) {
                intersection.accept(k, v, right.get(k));
            } else {
                leftOnly.accept(k, v);
            }
        }
        for (Map.Entry<K, V> e : right.entrySet()) {
            if (!left.containsKey(e.getKey())) {
                rightOnly.accept(e.getKey(), e.getValue());
            }
        }
    }
}
