package com.xzchaoo.commons.basic.drainloop;

import org.jctools.queues.MpscChunkedArrayQueue;

/**
 * 该demo介绍如何使用DrainLoop模型.
 * <p>
 * 假设该数据结构需要串行化4种动作:
 * 1. 动作1: 它的作用是产生2个动作2
 * 2. 动作2: 它的作用是产生2个动作3
 * 3. 动作3: 它是一个异步任务, 有最大并发度(maxWip)限制, 执行完毕之后会产生ack
 * 4. 动作3的ack
 *
 * <p>created at 2020-08-11
 *
 * @author xiangfeng.xzc
 */
public class DrainLoopDemo extends DrainLoop {
    private final MpscChunkedArrayQueue<Action1> a1q
        = new MpscChunkedArrayQueue<>(65536);

    private final MpscChunkedArrayQueue<Action2> a2q
        = new MpscChunkedArrayQueue<>(65536);

    private final MpscChunkedArrayQueue<Action3> a3q
        = new MpscChunkedArrayQueue<>(65536);

    /**
     * 该队列的大小其实取决于最大并发度, 没必要取65536
     */
    private final MpscChunkedArrayQueue<Action3Ack> a3Ack
        = new MpscChunkedArrayQueue<>(65536);

    private int maxWip = 4;
    private int wip;

    @Override
    protected void drainLoop0() {
        for (; ; ) {
            boolean changed = false;
            changed |= consumeA1();
            changed |= consumeA2();
            changed |= consumeA3();
            changed |= consumeA3Ack();
            if (!changed) {
                break;
            }
        }
    }

    /**
     * @return 如果本次执行会导致该数据结构的任务增多, 那么就返回true
     */
    private boolean consumeA1() {
        boolean changed = false;
        for (; ; ) {
            Action1 a1 = a1q.relaxedPoll();
            if (a1 == null) {
                break;
            }
            changed = true;
            for (int i = 0; i < 2; i++) {
                a2q.offer(new Action2());
            }
        }
        return changed;
    }

    private boolean consumeA2() {
        boolean changed = false;
        for (; ; ) {
            Action2 a2 = a2q.relaxedPoll();
            if (a2 == null) {
                break;
            }
            changed = true;
            for (int i = 0; i < 2; i++) {
                a3q.offer(new Action3());
            }
        }
        return changed;
    }

    private boolean consumeA3() {
        // a3 不会产生新的任务 故总是返回false
        for (; ; ) {
            // a3 有并发度限制
            if (wip < maxWip) {
                Action3 a3 = a3q.relaxedPoll();
                if (a3 == null) {
                    return false;
                }
                ++wip;
            } else {
                break;
            }
        }
        return false;
    }

    private boolean consumeA3Ack() {
        for (; ; ) {
            Action3Ack ack = a3Ack.relaxedPoll();
            if (ack == null) {
                break;
            }

            // 方案1: begin
            // --wip;
            // 方案1: end

            // 因为这是a3的ack, 如果此时a3有积压, 那么是可以立即处理的, 没必要等到 consumeA3 再去处理,
            // 当然想等也是可以的.
            // 方案2: begin
            Action3 a3 = a3q.relaxedPoll();
            if (a3 != null) {
                // execute a3
            } else {
                --wip;
            }
            // 方案2: end
        }
        return false;
    }

    public void a3Ack() {
        a3Ack.offer(new Action3Ack());
        drainLoop();
    }

    private static class Action1 {
    }

    private static class Action2 {
    }

    private static class Action3 {
    }

    private static class Action3Ack {
    }
}
