package com.xzchaoo.commons.basic;

import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Interface indicating something is done.
 * <p>created at 2020/7/26
 *
 * @author xzchaoo
 */
@FunctionalInterface
public interface Ack {
    /**
     * Returns an ack instance that calls runnable only once when it's {@link
     * #ack()} is called.
     *
     * @param runnable runnable
     * @return ack
     */
    static Ack once(Runnable runnable) {
        return new OnceAck(runnable);
    }

    static Ack noop() {
        return Noop.INSTANCE;
    }

    /**
     * Indicate that something is done. Implementation may be idempotent.
     */
    void ack();

    /**
     * @param executor
     * @param task
     * @deprecated use {@link Acks#execute(Ack, Executor, Runnable)}
     */
    @Deprecated
    default void safelyExecute(Executor executor, Runnable task) {
        try {
            executor.execute(() -> {
                try {
                    task.run();
                } finally {
                    ack();
                }
            });
        } catch (Throwable e) {
            ack();
            throw e;
        }
    }

    final class Noop implements Ack {
        public static final Noop INSTANCE = new Noop();

        private Noop() {}

        @Override
        public void ack() {
        }
    }

    /**
     * Ack implementation that call runnable only once when {@link #ack()} is
     * called.
     */
    @SuppressWarnings("WeakerAccess")
    final class OnceAck extends AtomicReference<Runnable> implements Ack {

        public OnceAck(Runnable runnable) {
            lazySet(Objects.requireNonNull(runnable));
        }

        @Override
        public void ack() {
            Runnable task = getAndSet(null);
            if (task != null) {
                task.run();
            }
        }
    }
}
