package com.xzchaoo.commons.basic.collection;

import lombok.Getter;
import lombok.Setter;

import java.util.AbstractList;
import java.util.Arrays;

/**
 * 在一些序列化(特别是json)的场景下, 数组Object[]会被按原本大小进行序列化, 这就要求用户使用大小刚好的数组.
 * 一个解决办法显然是使用ArrayList, 或使用 Arrays.asList().subList() 进行包装.
 * 于是这里提供了该类, 可以用于包装一个Object[]为List, 并且可以方便地调整大小.
 *
 * @author xzchaoo
 * <p> created at 2021/3/5
 */
@Getter
@Setter
public class ArrayListWrapper<E> extends AbstractList<E> {
    private Object[] array;
    private int offset;
    private int size;

    public ArrayListWrapper(int length) {
        this(new Object[length]);
    }

    public ArrayListWrapper(Object[] array) {
        this(array, 0, array.length);
    }

    public ArrayListWrapper(Object[] array, int offset, int size) {
        if (offset + size > array.length) {
            throw new IllegalArgumentException("offset + size > array.length");
        }
        this.array = array;
        this.offset = offset;
        this.size = size;
    }

    @Override
    public E get(int index) {
        // no index check
        return (E) array[offset + index];
    }

    @Override
    public int size() {
        return size;
    }

    @Override
    public void clear() {
        Arrays.fill(array, offset, offset + size, null);
    }

    public void deepClear() {
        Arrays.fill(array, null);
    }
}
