package com.xzchaoo.commons.basic.concurrent;

import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.concurrent.ThreadSafe;

import org.jctools.queues.MessagePassingQueue;
import org.jctools.queues.MpscChunkedArrayQueue;
import org.jctools.queues.MpscUnboundedArrayQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 提交的任务可以保证执行是串行的, 但是不一定在同一个线程上. 如果切换线程, 那么会保证可见性的.
 * copy from io.grpc.internal.SerializingExecutor
 * <p>created at 2020-11-02
 *
 * @author xiangfeng.xzc
 */
@ThreadSafe
public class SerializingExecutor implements Executor {
    private static final Logger LOGGER = LoggerFactory.getLogger(SerializingExecutor.class);
    private static final int CHUNK_SIZE = 4096;
    private final Executor executor;

    private final MessagePassingQueue<Runnable> queue;

    private final AtomicInteger wip = new AtomicInteger(0);

    public SerializingExecutor(Executor executor) {
        this(executor, CHUNK_SIZE, 0);
    }

    /**
     * 不指定大小就很危险
     *
     * @param executor
     * @param maxCapacity
     */
    public SerializingExecutor(Executor executor, int chunkSize, int maxCapacity) {
        this.executor = Objects.requireNonNull(executor);
        if (maxCapacity == 0) {
            this.queue = new MpscUnboundedArrayQueue<>(chunkSize);
        } else {
            this.queue = new MpscChunkedArrayQueue<>(chunkSize, maxCapacity);
        }
    }

    @Override
    public void execute(Runnable task) {
        if (!queue.offer(Objects.requireNonNull(task, "task is null"))) {
            throw new RejectedExecutionException("Task " + task + " rejected from " + this);
        }
        // 使用wip模式确保只有一个人在执行
        if (wip.getAndIncrement() != 0) {
            return;
        }

        try {
            executor.execute(this::drain);
        } catch (Throwable e) {
            // fallback
            this.drain();
            LOGGER.error("Exception caught when schedule drain", e);
        }
    }

    private void drain() {
        int delta = wip.get();
        do {
            Runnable r;
            while ((r = queue.poll()) != null) {
                try {
                    r.run();
                } catch (Throwable e) {
                    LOGGER.error("Exception while executing runnable " + r, e);
                }
            }
            delta = wip.addAndGet(-delta);
        } while (delta != 0);
    }
}
