package com.xzchaoo.commons.basic.concurrent;

import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;

import javax.annotation.Nonnull;

import org.jctools.queues.*;
import org.jctools.queues.atomic.MpscAtomicArrayQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Learn from grpc, io.grpc.SynchronizationContext.
 * All tasks submitted will be executed in their submit order serially. Tasks may be executed in different threads.
 * This instance becomes <strong>broken</strong> if any exceptions occur.
 * <ul>
 * <li>Ordering</li>
 * <li>Serialization</li>
 * <li>Non-reentrancy</li>
 * </ul>
 *
 * <p>created at 2020-10-30
 *
 * @author xiangfeng.xzc
 */
public class SynchronizationContext implements Executor {
    private static final Logger LOGGER = LoggerFactory.getLogger(SynchronizationContext.class);

    private static final AtomicReferenceFieldUpdater<SynchronizationContext, Thread> THREAD_UPDATER
            = AtomicReferenceFieldUpdater
            .newUpdater(SynchronizationContext.class, Thread.class, "thread");

    /**
     * TODO: use unlimited queue?
     */
    private final MessagePassingQueue<Runnable> q;
    private volatile Thread thread;

    private SynchronizationContext(MessagePassingQueue<Runnable> q) {
        this.q = q;
    }

    public static SynchronizationContext create() {
        return new SynchronizationContext(new MpscLinkedQueue<>());
    }

    public static SynchronizationContext create(int chunkSize) {
        return new SynchronizationContext(new MpscUnboundedArrayQueue<>(chunkSize));
    }

    public static SynchronizationContext create(int initialCapacity, int maxCapacity) {
        return new SynchronizationContext(new MpscChunkedArrayQueue<>(initialCapacity, maxCapacity));
    }

    /**
     * Just add a task to queue
     *
     * @param command
     */
    @Override
    public void execute(@Nonnull Runnable command) {
        executeLater(command);
        drain();
    }

    /**
     * Just add a task to queue.
     *
     * @param command
     */
    public void executeLater(@Nonnull Runnable command) {
        if (!q.offer(command)) {
            throw new IllegalStateException("queue is full");
        }
    }

    public boolean isCurrentThreadInContext() {
        return this.thread == Thread.currentThread();
    }

    public void drain() {
        // 这是与 基于wip的DrainLoop 相对应的另外一种方式, 利用Thread的CAS竞争
        Thread thread = Thread.currentThread();
        MessagePassingQueue<Runnable> q = this.q;
        do {
            if (!THREAD_UPDATER.compareAndSet(this, null, thread)) {
                return;
            }
            Runnable r;
            while ((r = q.poll()) != null) {
                try {
                    // user task should never throws exception
                    r.run();
                } catch (Throwable e) {
                    LOGGER.error("Exception caught when run task", e);
                }
            }
            THREAD_UPDATER.set(this, null);
        } while (!q.isEmpty());
    }
}
