package com.xzchaoo.commons.basic.objectconfig;

import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;

import lombok.extern.slf4j.Slf4j;

/**
 * @author xzchaoo
 */
@Slf4j
class BizConfigHolderImpl<T> implements BizConfigHolder<T> {
  private final    Config                         config;
  private final    String                         jsonPointer;
  private final    Class<T>                       clazz;
  private final    Runnable                       unListen;
  private volatile T                              biz;
  private final    CopyOnWriteArrayList<Runnable> listeners = new CopyOnWriteArrayList<>();

  BizConfigHolderImpl(Config config, String jsonPointer, Class<T> clazz) {
    this.config = Objects.requireNonNull(config);
    this.jsonPointer = jsonPointer;
    this.clazz = Objects.requireNonNull(clazz);
    // 第一次如果失败则抛异常
    update();
    // 后续失败会被 config 里捕获
    unListen = config.listen(this::update);
  }

  @Override
  public T get() {
    return biz;
  }

  @Override
  public void unbind() {
    unListen.run();
  }

  @Override
  public Runnable listen(Runnable listener) {
    listeners.add(listener);
    return () -> listeners.remove(listener);
  }

  private void update() {
    T newBiz = ConfigFactory.bindOnce(config.get(), jsonPointer, clazz);
    if (newBiz instanceof BizConfigHooks) {
      BizConfigHooks h = (BizConfigHooks) newBiz;
      h.validate();
    }
    this.biz = newBiz;
    fire();
  }

  private void fire() {
    for (Runnable listener : listeners) {
      try {
        listener.run();
      } catch (Exception e) {
        log.error("listener error", e);
      }
    }
  }
}
