package com.xzchaoo.commons.basic.objectconfig;

import java.io.File;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import lombok.SneakyThrows;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.dataformat.yaml.YAMLMapper;

/**
 * @author xzchaoo
 */
public final class ConfigFactory {
  public static final ObjectMapper JSON_OM = new ObjectMapper();
  public static final YAMLMapper   YAML_M  = new YAMLMapper();

  static {
    JSON_OM.findAndRegisterModules();
    JSON_OM.disable(SerializationFeature.FAIL_ON_EMPTY_BEANS);
    JSON_OM.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);

    YAML_M.findAndRegisterModules();
    YAML_M.disable(SerializationFeature.FAIL_ON_EMPTY_BEANS);
    YAML_M.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
  }

  private ConfigFactory() {
  }

  public static Config fromNode(ObjectNode node) {
    return new ConstConfig(node.deepCopy());
  }

  @SneakyThrows
  public static Config fromJson(File file) {
    return fromNode(JSON_OM.readTree(file), false);
  }

  @SneakyThrows
  public static Config fromJson(String content) {
    return fromNode(JSON_OM.readTree(content), false);
  }

  @SneakyThrows
  public static Config fromYaml(File file) {
    return fromNode(YAML_M.readTree(file), false);
  }

  @SneakyThrows
  public static Config fromYaml(String content) {
    return fromNode(YAML_M.readTree(content), false);
  }

  @SneakyThrows
  public static Config fromYaml(InputStream is) {
    return fromNode(YAML_M.readTree(is), false);
  }

  public static Config fromNode(JsonNode node) {
    return fromNode(node, true);
  }

  private static Config fromNode(JsonNode node, boolean copy) {
    if (node.isEmpty()) {
      node = JSON_OM.createObjectNode();
    }
    if (!node.isObject()) {
      throw new IllegalStateException("node is not ObjectNode");
    }
    return new ConstConfig(copy ? node.deepCopy() : (ObjectNode) node);
  }

  public static Config composite(Config... configs) {
    return new Composite(new ArrayList<>(Arrays.asList(configs)));
  }

  public static Config composite(List<Config> configs) {
    return new Composite(new ArrayList<>(configs));
  }

  @SneakyThrows
  public static <T> T bindOnce(ObjectNode node, String jsonPointer, Class<T> clazz) {
    JsonNode at = jsonPointer != null ? node.at(jsonPointer) : node;
    if (at.isMissingNode()) {
      return clazz.getDeclaredConstructor().newInstance();
    }
    if (!at.isObject()) {
      throw new IllegalStateException("node is not ObjectNode");
    }
    return bindOnce((ObjectNode) at, clazz);
  }

  @SneakyThrows
  public static <T> T bindOnce(ObjectNode node, Class<T> clazz) {
    return JSON_OM.treeToValue(node, clazz);
  }

  @SneakyThrows
  public static <T> T bindOnce(Config cfg, Class<T> clazz) {
    return JSON_OM.treeToValue(cfg.get(), clazz);
  }

  @SneakyThrows
  public static <T> BizConfigHolder<T> bind(Config config, String jsonPointer, Class<T> clazz) {
    return new BizConfigHolderImpl<>(config, jsonPointer, clazz);
  }
}
