package com.xzchaoo.commons.concurrent;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author xzchaoo
 */
public class KeyedExecutorImpl<Key> implements KeyedExecutor<Key> {
  private static final Logger log = LoggerFactory.getLogger(KeyedExecutorImpl.class);

  private final Executor        executor;
  private final Lock            lock   = new ReentrantLock();
  final         Map<Key, Queue> queues = new HashMap<>();

  public KeyedExecutorImpl(Executor executor) {
    this.executor = Objects.requireNonNull(executor);
  }

  @Override
  public void execute(Key key, Runnable r) {
    lock.lock();
    try {
      Queue q = queues.get(key);
      if (q == null) {
        Queue q2 = new Queue(key);
        queues.put(key, q2);
        executor.execute(() -> q2.drain(r));
      } else {
        // lazy create q.pending
        if (q.pending == null) {
          q.pending = new LinkedBlockingQueue<>();
        }
        q.pending.add(r);
      }

    } finally {
      lock.unlock();
    }
  }

  private boolean remove(Queue q) {
    lock.lock();
    try {
      if (q.pending == null || q.pending.isEmpty()) {
        queues.remove(q.key);
        return true;
      }
    } finally {
      lock.unlock();
    }
    return false;
  }

  private class Queue {
    final Key key;
    LinkedBlockingQueue<Runnable> pending;

    Queue(Key key) {
      this.key = key;
    }

    void drain(Runnable first) {
      // quick path
      run(first);
      if (remove(this)) {
        return;
      }

      // slow path
      while (true) {
        Runnable r = pending.poll();
        if (r != null) {
          run(r);
        } else if (remove(this)) {
          break;
        }
      }
    }

    void run(Runnable r) {
      try {
        r.run();
      } catch (Exception e) {
        log.error("run error", e);
      }
    }
  }
}
