/*
 * Decompiled with CFR 0.152.
 */
package com.oath.auth;

import com.oath.auth.CaCertKeyStoreProvider;
import com.oath.auth.JavaKeyStoreProvider;
import com.oath.auth.KeyManagerProxy;
import com.oath.auth.KeyRefresher;
import com.oath.auth.KeyRefresherException;
import com.oath.auth.KeyRefresherListener;
import com.oath.auth.TrustManagerProxy;
import com.oath.auth.TrustStore;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Paths;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Utils {
    private static final Logger LOG = LoggerFactory.getLogger(Utils.class);
    private static final String SSLCONTEXT_ALGORITHM_TLS12 = "TLSv1.2";
    private static final String SSLCONTEXT_ALGORITHM_TLS13 = "TLSv1.3";
    private static final String PROP_KEY_WAIT_TIME = "athenz.cert_refresher.key_wait_time";
    private static final String PROP_TLS_ALGORITHM = "athenz.cert_refresher.tls_algorithm";
    private static final char[] KEYSTORE_PASSWORD = "secret".toCharArray();
    private static final String DEFAULT_KEYSTORE_TYPE = "JKS";
    private static final long KEY_WAIT_TIME_MILLIS = TimeUnit.MINUTES.toMillis(Integer.parseInt(System.getProperty("athenz.cert_refresher.key_wait_time", "10")));

    public static KeyStore getKeyStore(String jksFilePath) throws IOException, KeyRefresherException {
        return Utils.getKeyStore(jksFilePath, KEYSTORE_PASSWORD);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static KeyStore getKeyStore(String jksFilePath, char[] password) throws IOException, KeyRefresherException {
        if (jksFilePath == null) throw new FileNotFoundException("jksFilePath is empty");
        if (jksFilePath.isEmpty()) {
            throw new FileNotFoundException("jksFilePath is empty");
        }
        String keyStoreFailMsg = "Unable to load " + jksFilePath + " as a KeyStore.  Please check the validity of the file.";
        try {
            KeyStore keyStore = KeyStore.getInstance(DEFAULT_KEYSTORE_TYPE);
            if (Paths.get(jksFilePath, new String[0]).isAbsolute()) {
                try (FileInputStream jksFileInputStream = new FileInputStream(jksFilePath);){
                    keyStore.load(jksFileInputStream, password);
                    KeyStore keyStore2 = keyStore;
                    return keyStore2;
                }
                catch (NoSuchAlgorithmException | CertificateException e) {
                    throw new KeyRefresherException(keyStoreFailMsg, e);
                }
            }
            try (InputStream jksFileInputStream = Utils.class.getClassLoader().getResourceAsStream(jksFilePath);){
                keyStore.load(jksFileInputStream, password);
                KeyStore keyStore3 = keyStore;
                return keyStore3;
            }
            catch (NoSuchAlgorithmException | CertificateException e) {
                throw new KeyRefresherException(keyStoreFailMsg, e);
            }
        }
        catch (KeyStoreException ex) {
            LOG.error("No Provider supports a KeyStoreSpi implementation for the specified type.", (Throwable)ex);
            return null;
        }
    }

    public static KeyManager[] getKeyManagers(String athenzPublicCert, String athenzPrivateKey) throws IOException, InterruptedException, KeyRefresherException {
        KeyManagerFactory keyManagerFactory;
        KeyStore keystore = Utils.createKeyStore(athenzPublicCert, athenzPrivateKey);
        try {
            keyManagerFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
            keyManagerFactory.init(keystore, KEYSTORE_PASSWORD);
        }
        catch (NoSuchAlgorithmException e) {
            throw new KeyRefresherException("No Provider supports a KeyManagerFactorySpi implementation for the specified algorithm.", e);
        }
        catch (UnrecoverableKeyException e) {
            throw new KeyRefresherException("key cannot be recovered (e.g. the given password is wrong).", e);
        }
        catch (KeyStoreException e) {
            throw new KeyRefresherException("Unable to initialize KeyManagerFactory.", e);
        }
        return keyManagerFactory.getKeyManagers();
    }

    public static KeyRefresher generateKeyRefresher(String trustStorePath, String athenzPublicCert, String athenzPrivateKey) throws FileNotFoundException, IOException, InterruptedException, KeyRefresherException {
        return Utils.generateKeyRefresher(trustStorePath, KEYSTORE_PASSWORD, athenzPublicCert, athenzPrivateKey, null);
    }

    public static KeyRefresher generateKeyRefresher(String trustStorePath, String trustStorePassword, String athenzPublicCert, String athenzPrivateKey) throws FileNotFoundException, IOException, InterruptedException, KeyRefresherException {
        return Utils.generateKeyRefresher(trustStorePath, trustStorePassword.toCharArray(), athenzPublicCert, athenzPrivateKey, null);
    }

    public static KeyRefresher generateKeyRefresher(String trustStorePath, char[] trustStorePassword, String athenzPublicCert, String athenzPrivateKey) throws FileNotFoundException, IOException, InterruptedException, KeyRefresherException {
        return Utils.generateKeyRefresher(trustStorePath, trustStorePassword, athenzPublicCert, athenzPrivateKey, null);
    }

    public static KeyRefresher generateKeyRefresher(String trustStorePath, char[] trustStorePassword, String athenzPublicCert, String athenzPrivateKey, KeyRefresherListener keyRefresherListener) throws FileNotFoundException, IOException, InterruptedException, KeyRefresherException {
        TrustStore trustStore = new TrustStore(trustStorePath, new JavaKeyStoreProvider(trustStorePath, trustStorePassword));
        return Utils.getKeyRefresher(athenzPublicCert, athenzPrivateKey, trustStore, keyRefresherListener);
    }

    public static KeyRefresher generateKeyRefresherFromCaCert(String caCertPath, String athenzPublicCert, String athenzPrivateKey) throws IOException, InterruptedException, KeyRefresherException {
        TrustStore trustStore = new TrustStore(caCertPath, new CaCertKeyStoreProvider(caCertPath));
        return Utils.getKeyRefresher(athenzPublicCert, athenzPrivateKey, trustStore);
    }

    static KeyRefresher getKeyRefresher(String athenzPublicCert, String athenzPrivateKey, TrustStore trustStore) throws IOException, InterruptedException, KeyRefresherException {
        return Utils.getKeyRefresher(athenzPublicCert, athenzPrivateKey, trustStore, null);
    }

    static KeyRefresher getKeyRefresher(String athenzPublicCert, String athenzPrivateKey, TrustStore trustStore, KeyRefresherListener keyRefresherListener) throws IOException, InterruptedException, KeyRefresherException {
        KeyRefresher keyRefresher;
        KeyManagerProxy keyManagerProxy = new KeyManagerProxy(Utils.getKeyManagers(athenzPublicCert, athenzPrivateKey));
        TrustManagerProxy trustManagerProxy = new TrustManagerProxy(trustStore.getTrustManagers());
        try {
            keyRefresher = new KeyRefresher(athenzPublicCert, athenzPrivateKey, trustStore, keyManagerProxy, trustManagerProxy, keyRefresherListener);
        }
        catch (NoSuchAlgorithmException e) {
            throw new KeyRefresherException(e);
        }
        return keyRefresher;
    }

    public static SSLContext buildSSLContext(KeyManagerProxy keyManagerProxy, TrustManagerProxy trustManagerProxy, String protocol) throws KeyRefresherException {
        SSLContext sslContext;
        try {
            sslContext = SSLContext.getInstance(protocol);
            sslContext.init(new KeyManager[]{keyManagerProxy}, new TrustManager[]{trustManagerProxy}, null);
        }
        catch (NoSuchAlgorithmException e) {
            throw new KeyRefresherException("No Provider supports a SSLContextSpi implementation for the specified protocol " + protocol, e);
        }
        catch (KeyManagementException e) {
            throw new KeyRefresherException("Unable to create SSLContext.", e);
        }
        return sslContext;
    }

    public static SSLContext buildSSLContext(KeyManagerProxy keyManagerProxy, TrustManagerProxy trustManagerProxy) throws KeyRefresherException {
        String protocol = System.getProperty(PROP_TLS_ALGORITHM);
        if (protocol != null && !protocol.isEmpty()) {
            return Utils.buildSSLContext(keyManagerProxy, trustManagerProxy, protocol);
        }
        try {
            return Utils.buildSSLContext(keyManagerProxy, trustManagerProxy, SSLCONTEXT_ALGORITHM_TLS13);
        }
        catch (KeyRefresherException ignored) {
            return Utils.buildSSLContext(keyManagerProxy, trustManagerProxy, SSLCONTEXT_ALGORITHM_TLS12);
        }
    }

    static Supplier<InputStream> inputStreamSupplierFromFile(File file) throws UncheckedIOException {
        return () -> {
            try {
                return new FileInputStream(file);
            }
            catch (FileNotFoundException e) {
                throw new UncheckedIOException(e);
            }
        };
    }

    static Supplier<InputStream> inputStreamSupplierFromResource(String resource) throws UncheckedIOException {
        return () -> {
            InputStream ret = Utils.class.getClassLoader().getResourceAsStream(resource);
            if (ret == null) {
                throw new UncheckedIOException(new FileNotFoundException("Certificate or private key file is empty " + resource));
            }
            return ret;
        };
    }

    static Supplier<InputStream> inputStreamSupplierFromString(String s) throws UncheckedIOException {
        return () -> new ByteArrayInputStream(s.getBytes(StandardCharsets.UTF_8));
    }

    public static KeyStore createKeyStore(String athenzPublicCert, String athenzPrivateKey) throws FileNotFoundException, IOException, InterruptedException, KeyRefresherException {
        Supplier<InputStream> keyFileSupplier;
        Supplier<InputStream> certFileSupplier;
        if (athenzPublicCert == null || athenzPublicCert.isEmpty()) {
            throw new FileNotFoundException("athenzPublicCert can not be empty");
        }
        if (athenzPrivateKey == null || athenzPrivateKey.isEmpty()) {
            throw new FileNotFoundException("athenzPrivateKey can not be empty");
        }
        Supplier<String> certLocationSupplier = () -> athenzPublicCert;
        Supplier<String> keyLocationSupplier = () -> athenzPrivateKey;
        if (Paths.get(athenzPublicCert, new String[0]).isAbsolute() && Paths.get(athenzPrivateKey, new String[0]).isAbsolute()) {
            File certFile = new File(athenzPublicCert);
            File keyFile = new File(athenzPrivateKey);
            certFileSupplier = Utils.inputStreamSupplierFromFile(certFile);
            keyFileSupplier = Utils.inputStreamSupplierFromFile(keyFile);
            long startTime = System.currentTimeMillis();
            while (!certFile.exists() || !keyFile.exists()) {
                long durationInMillis = System.currentTimeMillis() - startTime;
                if (durationInMillis > KEY_WAIT_TIME_MILLIS) {
                    throw new KeyRefresherException("KeyRefresher waited " + durationInMillis + " ms for valid public cert: " + athenzPublicCert + " or private key: " + athenzPrivateKey + " files. Giving up.");
                }
                LOG.error("Missing Athenz public certificate {} or private key {} files. Waiting {} ms", new Object[]{athenzPublicCert, athenzPrivateKey, durationInMillis});
                Thread.sleep(1000L);
            }
        } else {
            certFileSupplier = Utils.inputStreamSupplierFromResource(athenzPublicCert);
            keyFileSupplier = Utils.inputStreamSupplierFromResource(athenzPrivateKey);
        }
        try {
            return Utils.createKeyStore(certFileSupplier, certLocationSupplier, keyFileSupplier, keyLocationSupplier);
        }
        catch (UncheckedIOException e) {
            throw e.getCause();
        }
    }

    public static KeyStore createKeyStoreFromPems(String athenzPublicCertPem, String athenzPrivateKeyPem) throws IOException, KeyRefresherException {
        return Utils.createKeyStore(Utils.inputStreamSupplierFromString(athenzPublicCertPem), () -> "in memory certificate pem", Utils.inputStreamSupplierFromString(athenzPrivateKeyPem), () -> "in memory private key pem");
    }

    public static KeyStore createKeyStore(Supplier<InputStream> athenzPublicCertInputStream, Supplier<String> athenzPublicCertLocationSupplier, Supplier<InputStream> athenzPrivateKeyInputStream, Supplier<String> athenzPrivateKeyLocationSupplier) throws IOException, KeyRefresherException {
        KeyStore keyStore = null;
        try (InputStream publicCertStream = athenzPublicCertInputStream.get();
             InputStream privateKeyStream = athenzPrivateKeyInputStream.get();
             PEMParser pemParser = new PEMParser((Reader)new InputStreamReader(privateKeyStream));){
            PrivateKey privateKey;
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            JcaPEMKeyConverter pemConverter = new JcaPEMKeyConverter();
            Object key = pemParser.readObject();
            if (key instanceof PEMKeyPair) {
                PrivateKeyInfo pKeyInfo = ((PEMKeyPair)key).getPrivateKeyInfo();
                privateKey = pemConverter.getPrivateKey(pKeyInfo);
            } else if (key instanceof PrivateKeyInfo) {
                privateKey = pemConverter.getPrivateKey((PrivateKeyInfo)key);
            } else {
                throw new KeyRefresherException("Unknown object type: " + (key == null ? "null" : key.getClass().getName()));
            }
            List certificates = (List)cf.generateCertificates(publicCertStream);
            if (certificates.isEmpty()) {
                throw new KeyRefresherException("Certificate file contains empty certificate or an invalid certificate.");
            }
            String alias = ((X509Certificate)certificates.get(0)).getSubjectX500Principal().getName();
            if (LOG.isDebugEnabled()) {
                LOG.debug("{} number of certificates found. Using {} alias to create the keystore", (Object)certificates.size(), (Object)alias);
            }
            keyStore = KeyStore.getInstance(DEFAULT_KEYSTORE_TYPE);
            keyStore.load(null);
            keyStore.setKeyEntry(alias, privateKey, KEYSTORE_PASSWORD, certificates.toArray((Certificate[])new X509Certificate[certificates.size()]));
        }
        catch (NoSuchAlgorithmException | CertificateException ex) {
            String keyStoreFailMsg = "Unable to load " + athenzPublicCertLocationSupplier.get() + " as a KeyStore. Please check the validity of the file.";
            throw new KeyRefresherException(keyStoreFailMsg, ex);
        }
        catch (KeyStoreException ex) {
            LOG.error("No Provider supports a KeyStoreSpi implementation for the specified type.", (Throwable)ex);
        }
        return keyStore;
    }

    public static KeyStore generateTrustStore(InputStream inputStream) throws IOException, KeyRefresherException {
        KeyStore keyStore = null;
        try {
            CertificateFactory factory = CertificateFactory.getInstance("X.509");
            keyStore = KeyStore.getInstance(DEFAULT_KEYSTORE_TYPE);
            keyStore.load(null);
            for (Certificate certificate : factory.generateCertificates(inputStream)) {
                String alias = ((X509Certificate)certificate).getSubjectX500Principal().getName();
                keyStore.setCertificateEntry(alias, certificate);
            }
        }
        catch (NoSuchAlgorithmException | CertificateException e) {
            String string = "Unable to load the input stream as a KeyStore. Please check the content.";
            throw new KeyRefresherException(string, e);
        }
        catch (KeyStoreException ex) {
            LOG.error("No Provider supports a KeyStoreSpi implementation for the specified type {}", (Object)DEFAULT_KEYSTORE_TYPE, (Object)ex);
        }
        return keyStore;
    }
}

