/*
 * Decompiled with CFR 0.152.
 */
package com.oath.auth;

import com.oath.auth.CaCertKeyStoreProvider;
import com.oath.auth.JavaKeyStoreProvider;
import com.oath.auth.KeyManagerProxy;
import com.oath.auth.KeyRefresher;
import com.oath.auth.KeyRefresherException;
import com.oath.auth.KeyRefresherListener;
import com.oath.auth.TrustManagerProxy;
import com.oath.auth.TrustStore;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Paths;
import java.security.KeyFactory;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECKey;
import java.security.interfaces.RSAKey;
import java.security.spec.InvalidKeySpecException;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPrivateKey;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.jce.spec.ECPublicKeySpec;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.math.ec.FixedPointCombMultiplier;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Utils {
    private static final Logger LOG = LoggerFactory.getLogger(Utils.class);
    private static final String SSLCONTEXT_ALGORITHM_TLS12 = "TLSv1.2";
    private static final String SSLCONTEXT_ALGORITHM_TLS13 = "TLSv1.3";
    private static final String PROP_KEY_WAIT_TIME = "athenz.cert_refresher.key_wait_time";
    private static final String PROP_TLS_ALGORITHM = "athenz.cert_refresher.tls_algorithm";
    private static final String PROP_DISABLE_PUB_KEY_CHECK = "athenz.cert_refresher.disable_public_key_check";
    private static final String PROP_SKIP_BC_PROVIDER = "athenz.cert_refresher.skip_bc_provider";
    private static final char[] KEYSTORE_PASSWORD = "secret".toCharArray();
    private static final String DEFAULT_KEYSTORE_TYPE = "JKS";
    private static final long KEY_WAIT_TIME_MILLIS = TimeUnit.MINUTES.toMillis(Integer.parseInt(System.getProperty("athenz.cert_refresher.key_wait_time", "10")));
    private static boolean disablePublicKeyCheck = Boolean.parseBoolean(System.getProperty("athenz.cert_refresher.disable_public_key_check", "false"));

    public static void setDisablePublicKeyCheck(boolean state) {
        disablePublicKeyCheck = state;
    }

    public static KeyStore getKeyStore(String jksFilePath) throws IOException, KeyRefresherException {
        return Utils.getKeyStore(jksFilePath, KEYSTORE_PASSWORD);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static KeyStore getKeyStore(String jksFilePath, char[] password) throws IOException, KeyRefresherException {
        if (jksFilePath == null) throw new FileNotFoundException("jksFilePath is empty");
        if (jksFilePath.isEmpty()) {
            throw new FileNotFoundException("jksFilePath is empty");
        }
        String keyStoreFailMsg = "Unable to load " + jksFilePath + " as a KeyStore.  Please check the validity of the file.";
        try {
            KeyStore keyStore = KeyStore.getInstance(DEFAULT_KEYSTORE_TYPE);
            if (Paths.get(jksFilePath, new String[0]).isAbsolute()) {
                try (FileInputStream jksFileInputStream = new FileInputStream(jksFilePath);){
                    keyStore.load(jksFileInputStream, password);
                    KeyStore keyStore2 = keyStore;
                    return keyStore2;
                }
                catch (NoSuchAlgorithmException | CertificateException e) {
                    throw new KeyRefresherException(keyStoreFailMsg, e);
                }
            }
            try (InputStream jksFileInputStream = Utils.class.getClassLoader().getResourceAsStream(jksFilePath);){
                keyStore.load(jksFileInputStream, password);
                KeyStore keyStore3 = keyStore;
                return keyStore3;
            }
            catch (NoSuchAlgorithmException | CertificateException e) {
                throw new KeyRefresherException(keyStoreFailMsg, e);
            }
        }
        catch (KeyStoreException ex) {
            LOG.error("No Provider supports a KeyStoreSpi implementation for the specified type.", (Throwable)ex);
            return null;
        }
    }

    public static KeyManager[] getKeyManagers(String athenzPublicCert, String athenzPrivateKey) throws IOException, InterruptedException, KeyRefresherException {
        KeyStore keystore = Utils.createKeyStore(athenzPublicCert, athenzPrivateKey);
        return Utils.getKeyManagersFromKeyStore(keystore);
    }

    public static KeyManager[] getKeyManagersFromPems(String athenzPublicCertPem, String athenzPrivateKeyPem) throws IOException, KeyRefresherException {
        KeyStore keystore = Utils.createKeyStoreFromPems(athenzPublicCertPem, athenzPrivateKeyPem);
        return Utils.getKeyManagersFromKeyStore(keystore);
    }

    private static KeyManager[] getKeyManagersFromKeyStore(KeyStore keystore) throws KeyRefresherException {
        KeyManagerFactory keyManagerFactory;
        try {
            keyManagerFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
            keyManagerFactory.init(keystore, KEYSTORE_PASSWORD);
        }
        catch (NoSuchAlgorithmException e) {
            throw new KeyRefresherException("No Provider supports a KeyManagerFactorySpi implementation for the specified algorithm.", e);
        }
        catch (UnrecoverableKeyException e) {
            throw new KeyRefresherException("key cannot be recovered (e.g. the given password is wrong).", e);
        }
        catch (KeyStoreException e) {
            throw new KeyRefresherException("Unable to initialize KeyManagerFactory.", e);
        }
        return keyManagerFactory.getKeyManagers();
    }

    public static KeyRefresher generateKeyRefresher(String trustStorePath, String athenzPublicCert, String athenzPrivateKey) throws FileNotFoundException, IOException, InterruptedException, KeyRefresherException {
        return Utils.generateKeyRefresher(trustStorePath, KEYSTORE_PASSWORD, athenzPublicCert, athenzPrivateKey, null);
    }

    public static KeyRefresher generateKeyRefresher(String trustStorePath, String trustStorePassword, String athenzPublicCert, String athenzPrivateKey) throws FileNotFoundException, IOException, InterruptedException, KeyRefresherException {
        return Utils.generateKeyRefresher(trustStorePath, trustStorePassword.toCharArray(), athenzPublicCert, athenzPrivateKey, null);
    }

    public static KeyRefresher generateKeyRefresher(String trustStorePath, char[] trustStorePassword, String athenzPublicCert, String athenzPrivateKey) throws FileNotFoundException, IOException, InterruptedException, KeyRefresherException {
        return Utils.generateKeyRefresher(trustStorePath, trustStorePassword, athenzPublicCert, athenzPrivateKey, null);
    }

    public static KeyRefresher generateKeyRefresher(String trustStorePath, char[] trustStorePassword, String athenzPublicCert, String athenzPrivateKey, KeyRefresherListener keyRefresherListener) throws FileNotFoundException, IOException, InterruptedException, KeyRefresherException {
        TrustStore trustStore = null;
        if (trustStorePath != null && !trustStorePath.isEmpty()) {
            trustStore = new TrustStore(trustStorePath, new JavaKeyStoreProvider(trustStorePath, trustStorePassword));
        }
        return Utils.getKeyRefresher(athenzPublicCert, athenzPrivateKey, trustStore, keyRefresherListener);
    }

    public static KeyRefresher generateKeyRefresherFromCaCert(String caCertPath, String athenzPublicCert, String athenzPrivateKey) throws IOException, InterruptedException, KeyRefresherException {
        TrustStore trustStore = null;
        if (caCertPath != null && !caCertPath.isEmpty()) {
            trustStore = new TrustStore(caCertPath, new CaCertKeyStoreProvider(caCertPath));
        }
        return Utils.getKeyRefresher(athenzPublicCert, athenzPrivateKey, trustStore);
    }

    static KeyRefresher getKeyRefresher(String athenzPublicCert, String athenzPrivateKey, TrustStore trustStore) throws IOException, InterruptedException, KeyRefresherException {
        return Utils.getKeyRefresher(athenzPublicCert, athenzPrivateKey, trustStore, null);
    }

    static KeyRefresher getKeyRefresher(String athenzPublicCert, String athenzPrivateKey, TrustStore trustStore, KeyRefresherListener keyRefresherListener) throws IOException, InterruptedException, KeyRefresherException {
        KeyRefresher keyRefresher;
        KeyManagerProxy keyManagerProxy = new KeyManagerProxy(Utils.getKeyManagers(athenzPublicCert, athenzPrivateKey));
        TrustManagerProxy trustManagerProxy = null;
        if (trustStore != null) {
            trustManagerProxy = new TrustManagerProxy(trustStore.getTrustManagers());
        }
        try {
            keyRefresher = new KeyRefresher(athenzPublicCert, athenzPrivateKey, trustStore, keyManagerProxy, trustManagerProxy, keyRefresherListener);
        }
        catch (NoSuchAlgorithmException e) {
            throw new KeyRefresherException(e);
        }
        return keyRefresher;
    }

    public static SSLContext buildSSLContext(KeyManagerProxy keyManagerProxy, TrustManagerProxy trustManagerProxy, String protocol) throws KeyRefresherException {
        SSLContext sslContext;
        try {
            TrustManager[] trustManagerArray;
            sslContext = SSLContext.getInstance(protocol);
            KeyManager[] keyManagerArray = new KeyManager[]{keyManagerProxy};
            if (trustManagerProxy == null) {
                trustManagerArray = null;
            } else {
                TrustManager[] trustManagerArray2 = new TrustManager[1];
                trustManagerArray = trustManagerArray2;
                trustManagerArray2[0] = trustManagerProxy;
            }
            sslContext.init(keyManagerArray, trustManagerArray, null);
        }
        catch (NoSuchAlgorithmException e) {
            throw new KeyRefresherException("No Provider supports a SSLContextSpi implementation for the specified protocol " + protocol, e);
        }
        catch (KeyManagementException e) {
            throw new KeyRefresherException("Unable to create SSLContext.", e);
        }
        return sslContext;
    }

    public static SSLContext buildSSLContext(KeyManagerProxy keyManagerProxy, TrustManagerProxy trustManagerProxy) throws KeyRefresherException {
        String protocol = System.getProperty(PROP_TLS_ALGORITHM);
        if (protocol != null && !protocol.isEmpty()) {
            return Utils.buildSSLContext(keyManagerProxy, trustManagerProxy, protocol);
        }
        try {
            return Utils.buildSSLContext(keyManagerProxy, trustManagerProxy, SSLCONTEXT_ALGORITHM_TLS13);
        }
        catch (KeyRefresherException ignored) {
            return Utils.buildSSLContext(keyManagerProxy, trustManagerProxy, SSLCONTEXT_ALGORITHM_TLS12);
        }
    }

    public static SSLContext buildSSLContext(String caCertsPem, String athenzPublicCertPem, String athenzPrivateKeyPem) throws KeyRefresherException, IOException {
        TrustManagerProxy trustManagerProxy = null;
        if (caCertsPem != null) {
            TrustStore trustStore = new TrustStore(null, new CaCertKeyStoreProvider(Utils.inputStreamSupplierFromString(caCertsPem)));
            trustManagerProxy = new TrustManagerProxy(trustStore.getTrustManagers());
        }
        KeyManagerProxy keyManagerProxy = new KeyManagerProxy(Utils.getKeyManagersFromPems(athenzPublicCertPem, athenzPrivateKeyPem));
        String protocol = System.getProperty(PROP_TLS_ALGORITHM, SSLCONTEXT_ALGORITHM_TLS13);
        return Utils.buildSSLContext(keyManagerProxy, trustManagerProxy, protocol);
    }

    static Supplier<InputStream> inputStreamSupplierFromFile(File file) throws UncheckedIOException {
        return () -> {
            try {
                return new FileInputStream(file);
            }
            catch (FileNotFoundException e) {
                throw new UncheckedIOException(e);
            }
        };
    }

    static Supplier<InputStream> inputStreamSupplierFromResource(String resource) throws UncheckedIOException {
        return () -> {
            InputStream ret = Utils.class.getClassLoader().getResourceAsStream(resource);
            if (ret == null) {
                throw new UncheckedIOException(new FileNotFoundException("Certificate or private key file is empty " + resource));
            }
            return ret;
        };
    }

    static Supplier<InputStream> inputStreamSupplierFromString(String s) throws UncheckedIOException {
        return () -> new ByteArrayInputStream(s.getBytes(StandardCharsets.UTF_8));
    }

    public static KeyStore createKeyStore(String athenzPublicCert, String athenzPrivateKey) throws FileNotFoundException, IOException, InterruptedException, KeyRefresherException {
        Supplier<InputStream> keyFileSupplier;
        Supplier<InputStream> certFileSupplier;
        if (athenzPublicCert == null || athenzPublicCert.isEmpty()) {
            throw new FileNotFoundException("athenzPublicCert can not be empty");
        }
        if (athenzPrivateKey == null || athenzPrivateKey.isEmpty()) {
            throw new FileNotFoundException("athenzPrivateKey can not be empty");
        }
        Supplier<String> certLocationSupplier = () -> athenzPublicCert;
        Supplier<String> keyLocationSupplier = () -> athenzPrivateKey;
        if (Paths.get(athenzPublicCert, new String[0]).isAbsolute() && Paths.get(athenzPrivateKey, new String[0]).isAbsolute()) {
            File certFile = new File(athenzPublicCert);
            File keyFile = new File(athenzPrivateKey);
            certFileSupplier = Utils.inputStreamSupplierFromFile(certFile);
            keyFileSupplier = Utils.inputStreamSupplierFromFile(keyFile);
            long startTime = System.currentTimeMillis();
            while (!certFile.exists() || !keyFile.exists()) {
                long durationInMillis = System.currentTimeMillis() - startTime;
                if (durationInMillis > KEY_WAIT_TIME_MILLIS) {
                    throw new KeyRefresherException("KeyRefresher waited " + durationInMillis + " ms for valid public cert: " + athenzPublicCert + " or private key: " + athenzPrivateKey + " files. Giving up.");
                }
                LOG.error("Missing Athenz public certificate {} or private key {} files. Waiting {} ms", new Object[]{athenzPublicCert, athenzPrivateKey, durationInMillis});
                Thread.sleep(1000L);
            }
        } else {
            certFileSupplier = Utils.inputStreamSupplierFromResource(athenzPublicCert);
            keyFileSupplier = Utils.inputStreamSupplierFromResource(athenzPrivateKey);
        }
        try {
            return Utils.createKeyStore(certFileSupplier, certLocationSupplier, keyFileSupplier, keyLocationSupplier);
        }
        catch (UncheckedIOException e) {
            throw e.getCause();
        }
    }

    public static KeyStore createKeyStoreFromPems(String athenzPublicCertPem, String athenzPrivateKeyPem) throws IOException, KeyRefresherException {
        return Utils.createKeyStore(Utils.inputStreamSupplierFromString(athenzPublicCertPem), () -> "in memory certificate pem", Utils.inputStreamSupplierFromString(athenzPrivateKeyPem), () -> "in memory private key pem");
    }

    public static KeyStore createKeyStore(Supplier<InputStream> athenzPublicCertInputStream, Supplier<String> athenzPublicCertLocationSupplier, Supplier<InputStream> athenzPrivateKeyInputStream, Supplier<String> athenzPrivateKeyLocationSupplier) throws IOException, KeyRefresherException {
        KeyStore keyStore = null;
        try (InputStream publicCertStream = athenzPublicCertInputStream.get();
             InputStream privateKeyStream = athenzPrivateKeyInputStream.get();
             PEMParser pemParser = new PEMParser((Reader)new InputStreamReader(privateKeyStream));){
            PrivateKey privateKey;
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            JcaPEMKeyConverter pemConverter = new JcaPEMKeyConverter();
            Object key = pemParser.readObject();
            if (key instanceof PEMKeyPair) {
                PrivateKeyInfo pKeyInfo = ((PEMKeyPair)key).getPrivateKeyInfo();
                privateKey = pemConverter.getPrivateKey(pKeyInfo);
            } else if (key instanceof PrivateKeyInfo) {
                privateKey = pemConverter.getPrivateKey((PrivateKeyInfo)key);
            } else {
                throw new KeyRefresherException("Unknown object type: " + (key == null ? "null" : key.getClass().getName()));
            }
            List certificates = (List)cf.generateCertificates(publicCertStream);
            if (certificates.isEmpty()) {
                throw new KeyRefresherException("Certificate file contains empty certificate or an invalid certificate.");
            }
            String alias = ((X509Certificate)certificates.get(0)).getSubjectX500Principal().getName();
            if (LOG.isDebugEnabled()) {
                LOG.debug("{} number of certificates found. Using {} alias to create the keystore", (Object)certificates.size(), (Object)alias);
            }
            Utils.verifyPrivateKeyCertsMatch(privateKey, certificates);
            keyStore = KeyStore.getInstance(DEFAULT_KEYSTORE_TYPE);
            keyStore.load(null);
            keyStore.setKeyEntry(alias, privateKey, KEYSTORE_PASSWORD, certificates.toArray((Certificate[])new X509Certificate[certificates.size()]));
        }
        catch (NoSuchAlgorithmException | CertificateException ex) {
            String keyStoreFailMsg = "Unable to load private key: " + athenzPrivateKeyLocationSupplier.get() + " and certificate: " + athenzPublicCertLocationSupplier.get() + " as a KeyStore. Please check the validity of the files.";
            throw new KeyRefresherException(keyStoreFailMsg, ex);
        }
        catch (KeyStoreException ex) {
            LOG.error("No Provider supports a KeyStoreSpi implementation for the specified type.", (Throwable)ex);
        }
        return keyStore;
    }

    static boolean verifyPrivatePublicKeyMatch(PrivateKey privateKey, PublicKey publicKey) throws NoSuchAlgorithmException, InvalidKeySpecException {
        if (publicKey instanceof RSAKey) {
            if (!(privateKey instanceof RSAKey)) {
                return false;
            }
            RSAKey pubRSAKey = (RSAKey)((Object)publicKey);
            RSAKey prvRSAKey = (RSAKey)((Object)privateKey);
            return pubRSAKey.getModulus().compareTo(prvRSAKey.getModulus()) == 0;
        }
        if (publicKey instanceof ECKey) {
            if (!(privateKey instanceof ECKey)) {
                return false;
            }
            KeyFactory keyFactory = KeyFactory.getInstance(privateKey.getAlgorithm());
            BCECPrivateKey ecPrivKey = (BCECPrivateKey)privateKey;
            FixedPointCombMultiplier ecMultiplier = new FixedPointCombMultiplier();
            ECParameterSpec ecParamSpec = ecPrivKey.getParameters();
            ECPoint ecPointQ = ecMultiplier.multiply(ecParamSpec.getG(), ecPrivKey.getD());
            ECPublicKeySpec prvKeySpec = new ECPublicKeySpec(ecPointQ, ecParamSpec);
            ECPublicKeySpec pubKeySpec = keyFactory.getKeySpec(publicKey, ECPublicKeySpec.class);
            return prvKeySpec.getQ().equals(pubKeySpec.getQ()) && prvKeySpec.getParams().equals((Object)pubKeySpec.getParams());
        }
        return false;
    }

    static void verifyPrivateKeyCertsMatch(PrivateKey privateKey, List<? extends Certificate> certificates) throws KeyRefresherException {
        if (disablePublicKeyCheck) {
            return;
        }
        try {
            for (Certificate certificate : certificates) {
                if (!Utils.verifyPrivatePublicKeyMatch(privateKey, certificate.getPublicKey())) continue;
                return;
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        throw new KeyRefresherException("Public key mismatch");
    }

    public static KeyStore generateTrustStore(InputStream inputStream) throws IOException, KeyRefresherException {
        KeyStore keyStore = null;
        try {
            CertificateFactory factory = CertificateFactory.getInstance("X.509");
            keyStore = KeyStore.getInstance(DEFAULT_KEYSTORE_TYPE);
            keyStore.load(null);
            for (Certificate certificate : factory.generateCertificates(inputStream)) {
                String alias = ((X509Certificate)certificate).getSubjectX500Principal().getName();
                keyStore.setCertificateEntry(alias, certificate);
            }
        }
        catch (NoSuchAlgorithmException | CertificateException e) {
            String string = "Unable to load the input stream as a KeyStore. Please check the content.";
            throw new KeyRefresherException(string, e);
        }
        catch (KeyStoreException ex) {
            LOG.error("No Provider supports a KeyStoreSpi implementation for the specified type {}", (Object)DEFAULT_KEYSTORE_TYPE, (Object)ex);
        }
        return keyStore;
    }

    static {
        boolean skipBCProvider = Boolean.parseBoolean(System.getProperty(PROP_SKIP_BC_PROVIDER, "false"));
        if (!skipBCProvider) {
            Security.addProvider((Provider)new BouncyCastleProvider());
        }
    }
}

