/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.elide.graphql.subscriptions.websocket;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.yahoo.elide.Elide;
import com.yahoo.elide.core.datastore.DataStore;
import com.yahoo.elide.graphql.subscriptions.websocket.ConnectionInfo;
import com.yahoo.elide.graphql.subscriptions.websocket.RequestHandler;
import com.yahoo.elide.graphql.subscriptions.websocket.protocol.Complete;
import com.yahoo.elide.graphql.subscriptions.websocket.protocol.ConnectionAck;
import com.yahoo.elide.graphql.subscriptions.websocket.protocol.MessageType;
import com.yahoo.elide.graphql.subscriptions.websocket.protocol.Pong;
import com.yahoo.elide.graphql.subscriptions.websocket.protocol.Subscribe;
import com.yahoo.elide.graphql.subscriptions.websocket.protocol.WebSocketCloseReasons;
import graphql.GraphQL;
import java.io.IOException;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import javax.websocket.CloseReason;
import javax.websocket.Session;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionHandler {
    private static final Logger log = LoggerFactory.getLogger(SessionHandler.class);
    protected DataStore topicStore;
    protected Elide elide;
    protected GraphQL api;
    protected Session wrappedSession;
    protected Map<String, RequestHandler> activeRequests;
    protected ConnectionInfo connectionInfo;
    protected ObjectMapper mapper;
    protected int connectionTimeoutMs;
    protected int maxSubscriptions;
    protected Thread timeoutThread;
    protected boolean initialized = false;
    protected boolean sendPingOnSubscribe = false;
    protected boolean verboseErrors = false;
    protected ExecutorService executorService;
    protected boolean isOpen = true;

    public SessionHandler(Session wrappedSession, DataStore topicStore, Elide elide, GraphQL api, int connectionTimeoutMs, int maxSubscriptions, ConnectionInfo connectionInfo, boolean sendPingOnSubscribe, boolean verboseErrors, ExecutorService executorService) {
        Preconditions.checkState((maxSubscriptions > 0 ? 1 : 0) != 0);
        this.wrappedSession = wrappedSession;
        this.topicStore = topicStore;
        this.elide = elide;
        this.api = api;
        this.connectionInfo = connectionInfo;
        this.mapper = elide.getMapper().getObjectMapper();
        this.activeRequests = new ConcurrentHashMap<String, RequestHandler>();
        this.connectionTimeoutMs = connectionTimeoutMs;
        this.maxSubscriptions = maxSubscriptions;
        this.sendPingOnSubscribe = sendPingOnSubscribe;
        this.verboseErrors = verboseErrors;
        this.executorService = executorService == null ? Executors.newFixedThreadPool(maxSubscriptions) : executorService;
        this.timeoutThread = new Thread(new ConnectionTimer());
        this.timeoutThread.start();
    }

    public synchronized void close(CloseReason reason) throws IOException {
        log.debug("SessionHandler closing");
        this.isOpen = false;
        for (Map.Entry<String, RequestHandler> item : this.activeRequests.entrySet()) {
            RequestHandler handler = item.getValue();
            handler.safeClose();
        }
        this.wrappedSession.close(reason);
        this.executorService.shutdownNow();
        log.debug("SessionHandler closed");
    }

    protected void close(String protocolID) {
        this.activeRequests.remove(protocolID);
    }

    public void handleRequest(String message) {
        log.debug("Received Message: {} {}", (Object)this.wrappedSession.getId(), (Object)message);
        try {
            MessageType messageType;
            JsonNode type = this.mapper.readTree(message).get("type");
            if (type == null) {
                this.safeClose(WebSocketCloseReasons.INVALID_MESSAGE);
                return;
            }
            try {
                messageType = MessageType.valueOf(type.textValue().toUpperCase(Locale.ROOT));
            }
            catch (IllegalArgumentException e) {
                this.safeClose(WebSocketCloseReasons.INVALID_MESSAGE);
                return;
            }
            switch (messageType) {
                case PING: {
                    this.handlePing();
                    return;
                }
                case PONG: {
                    return;
                }
                case CONNECTION_INIT: {
                    this.handleConnectionInit();
                    return;
                }
                case COMPLETE: {
                    Complete complete = (Complete)this.mapper.readValue(message, Complete.class);
                    this.handleComplete(complete);
                    return;
                }
                case SUBSCRIBE: {
                    Subscribe subscribe = (Subscribe)this.mapper.readValue(message, Subscribe.class);
                    this.handleSubscribe(subscribe);
                    return;
                }
            }
            this.safeClose(WebSocketCloseReasons.INVALID_MESSAGE);
            return;
        }
        catch (JsonProcessingException e) {
            this.safeClose(WebSocketCloseReasons.INVALID_MESSAGE);
            return;
        }
    }

    protected void handlePing() {
        this.safeSendPong();
    }

    protected void handleConnectionInit() {
        if (this.initialized) {
            this.safeClose(WebSocketCloseReasons.MULTIPLE_INIT);
            return;
        }
        this.timeoutThread.interrupt();
        this.safeSendConnectionAck();
        this.initialized = true;
    }

    protected void handleSubscribe(final Subscribe subscribe) {
        if (!this.initialized) {
            this.safeClose(WebSocketCloseReasons.UNAUTHORIZED);
            return;
        }
        String protocolID = subscribe.getId();
        if (this.activeRequests.containsKey(protocolID)) {
            this.safeClose(new CloseReason(WebSocketCloseReasons.createCloseCode(WebSocketCloseReasons.CloseCode.DUPLICATE_ID.getCode()), "Subscriber for " + protocolID + " already exists"));
            return;
        }
        if (this.activeRequests.size() >= this.maxSubscriptions) {
            this.safeClose(WebSocketCloseReasons.MAX_SUBSCRIPTIONS);
            return;
        }
        final RequestHandler requestHandler = new RequestHandler(this, this.topicStore, this.elide, this.api, protocolID, UUID.randomUUID(), this.connectionInfo, this.sendPingOnSubscribe, this.verboseErrors);
        this.activeRequests.put(protocolID, requestHandler);
        this.executorService.submit(new Runnable(){

            @Override
            public void run() {
                requestHandler.handleRequest(subscribe);
            }
        });
    }

    protected void handleComplete(Complete complete) {
        String protocolID = complete.getId();
        RequestHandler handler = this.activeRequests.remove(protocolID);
        if (handler != null) {
            handler.safeClose();
        }
    }

    protected void safeSendConnectionAck() {
        ObjectMapper mapper = this.elide.getElideSettings().getMapper().getObjectMapper();
        ConnectionAck ack = new ConnectionAck();
        try {
            this.sendMessage(mapper.writeValueAsString((Object)ack));
        }
        catch (JsonProcessingException e) {
            log.error("UNEXPECTED Json Serialization Error {}", (Object)e.getMessage());
            this.safeClose(WebSocketCloseReasons.INTERNAL_ERROR);
        }
    }

    protected void safeSendPong() {
        ObjectMapper mapper = this.elide.getElideSettings().getMapper().getObjectMapper();
        Pong pong = new Pong();
        try {
            this.sendMessage(mapper.writeValueAsString((Object)pong));
        }
        catch (JsonProcessingException e) {
            log.error("UNEXPECTED Json Serialization Error {}", (Object)e.getMessage());
            this.safeClose(WebSocketCloseReasons.INTERNAL_ERROR);
        }
    }

    protected void safeClose(CloseReason reason) {
        log.debug("Closing session handler: {} {}", (Object)this.wrappedSession.getId(), (Object)reason);
        try {
            this.close(reason);
        }
        catch (Exception e) {
            log.error("UNEXPECTED: Closing {} failed for {}", (Object)this.wrappedSession.getId(), (Object)e.getMessage());
        }
    }

    public synchronized void sendMessage(String message) {
        if (this.isOpen) {
            try {
                this.wrappedSession.getAsyncRemote().sendText(message);
                return;
            }
            catch (Exception e) {
                log.debug("UNEXPECTED: Sending message {} failed for {}", (Object)message, (Object)e.getMessage());
                this.safeClose(WebSocketCloseReasons.INTERNAL_ERROR);
            }
        }
        log.debug("UNEXPECTED: Sending message {} on closed session", (Object)message);
    }

    private class ConnectionTimer
    implements Runnable {
        private ConnectionTimer() {
        }

        @Override
        public void run() {
            try {
                Thread.sleep(SessionHandler.this.connectionTimeoutMs);
                if (SessionHandler.this.activeRequests.size() == 0) {
                    SessionHandler.this.safeClose(WebSocketCloseReasons.CONNECTION_TIMEOUT);
                }
            }
            catch (InterruptedException e) {
                log.debug("UNEXPECTED: Timeout thread interrupted: " + e.getMessage());
            }
        }
    }
}

