/*
 * Decompiled with CFR 0.152.
 */
package ai.vespa.airlift.zstd;

import ai.vespa.airlift.zstd.FrameHeader;
import ai.vespa.airlift.zstd.Util;
import ai.vespa.airlift.zstd.XxHash64;
import ai.vespa.airlift.zstd.ZstdBlockDecompressor;
import ai.vespa.airlift.zstd.ZstdFrameDecompressor;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import sun.misc.Unsafe;

public class ZstdInputStream
extends InputStream {
    private static final int DEFAULT_BUFFER_SIZE = 8192;
    private static final int BUFFER_SIZE_MASK = -8192;
    private static final int MAX_WINDOW_SIZE = 0x800000;
    private final InputStream inputStream;
    private byte[] inputBuffer;
    private int inputPosition;
    private int inputEnd;
    private byte[] outputBuffer;
    private int outputPosition;
    private int outputEnd;
    private boolean isClosed;
    private boolean seenEof;
    private boolean lastBlock;
    private boolean singleSegmentFlag;
    private boolean contentChecksumFlag;
    private long skipBytes;
    private int windowSize;
    private int blockMaximumSize = 131072;
    private int curBlockSize;
    private int curBlockType = -1;
    private FrameHeader curHeader;
    private ZstdBlockDecompressor blockDecompressor;
    private XxHash64 hasher;
    private long evictedInput;

    public ZstdInputStream(InputStream inp, int initialBufferSize) {
        this.inputStream = inp;
        this.inputBuffer = new byte[initialBufferSize];
        this.outputBuffer = new byte[initialBufferSize];
    }

    public ZstdInputStream(InputStream inp) {
        this(inp, 8192);
    }

    @Override
    public int available() {
        return this.outputAvailable();
    }

    @Override
    public int read() throws IOException {
        this.throwIfClosed();
        if (this.ensureGotOutput()) {
            byte b = this.outputBuffer[this.outputPosition++];
            return b & 0xFF;
        }
        return -1;
    }

    @Override
    public int read(byte[] b) throws IOException {
        return this.read(b, 0, b.length);
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        this.throwIfClosed();
        if (this.ensureGotOutput()) {
            len = Math.min(this.outputAvailable(), len);
            System.arraycopy(this.outputBuffer, this.outputPosition, b, off, len);
            this.outputPosition += len;
            return len;
        }
        return -1;
    }

    @Override
    public void close() throws IOException {
        this.throwIfClosed();
        if (!this.seenEof) {
            this.inputStream.close();
        }
        this.isClosed = true;
    }

    private void check(boolean condition, String reason) {
        Util.verify(condition, this.curInputFilePosition(), reason);
    }

    private boolean ensureGotOutput() throws IOException {
        while (this.outputAvailable() == 0 && !this.seenEof) {
            if (!this.ensureGotFrameHeader() || !this.ensureGotBlock()) continue;
            this.decompressBlock();
        }
        if (this.outputAvailable() > 0) {
            return true;
        }
        this.check(this.seenEof, "unable to decode to EOF");
        this.check(this.inputAvailable() == 0, "leftover input at end of file");
        this.check(this.curHeader == null, "unfinished frame at end of file");
        return false;
    }

    private void readMoreInput() throws IOException {
        this.ensureInputSpace(1024);
        int got = this.inputStream.read(this.inputBuffer, this.inputEnd, this.inputSpace());
        if (got == -1) {
            this.seenEof = true;
            this.inputStream.close();
        } else {
            this.inputEnd += got;
        }
    }

    private ByteBuffer inputBB() {
        ByteBuffer bb = ByteBuffer.wrap(this.inputBuffer, this.inputPosition, this.inputAvailable());
        bb.order(ByteOrder.LITTLE_ENDIAN);
        return bb;
    }

    private boolean ensureGotFrameHeader() throws IOException {
        if (this.curHeader != null) {
            return true;
        }
        if (this.inputAvailable() < 8) {
            this.readMoreInput();
            return false;
        }
        ByteBuffer bb = this.inputBB();
        int magic = bb.getInt();
        if (magic >= 407710288 && magic <= 407710303) {
            this.inputPosition += 4;
            this.skipBytes = ((long)bb.getInt() & 0xFFFFFFFFL) + 4L;
            this.inputPosition += 4;
            while (this.skipBytes > 0L) {
                if (this.skipBytes <= (long)this.inputAvailable()) {
                    this.inputPosition = (int)((long)this.inputPosition + this.skipBytes);
                    this.skipBytes = 0L;
                    continue;
                }
                this.skipBytes -= (long)this.inputAvailable();
                this.inputPosition = this.inputEnd;
                this.readMoreInput();
                if (!this.seenEof) continue;
                throw Util.fail(this.curInputFilePosition(), "unfinished skip frame at end of file");
            }
            return false;
        }
        if (magic == -47205080) {
            int fhDesc = 0xFF & bb.get();
            int frameContentSizeFlag = (fhDesc & 0xC0) >> 6;
            this.singleSegmentFlag = (fhDesc & 0x20) != 0;
            this.contentChecksumFlag = (fhDesc & 4) != 0;
            int dictionaryIdFlag = fhDesc & 3;
            int fhSize = 5;
            fhSize = frameContentSizeFlag == 0 ? (fhSize += this.singleSegmentFlag ? 1 : 0) : (fhSize += 1 << frameContentSizeFlag);
            fhSize += this.singleSegmentFlag ? 0 : 1;
            if ((fhSize += 1 << dictionaryIdFlag >> 1) > this.inputAvailable()) {
                this.readMoreInput();
                return false;
            }
            this.inputPosition += 4;
            this.curHeader = this.readFrameHeader();
            this.inputPosition += fhSize - 4;
            this.startFrame();
            return true;
        }
        throw Util.fail(this.curInputFilePosition(), "Invalid magic prefix: " + magic);
    }

    private void startFrame() {
        this.blockDecompressor = new ZstdBlockDecompressor(this.curHeader);
        this.check(this.outputPosition == this.outputEnd, "orphan output present");
        this.outputPosition = 0;
        this.outputEnd = 0;
        if (this.singleSegmentFlag) {
            if (this.curHeader.contentSize > 0x800000L) {
                throw Util.fail(this.curInputFilePosition(), "Single segment too large: " + this.curHeader.contentSize);
            }
            this.blockMaximumSize = this.windowSize = (int)this.curHeader.contentSize;
            this.ensureOutputSpace(this.windowSize);
        } else {
            if (this.curHeader.windowSize > 0x800000) {
                throw Util.fail(this.curInputFilePosition(), "Window size too large: " + this.curHeader.windowSize);
            }
            this.windowSize = this.curHeader.windowSize;
            this.blockMaximumSize = Math.min(this.windowSize, 131072);
            this.ensureOutputSpace(this.blockMaximumSize + this.windowSize);
        }
        if (this.contentChecksumFlag) {
            this.hasher = new XxHash64();
        }
    }

    private boolean ensureGotBlock() throws IOException {
        this.check(this.curHeader != null, "no current frame");
        if (this.curBlockType == -1) {
            if (this.inputAvailable() < 3) {
                this.readMoreInput();
                return false;
            }
            int blkHeader = this.nextByte() | this.nextByte() << 8 | this.nextByte() << 16;
            this.lastBlock = (blkHeader & 1) != 0;
            this.curBlockType = (blkHeader & 6) >> 1;
            this.curBlockSize = blkHeader >> 3;
            this.ensureInputSpace(this.curBlockSize + 4);
        }
        if (this.inputAvailable() < this.curBlockSize + (this.contentChecksumFlag ? 4 : 0)) {
            this.readMoreInput();
            return false;
        }
        return true;
    }

    int nextByte() {
        int r = 0xFF & this.inputBuffer[this.inputPosition];
        ++this.inputPosition;
        return r;
    }

    long inputAddress() {
        return Unsafe.ARRAY_BYTE_BASE_OFFSET + this.inputPosition;
    }

    long inputLimit() {
        return Unsafe.ARRAY_BYTE_BASE_OFFSET + this.inputEnd;
    }

    long outputAddress() {
        return Unsafe.ARRAY_BYTE_BASE_OFFSET + this.outputEnd;
    }

    long outputLimit() {
        return Unsafe.ARRAY_BYTE_BASE_OFFSET + this.outputBuffer.length;
    }

    int decodeRaw() {
        this.check(this.inputAddress() + (long)this.curBlockSize <= this.inputLimit(), "Not enough input bytes");
        this.check(this.outputAddress() + (long)this.curBlockSize <= this.outputLimit(), "Not enough output space");
        return ZstdBlockDecompressor.decodeRawBlock(this.inputBuffer, this.inputAddress(), this.curBlockSize, this.outputBuffer, this.outputAddress(), this.outputLimit());
    }

    int decodeRle() {
        this.check(this.inputAddress() + 1L <= this.inputLimit(), "Not enough input bytes");
        this.check(this.outputAddress() + (long)this.curBlockSize <= this.outputLimit(), "Not enough output space");
        return ZstdBlockDecompressor.decodeRleBlock(this.curBlockSize, this.inputBuffer, this.inputAddress(), this.outputBuffer, this.outputAddress(), this.outputLimit());
    }

    int decodeCompressed() {
        this.check(this.inputAddress() + (long)this.curBlockSize <= this.inputLimit(), "Not enough input bytes");
        this.check(this.outputAddress() + (long)this.blockMaximumSize <= this.outputLimit(), "Not enough output space");
        return this.blockDecompressor.decodeCompressedBlock(this.inputBuffer, this.inputAddress(), this.curBlockSize, this.outputBuffer, this.outputAddress(), this.outputLimit(), this.windowSize, Unsafe.ARRAY_BYTE_BASE_OFFSET);
    }

    private void decompressBlock() {
        this.check(this.outputPosition == this.outputEnd, "orphan output present");
        switch (this.curBlockType) {
            case 0: {
                this.ensureOutputSpace(this.curBlockSize);
                this.outputEnd += this.decodeRaw();
                this.inputPosition += this.curBlockSize;
                break;
            }
            case 1: {
                this.ensureOutputSpace(this.curBlockSize);
                this.outputEnd += this.decodeRle();
                ++this.inputPosition;
                break;
            }
            case 2: {
                this.check(this.curBlockSize < this.blockMaximumSize, "compressed block must be smaller than Block_Maximum_Size");
                this.ensureOutputSpace(this.blockMaximumSize);
                this.outputEnd += this.decodeCompressed();
                this.inputPosition += this.curBlockSize;
                break;
            }
            default: {
                throw Util.fail(this.curInputFilePosition(), "Invalid block type " + this.curBlockType);
            }
        }
        if (this.contentChecksumFlag) {
            this.hasher.update(this.outputBuffer, this.outputPosition, this.outputAvailable());
        }
        this.curBlockType = -1;
        if (this.lastBlock) {
            this.curHeader = null;
            this.blockDecompressor = null;
            if (this.contentChecksumFlag) {
                this.check(this.inputAvailable() >= 4, "missing checksum data");
                long hash = this.hasher.hash();
                int checksum = this.inputBB().getInt();
                if (checksum != (int)hash) {
                    throw Util.fail(this.curInputFilePosition(), String.format("Bad checksum. Expected: %s, actual: %s", Integer.toHexString(checksum), Integer.toHexString((int)hash)));
                }
                this.inputPosition += 4;
                this.hasher = null;
            }
        }
    }

    private int inputAvailable() {
        return this.inputEnd - this.inputPosition;
    }

    private int inputSpace() {
        return this.inputBuffer.length - this.inputEnd;
    }

    private long curInputFilePosition() {
        return this.evictedInput + (long)this.inputPosition;
    }

    private void ensureInputSpace(int size) {
        if (this.inputSpace() < size) {
            if (size < this.inputPosition) {
                System.arraycopy(this.inputBuffer, this.inputPosition, this.inputBuffer, 0, this.inputAvailable());
            } else {
                int newSize = this.inputBuffer.length + size + 8192 & 0xFFFFE000;
                byte[] newBuf = new byte[newSize];
                System.arraycopy(this.inputBuffer, this.inputPosition, newBuf, 0, this.inputAvailable());
                this.inputBuffer = newBuf;
            }
            this.evictedInput += (long)this.inputPosition;
            this.inputEnd = this.inputAvailable();
            this.inputPosition = 0;
        }
    }

    private int outputAvailable() {
        return this.outputEnd - this.outputPosition;
    }

    private int outputSpace() {
        return this.outputBuffer.length - this.outputEnd;
    }

    private void ensureOutputSpace(int size) {
        if (this.outputSpace() < size) {
            byte[] newBuf;
            this.check(this.outputAvailable() == 0, "logic error");
            if (this.windowSize * 4 + size < this.outputPosition) {
                newBuf = this.outputBuffer;
            } else {
                int newSize = this.outputBuffer.length + this.windowSize * 4 + size + 8192 & 0xFFFFE000;
                newBuf = new byte[newSize];
            }
            int sizeToKeep = Math.min(this.outputPosition, this.windowSize);
            System.arraycopy(this.outputBuffer, this.outputPosition - sizeToKeep, newBuf, 0, sizeToKeep);
            this.outputBuffer = newBuf;
            this.outputEnd = sizeToKeep;
            this.outputPosition = sizeToKeep;
        }
    }

    private void throwIfClosed() throws IOException {
        if (this.isClosed) {
            throw new IOException("Input stream is already closed");
        }
    }

    private FrameHeader readFrameHeader() {
        long base = Unsafe.ARRAY_BYTE_BASE_OFFSET + this.inputPosition;
        long limit = Unsafe.ARRAY_BYTE_BASE_OFFSET + this.inputEnd;
        return ZstdFrameDecompressor.readFrameHeader(this.inputBuffer, base, limit);
    }
}

