/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.http.client.core.communication;

import com.yahoo.vespa.http.client.FeedConnectException;
import com.yahoo.vespa.http.client.FeedProtocolException;
import com.yahoo.vespa.http.client.Result;
import com.yahoo.vespa.http.client.config.Endpoint;
import com.yahoo.vespa.http.client.core.Document;
import com.yahoo.vespa.http.client.core.EndpointResult;
import com.yahoo.vespa.http.client.core.Exceptions;
import com.yahoo.vespa.http.client.core.ServerResponseException;
import com.yahoo.vespa.http.client.core.communication.DocumentQueue;
import com.yahoo.vespa.http.client.core.communication.EndpointResultQueue;
import com.yahoo.vespa.http.client.core.communication.GatewayConnection;
import com.yahoo.vespa.http.client.core.communication.GatewayThrottler;
import com.yahoo.vespa.http.client.core.operationProcessor.EndPointResultFactory;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;

class IOThread
implements Runnable,
AutoCloseable {
    private static Logger log = Logger.getLogger(IOThread.class.getName());
    private final Endpoint endpoint;
    private final GatewayConnection client;
    private final DocumentQueue documentQueue;
    private final EndpointResultQueue resultQueue;
    private final Thread thread;
    private final int clusterId;
    private final CountDownLatch running = new CountDownLatch(1);
    private final CountDownLatch stopSignal = new CountDownLatch(1);
    private final int maxChunkSizeBytes;
    private final int maxInFlightRequests;
    private final long localQueueTimeOut;
    private final GatewayThrottler gatewayThrottler;
    private final Random random = new Random();
    private final AtomicInteger wrongSessionDetectedCounter = new AtomicInteger(0);
    private final AtomicInteger wrongVersionDetectedCounter = new AtomicInteger(0);
    private final AtomicInteger problemStatusCodeFromServerCounter = new AtomicInteger(0);
    private final AtomicInteger executeProblemsCounter = new AtomicInteger(0);
    private final AtomicInteger docsReceivedCounter = new AtomicInteger(0);
    private final AtomicInteger statusReceivedCounter = new AtomicInteger(0);
    private final AtomicInteger pendingDocumentStatusCount = new AtomicInteger(0);
    private final AtomicInteger successfulHandshakes = new AtomicInteger(0);
    private final AtomicInteger lastGatewayProcessTimeMillis = new AtomicInteger(0);

    IOThread(ThreadGroup ioThreadGroup, EndpointResultQueue endpointResultQueue, GatewayConnection client, int clusterId, int maxChunkSizeBytes, int maxInFlightRequests, long localQueueTimeOut, DocumentQueue documentQueue, long maxSleepTimeMs) {
        this.documentQueue = documentQueue;
        this.endpoint = client.getEndpoint();
        this.client = client;
        this.resultQueue = endpointResultQueue;
        this.clusterId = clusterId;
        this.maxChunkSizeBytes = maxChunkSizeBytes;
        this.maxInFlightRequests = maxInFlightRequests;
        this.gatewayThrottler = new GatewayThrottler(maxSleepTimeMs);
        this.thread = new Thread(ioThreadGroup, this, "IOThread " + this.endpoint);
        this.thread.setDaemon(true);
        this.localQueueTimeOut = localQueueTimeOut;
        this.thread.start();
    }

    public Endpoint getEndpoint() {
        return this.endpoint;
    }

    public ConnectionStats getConnectionStats() {
        return new ConnectionStats(this.wrongSessionDetectedCounter.get(), this.wrongVersionDetectedCounter.get(), this.problemStatusCodeFromServerCounter.get(), this.executeProblemsCounter.get(), this.docsReceivedCounter.get(), this.statusReceivedCounter.get(), this.pendingDocumentStatusCount.get(), this.successfulHandshakes.get(), this.lastGatewayProcessTimeMillis.get());
    }

    @Override
    public void close() {
        this.documentQueue.close();
        if (this.stopSignal.getCount() == 0L) {
            return;
        }
        this.stopSignal.countDown();
        log.finer("Closed called.");
        int size = this.resultQueue.getPendingSize();
        if (size > 0) {
            log.info("We have outstanding operations (" + size + ") , trying to fetch responses.");
            try {
                this.processResponse(this.client.drain());
            }
            catch (Throwable e) {
                log.log(Level.SEVERE, "Some failures while trying to get latest responses from vespa.", e);
            }
        }
        try {
            this.client.close();
        }
        finally {
            this.drainDocumentQueueWhenFailingPermanently(new Exception("Closed call, did not manage to process everything so failing this document."));
        }
        log.fine("Session to " + this.endpoint + " closed.");
    }

    public void post(Document document) throws InterruptedException {
        this.documentQueue.put(document, true);
    }

    public String toString() {
        return "I/O thread (for " + this.endpoint + ")";
    }

    List<Document> getNextDocsForFeeding(long maxWaitUnits, TimeUnit timeUnit) {
        Document document;
        ArrayList<Document> docsForSendChunk = new ArrayList<Document>();
        int chunkSizeBytes = 0;
        try {
            this.drainFirstDocumentsInQueueIfOld();
            Document doc = this.documentQueue.poll(maxWaitUnits, timeUnit);
            if (doc != null) {
                docsForSendChunk.add(doc);
                chunkSizeBytes = doc.size();
            }
        }
        catch (InterruptedException ie) {
            log.fine("Got break signal while waiting for new documents to feed");
            return docsForSendChunk;
        }
        int thisMaxChunkSizeBytes = this.randomize(this.maxChunkSizeBytes);
        int thisMaxInFlightRequests = this.randomize(this.maxInFlightRequests);
        for (int pendingSize = 1 + this.resultQueue.getPendingSize(); chunkSizeBytes < thisMaxChunkSizeBytes && pendingSize < thisMaxInFlightRequests; chunkSizeBytes += document.size(), ++pendingSize) {
            this.drainFirstDocumentsInQueueIfOld();
            document = this.documentQueue.poll();
            if (document == null) break;
            docsForSendChunk.add(document);
        }
        if (log.isLoggable(Level.FINEST)) {
            log.finest("Chunk has " + docsForSendChunk.size() + " docs with a size " + chunkSizeBytes + " bytes");
        }
        this.docsReceivedCounter.addAndGet(docsForSendChunk.size());
        return docsForSendChunk;
    }

    private int randomize(int limit) {
        double multiplier = 0.75 + 0.25 * this.random.nextDouble();
        return Math.max(1, (int)((double)limit * multiplier));
    }

    private void addDocumentsToResultQueue(List<Document> docs) {
        for (Document doc : docs) {
            this.resultQueue.operationSent(doc.getOperationId());
        }
    }

    private void markDocumentAsFailed(List<Document> docs, ServerResponseException servletException) {
        for (Document doc : docs) {
            this.resultQueue.failOperation(EndPointResultFactory.createTransientError(this.endpoint, doc.getOperationId(), servletException), this.clusterId);
        }
    }

    private InputStream sendAndReceive(List<Document> docs) throws IOException, ServerResponseException {
        try {
            return this.client.writeOperations(docs);
        }
        catch (ServerResponseException ser) {
            this.markDocumentAsFailed(docs, ser);
            throw ser;
        }
        catch (Exception e) {
            this.markDocumentAsFailed(docs, new ServerResponseException(e.getMessage()));
            throw e;
        }
    }

    private ProcessResponse processResponse(InputStream serverResponse) throws IOException {
        Collection<EndpointResult> endpointResults = EndPointResultFactory.createResult(this.endpoint, serverResponse);
        this.statusReceivedCounter.addAndGet(endpointResults.size());
        int transientErrors = 0;
        for (EndpointResult endpointResult : endpointResults) {
            if (endpointResult.getDetail().getResultType() == Result.ResultType.TRANSITIVE_ERROR) {
                ++transientErrors;
            }
            this.resultQueue.resultReceived(endpointResult, this.clusterId);
        }
        return new ProcessResponse(transientErrors, endpointResults.size());
    }

    private ProcessResponse feedDocumentAndProcessResults(List<Document> docs) throws ServerResponseException, IOException {
        this.addDocumentsToResultQueue(docs);
        long startTime = System.currentTimeMillis();
        InputStream serverResponse = this.sendAndReceive(docs);
        ProcessResponse processResponse = this.processResponse(serverResponse);
        this.lastGatewayProcessTimeMillis.set((int)(System.currentTimeMillis() - startTime));
        return processResponse;
    }

    private ProcessResponse pullAndProcessData(long maxWaitTimeMs) throws ServerResponseException, IOException {
        List<Object> nextDocsForFeeding;
        int pendingResultQueueSize = this.resultQueue.getPendingSize();
        this.pendingDocumentStatusCount.set(pendingResultQueueSize);
        List<Object> list = nextDocsForFeeding = pendingResultQueueSize > this.maxInFlightRequests ? new ArrayList() : this.getNextDocsForFeeding(maxWaitTimeMs, TimeUnit.MILLISECONDS);
        if (nextDocsForFeeding.isEmpty() && pendingResultQueueSize == 0) {
            log.finest("No document awaiting feeding, not waiting for results.");
            return new ProcessResponse(0, 0);
        }
        log.finest("Awaiting " + pendingResultQueueSize + " results.");
        ProcessResponse processResponse = this.feedDocumentAndProcessResults(nextDocsForFeeding);
        if (pendingResultQueueSize > this.maxInFlightRequests && processResponse.processResultsCount == 0) {
            try {
                Thread.sleep(300L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        return processResponse;
    }

    private ThreadState cycle(ThreadState threadState) {
        switch (threadState) {
            case DISCONNECTED: {
                try {
                    if (!this.client.connect()) {
                        log.log(Level.WARNING, "Could not connect to endpoint: '" + this.endpoint + "'. Will re-try.");
                        this.drainFirstDocumentsInQueueIfOld();
                        return ThreadState.DISCONNECTED;
                    }
                    return ThreadState.CONNECTED;
                }
                catch (Throwable throwable1) {
                    this.drainFirstDocumentsInQueueIfOld();
                    log.log(Level.INFO, "Failed connecting to endpoint: '" + this.endpoint + "'. Will re-try connecting. Failed with '" + Exceptions.toMessageString(throwable1) + "'", throwable1);
                    this.executeProblemsCounter.incrementAndGet();
                    return ThreadState.DISCONNECTED;
                }
            }
            case CONNECTED: {
                try {
                    this.client.handshake();
                    this.successfulHandshakes.getAndIncrement();
                }
                catch (ServerResponseException ser) {
                    this.executeProblemsCounter.incrementAndGet();
                    log.log(Level.INFO, "Failed talking to endpoint. Handshake with server endpoint '" + this.endpoint + "' failed. Will re-try handshake. Failed with '" + Exceptions.toMessageString(ser) + "'", ser);
                    this.drainFirstDocumentsInQueueIfOld();
                    this.resultQueue.onEndpointError(new FeedProtocolException(ser.getResponseCode(), ser.getResponseString(), ser, this.endpoint));
                    return ThreadState.CONNECTED;
                }
                catch (Throwable throwable) {
                    this.executeProblemsCounter.incrementAndGet();
                    this.resultQueue.onEndpointError(new FeedConnectException(throwable, this.endpoint));
                    log.log(Level.INFO, "Failed talking to endpoint. Handshake with server endpoint '" + this.endpoint + "' failed. Will re-try handshake. Failed with '" + Exceptions.toMessageString(throwable) + "'", throwable);
                    this.drainFirstDocumentsInQueueIfOld();
                    this.client.close();
                    return ThreadState.DISCONNECTED;
                }
                return ThreadState.SESSION_SYNCED;
            }
            case SESSION_SYNCED: {
                try {
                    ProcessResponse processResponse = this.pullAndProcessData(1L);
                    this.gatewayThrottler.handleCall(processResponse.transitiveErrorCount);
                }
                catch (ServerResponseException ser) {
                    log.log(Level.INFO, "Problems while handing data over to endpoint '" + this.endpoint + "'. Will re-try. Endpoint responded with an unexpected HTTP response code. '" + Exceptions.toMessageString(ser) + "'", ser);
                    return ThreadState.CONNECTED;
                }
                catch (Throwable e) {
                    log.log(Level.INFO, "Problems while handing data over to endpoint '" + this.endpoint + "'. Will re-try. Connection level error. Failed with '" + Exceptions.toMessageString(e) + "'", e);
                    this.client.close();
                    return ThreadState.DISCONNECTED;
                }
                return ThreadState.SESSION_SYNCED;
            }
        }
        log.severe("Should never get here.");
        this.client.close();
        return ThreadState.DISCONNECTED;
    }

    private void sleepIfProblemsGettingSyncedConnection(ThreadState newState, ThreadState oldState) {
        if (newState == ThreadState.SESSION_SYNCED) {
            return;
        }
        if (newState == ThreadState.CONNECTED && oldState == ThreadState.DISCONNECTED) {
            return;
        }
        try {
            if (this.stopSignal.getCount() > 0L || !this.documentQueue.isEmpty()) {
                Thread.sleep(this.gatewayThrottler.distribute(3000));
            }
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    @Override
    public void run() {
        ThreadState threadState = ThreadState.DISCONNECTED;
        while (this.stopSignal.getCount() > 0L || !this.documentQueue.isEmpty()) {
            ThreadState oldState = threadState;
            threadState = this.cycle(threadState);
            this.sleepIfProblemsGettingSyncedConnection(threadState, oldState);
        }
        log.finer(this.toString() + " exiting, documentQueue.size()=" + this.documentQueue.size());
        this.running.countDown();
    }

    private void drainFirstDocumentsInQueueIfOld() {
        Optional<Document> document;
        while ((document = this.documentQueue.pollDocumentIfTimedoutInQueue(this.localQueueTimeOut)).isPresent()) {
            EndpointResult endpointResult = EndPointResultFactory.createTransientError(this.endpoint, document.get().getOperationId(), new Exception("Not sending document operation, timed out in queue after " + document.get().timeInQueueMillis() + " ms."));
            this.resultQueue.failOperation(endpointResult, this.clusterId);
        }
        return;
    }

    private void drainDocumentQueueWhenFailingPermanently(Exception exception) {
        this.resultQueue.failPending(exception);
        for (Document document : this.documentQueue.removeAllDocuments()) {
            EndpointResult endpointResult = EndPointResultFactory.createError(this.endpoint, document.getOperationId(), exception);
            this.resultQueue.failOperation(endpointResult, this.clusterId);
        }
    }

    private static class ProcessResponse {
        private final int transitiveErrorCount;
        private final int processResultsCount;

        ProcessResponse(int transitiveErrorCount, int processResultsCount) {
            this.transitiveErrorCount = transitiveErrorCount;
            this.processResultsCount = processResultsCount;
        }
    }

    public static class ConnectionStats {
        public final int wrongSessionDetectedCounter;
        public final int wrongVersionDetectedCounter;
        public final int problemStatusCodeFromServerCounter;
        public final int executeProblemsCounter;
        public final int docsReceivedCounter;
        public final int statusReceivedCounter;
        public final int pendingDocumentStatusCount;
        public final int successfullHandshakes;
        public final int lastGatewayProcessTimeMillis;

        ConnectionStats(int wrongSessionDetectedCounter, int wrongVersionDetectedCounter, int problemStatusCodeFromServerCounter, int executeProblemsCounter, int docsReceivedCounter, int statusReceivedCounter, int pendingDocumentStatusCount, int successfullHandshakes, int lastGatewayProcessTimeMillis) {
            this.wrongSessionDetectedCounter = wrongSessionDetectedCounter;
            this.wrongVersionDetectedCounter = wrongVersionDetectedCounter;
            this.problemStatusCodeFromServerCounter = problemStatusCodeFromServerCounter;
            this.executeProblemsCounter = executeProblemsCounter;
            this.docsReceivedCounter = docsReceivedCounter;
            this.statusReceivedCounter = statusReceivedCounter;
            this.pendingDocumentStatusCount = pendingDocumentStatusCount;
            this.successfullHandshakes = successfullHandshakes;
            this.lastGatewayProcessTimeMillis = lastGatewayProcessTimeMillis;
        }
    }

    private static enum ThreadState {
        DISCONNECTED,
        CONNECTED,
        SESSION_SYNCED;

    }
}

