/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.tensor.serialization;

import com.yahoo.io.GrowableByteBuffer;
import com.yahoo.tensor.Tensor;
import com.yahoo.tensor.TensorAddress;
import com.yahoo.tensor.TensorType;
import com.yahoo.tensor.serialization.BinaryFormat;
import com.yahoo.tensor.serialization.TypedBinaryFormat;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

class SparseBinaryFormat
implements BinaryFormat {
    private final TensorType.Value serializationValueType;

    SparseBinaryFormat() {
        this(TensorType.Value.DOUBLE);
    }

    SparseBinaryFormat(TensorType.Value serializationValueType) {
        this.serializationValueType = serializationValueType;
    }

    @Override
    public void encode(GrowableByteBuffer buffer, Tensor tensor) {
        this.encodeDimensions(buffer, tensor.type().dimensions());
        this.encodeCells(buffer, tensor);
    }

    private void encodeDimensions(GrowableByteBuffer buffer, List<TensorType.Dimension> sortedDimensions) {
        buffer.putInt1_4Bytes(sortedDimensions.size());
        for (TensorType.Dimension dimension : sortedDimensions) {
            buffer.putUtf8String(dimension.name());
        }
    }

    private void encodeCells(GrowableByteBuffer buffer, Tensor tensor) {
        buffer.putInt1_4Bytes(tensor.sizeAsInt());
        switch (this.serializationValueType) {
            case DOUBLE: {
                this.encodeCells(buffer, tensor, buffer::putDouble);
                break;
            }
            case FLOAT: {
                this.encodeCells(buffer, tensor, val -> buffer.putFloat(val.floatValue()));
                break;
            }
            case BFLOAT16: {
                this.encodeCells(buffer, tensor, val -> buffer.putShort(TypedBinaryFormat.bFloat16BitsFromFloat(val.floatValue())));
                break;
            }
            case INT8: {
                this.encodeCells(buffer, tensor, val -> buffer.put((byte)val.floatValue()));
            }
        }
    }

    private void encodeCells(GrowableByteBuffer buffer, Tensor tensor, Consumer<Double> consumer) {
        Iterator<Tensor.Cell> i = tensor.cellIterator();
        while (i.hasNext()) {
            Tensor.Cell cell = i.next();
            this.encodeAddress(buffer, cell.getKey());
            consumer.accept(cell.getValue());
        }
    }

    private void encodeAddress(GrowableByteBuffer buffer, TensorAddress address) {
        for (int i = 0; i < address.size(); ++i) {
            buffer.putUtf8String(address.label(i));
        }
    }

    @Override
    public Tensor decode(Optional<TensorType> optionalType, GrowableByteBuffer buffer) {
        TensorType type;
        if (optionalType.isPresent()) {
            type = optionalType.get();
            if (type.valueType() != this.serializationValueType) {
                throw new IllegalArgumentException("Tensor value type mismatch. Value type " + String.valueOf((Object)type.valueType()) + " is not " + String.valueOf((Object)this.serializationValueType));
            }
            TensorType serializedType = this.decodeType(buffer);
            if (!serializedType.isAssignableTo(type)) {
                throw new IllegalArgumentException("Type/instance mismatch: A tensor of type " + String.valueOf(serializedType) + " cannot be assigned to type " + String.valueOf(type));
            }
        } else {
            type = this.decodeType(buffer);
        }
        Tensor.Builder builder = Tensor.Builder.of(type);
        this.decodeCells(buffer, builder, type);
        return builder.build();
    }

    private TensorType decodeType(GrowableByteBuffer buffer) {
        int numDimensions = buffer.getInt1_4Bytes();
        TensorType.Builder builder = new TensorType.Builder(this.serializationValueType);
        for (int i = 0; i < numDimensions; ++i) {
            builder.mapped(buffer.getUtf8String());
        }
        return builder.build();
    }

    private void decodeCells(GrowableByteBuffer buffer, Tensor.Builder builder, TensorType type) {
        switch (this.serializationValueType) {
            case DOUBLE: {
                this.decodeCells(buffer, builder, type, buffer::getDouble);
                break;
            }
            case FLOAT: {
                this.decodeCells(buffer, builder, type, () -> buffer.getFloat());
                break;
            }
            case BFLOAT16: {
                this.decodeCells(buffer, builder, type, () -> TypedBinaryFormat.floatFromBFloat16Bits(buffer.getShort()));
                break;
            }
            case INT8: {
                this.decodeCells(buffer, builder, type, () -> buffer.get());
            }
        }
    }

    private void decodeCells(GrowableByteBuffer buffer, Tensor.Builder builder, TensorType type, Supplier<Double> supplier) {
        long numCells = buffer.getInt1_4Bytes();
        for (long i = 0L; i < numCells; ++i) {
            Tensor.Builder.CellBuilder cellBuilder = builder.cell();
            this.decodeAddress(buffer, cellBuilder, type);
            cellBuilder.value(supplier.get());
        }
    }

    private void decodeAddress(GrowableByteBuffer buffer, Tensor.Builder.CellBuilder builder, TensorType type) {
        for (TensorType.Dimension dimension : type.dimensions()) {
            String label = buffer.getUtf8String();
            if (label.isEmpty()) continue;
            builder.label(dimension.name(), label);
        }
    }
}

