/*
 * Decompiled with CFR 0.152.
 */
package com.yandex.ydb.table.impl.pool;

import com.yandex.ydb.table.impl.pool.AsyncPool;
import com.yandex.ydb.table.impl.pool.PooledObjectHandler;
import io.netty.util.Timeout;
import io.netty.util.Timer;
import io.netty.util.TimerTask;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Deque;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;

public final class FixedAsyncPool<T>
implements AsyncPool<T> {
    private final Deque<PooledObject<T>> objects = new LinkedList<PooledObject<T>>();
    private final AtomicInteger acquiredObjectsCount = new AtomicInteger(0);
    private final Queue<PendingAcquireTask> pendingAcquireTasks = new ConcurrentLinkedQueue<PendingAcquireTask>();
    private final AtomicInteger pendingAcquireCount = new AtomicInteger(0);
    private final PooledObjectHandler<T> handler;
    private final Timer timer;
    private final KeepAliveTask keepAliveTask;
    private final int minSize;
    private final int maxSize;
    private final int waitQueueMaxSize;
    private volatile boolean closed = false;

    public FixedAsyncPool(PooledObjectHandler<T> handler, Timer timer, int minSize, int maxSize, int waitQueueMaxSize, long keepAliveTimeMillis, long maxIdleTimeMillis) {
        this.handler = handler;
        this.timer = timer;
        this.minSize = minSize;
        this.maxSize = maxSize;
        this.waitQueueMaxSize = waitQueueMaxSize;
        int keepAliveBatchSize = Math.max(2, maxSize / 10);
        this.keepAliveTask = new KeepAliveTask(keepAliveTimeMillis, maxIdleTimeMillis, keepAliveBatchSize);
        this.keepAliveTask.scheduleNext(this.timer);
    }

    @Override
    public int getAcquiredCount() {
        return this.acquiredObjectsCount.get();
    }

    @Override
    public CompletableFuture<T> acquire(Duration timeout) {
        CompletableFuture promise = new CompletableFuture();
        try {
            if (this.closed) {
                promise.completeExceptionally(new IllegalStateException("pool was closed"));
                return promise;
            }
            int count = this.acquiredObjectsCount.get();
            if (count < this.maxSize && this.acquiredObjectsCount.compareAndSet(count, count + 1)) {
                assert (count >= 0);
                this.doAcquireOrCreate(promise);
                return promise;
            }
            long timeoutMillis = timeout.toMillis();
            if (timeoutMillis <= 0L) {
                promise.completeExceptionally(new IllegalStateException("too many acquired objects"));
            } else if (this.pendingAcquireCount.getAndIncrement() < this.waitQueueMaxSize) {
                this.pendingAcquireTasks.offer(new PendingAcquireTask(promise, this.timer, timeoutMillis));
                this.runPendingAcquireTasks();
            } else {
                this.pendingAcquireCount.decrementAndGet();
                promise.completeExceptionally(new IllegalStateException("too many outstanding acquire operations"));
            }
        }
        catch (Throwable cause) {
            promise.completeExceptionally(cause);
        }
        return promise;
    }

    @Override
    public void release(T object) {
        if (this.closed) {
            this.handler.destroy(object);
            throw new IllegalStateException("pool was closed");
        }
        if (this.handler.isValid(object)) {
            if (this.tryToMoveObjectToPendingTask(object)) {
                return;
            }
            this.offerObject(new PooledObject<T>(object, System.currentTimeMillis()));
            this.acquiredObjectsCount.decrementAndGet();
        } else {
            this.acquiredObjectsCount.decrementAndGet();
            this.handler.destroy(object);
        }
        this.runPendingAcquireTasks();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private PooledObject<T> pollObject() {
        Deque<PooledObject<T>> deque = this.objects;
        synchronized (deque) {
            return this.objects.pollLast();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void offerObject(PooledObject<T> object) {
        Deque<PooledObject<T>> deque = this.objects;
        synchronized (deque) {
            this.objects.offerLast(object);
        }
    }

    private void doAcquireOrCreate(CompletableFuture<T> promise) {
        assert (this.acquiredObjectsCount.get() > 0);
        try {
            PooledObject<T> object = this.pollObject();
            if (object != null) {
                this.onAcquire(promise, object.getValue(), null);
                return;
            }
            CompletableFuture<T> future = this.handler.create();
            if (future.isDone() && !future.isCompletedExceptionally()) {
                this.onAcquire(promise, future.getNow(null), null);
                return;
            }
            future.whenComplete((o, ex) -> {
                if (ex != null) {
                    this.acquiredObjectsCount.decrementAndGet();
                    this.onAcquire(promise, null, (Throwable)ex);
                } else if (!promise.complete(o)) {
                    this.release(o);
                }
            });
        }
        catch (Throwable t) {
            this.acquiredObjectsCount.decrementAndGet();
            this.onAcquire(promise, null, t);
        }
    }

    private void onAcquire(CompletableFuture<T> promise, T object, Throwable error) {
        assert (object == null != (error == null));
        if (this.closed) {
            if (error == null) {
                this.handler.destroy(object);
            }
            promise.completeExceptionally(new IllegalStateException("pool was closed"));
        } else if (error == null) {
            promise.complete(object);
        } else {
            promise.completeExceptionally(error);
            this.runPendingAcquireTasks();
        }
    }

    private boolean tryToMoveObjectToPendingTask(T object) {
        PendingAcquireTask task = this.pendingAcquireTasks.poll();
        if (task != null && task.timeout.cancel()) {
            this.pendingAcquireCount.decrementAndGet();
            this.onAcquire(task.promise, object, null);
            return true;
        }
        return false;
    }

    private void runPendingAcquireTasks() {
        int count;
        while ((count = this.acquiredObjectsCount.get()) < this.maxSize && this.acquiredObjectsCount.compareAndSet(count, count + 1)) {
            PendingAcquireTask task = this.pendingAcquireTasks.poll();
            if (task != null && task.timeout.cancel()) {
                this.pendingAcquireCount.decrementAndGet();
                this.doAcquireOrCreate(task.promise);
                continue;
            }
            this.acquiredObjectsCount.decrementAndGet();
            break;
        }
        assert (this.pendingAcquireCount.get() >= 0);
        assert (this.acquiredObjectsCount.get() >= 0);
    }

    @Override
    public void close() {
        PooledObject<T> object;
        PendingAcquireTask task;
        if (this.closed) {
            return;
        }
        this.keepAliveTask.stop();
        IllegalStateException ex = new IllegalStateException("pool was closed");
        while ((task = this.pendingAcquireTasks.poll()) != null) {
            task.promise.completeExceptionally(ex);
            task.timeout.cancel();
        }
        while ((object = this.pollObject()) != null) {
            this.handler.destroy(object.getValue()).join();
        }
        this.acquiredObjectsCount.set(0);
        this.pendingAcquireCount.set(0);
        this.closed = true;
    }

    private final class KeepAliveTask
    implements TimerTask {
        private final long keepAliveTimeMillis;
        private final long maxIdleTimeMillis;
        private final int batchSize;
        private volatile boolean stoped = false;

        KeepAliveTask(long keepAliveTimeMillis, long maxIdleTimeMillis, int batchSize) {
            this.keepAliveTimeMillis = keepAliveTimeMillis;
            this.maxIdleTimeMillis = maxIdleTimeMillis;
            this.batchSize = batchSize;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void run(Timeout timeout) {
            if (this.stoped) {
                return;
            }
            long nowMillis = System.currentTimeMillis();
            ArrayList toDestroy = new ArrayList();
            ArrayList toKeepAlive = new ArrayList();
            Deque deque = FixedAsyncPool.this.objects;
            synchronized (deque) {
                Iterator it = FixedAsyncPool.this.objects.iterator();
                while (it.hasNext()) {
                    PooledObject o = (PooledObject)it.next();
                    if (nowMillis - o.getPooledAt() >= this.maxIdleTimeMillis && FixedAsyncPool.this.objects.size() > FixedAsyncPool.this.minSize) {
                        toDestroy.add(o);
                        it.remove();
                        continue;
                    }
                    if (nowMillis - o.getKeepAlivedAt() < this.keepAliveTimeMillis) continue;
                    toKeepAlive.add(o);
                }
            }
            CompletableFuture<Void> destroy = this.destroy(toDestroy);
            CompletableFuture<Void> keepAlive = this.keepAlive(toKeepAlive);
            CompletableFuture.allOf(destroy, keepAlive).whenComplete((aVoid, throwable) -> {
                if (!this.stoped) {
                    this.scheduleNext(timeout.timer());
                }
            });
        }

        private CompletableFuture<Void> keepAlive(List<PooledObject<T>> objects) {
            if (objects.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            PooledObject[] objectsArr = objects.toArray(new PooledObject[0]);
            Arrays.sort(objectsArr, Comparator.comparing(PooledObject::getKeepAlivedAt));
            int size = Math.min(objectsArr.length, this.batchSize);
            CompletableFuture[] futures = new CompletableFuture[size];
            for (int i = 0; i < size; ++i) {
                PooledObject pooledObject = objectsArr[i];
                futures[i] = FixedAsyncPool.this.handler.keepAlive(pooledObject.getValue()).whenComplete((aVoid, t) -> pooledObject.setKeepAlivedAt(System.currentTimeMillis()));
            }
            return CompletableFuture.allOf(futures);
        }

        private CompletableFuture<Void> destroy(List<PooledObject<T>> objects) {
            if (objects.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            CompletableFuture[] futures = new CompletableFuture[objects.size()];
            for (int i = 0; i < objects.size(); ++i) {
                futures[i] = FixedAsyncPool.this.handler.destroy(objects.get(i).getValue());
            }
            return CompletableFuture.allOf(futures);
        }

        void scheduleNext(Timer timer) {
            long delayMillis = Math.min(1000L, this.keepAliveTimeMillis / 2L);
            timer.newTimeout((TimerTask)this, delayMillis, TimeUnit.MILLISECONDS);
        }

        void stop() {
            this.stoped = true;
        }
    }

    private final class PendingAcquireTask
    implements TimerTask {
        final CompletableFuture<T> promise;
        final long timeoutMillis;
        final Timeout timeout;

        PendingAcquireTask(CompletableFuture<T> promise, Timer timer, long timeoutMillis) {
            this.promise = promise;
            this.timeoutMillis = timeoutMillis;
            this.timeout = timer.newTimeout((TimerTask)this, timeoutMillis, TimeUnit.MILLISECONDS);
        }

        public void run(Timeout timeout) {
            int count = FixedAsyncPool.this.pendingAcquireCount.decrementAndGet();
            assert (count >= 0);
            FixedAsyncPool.this.pendingAcquireTasks.remove(this);
            String msg = "cannot acquire object within " + this.timeoutMillis + "ms";
            FixedAsyncPool.this.onAcquire(this.promise, null, new TimeoutException(msg));
        }
    }

    private static final class PooledObject<T> {
        private final T value;
        private final long pooledAt;
        private volatile long keepAlivedAt;

        PooledObject(T value, long pooledAt) {
            this.value = value;
            this.pooledAt = pooledAt;
            this.keepAlivedAt = pooledAt;
        }

        T getValue() {
            return this.value;
        }

        long getPooledAt() {
            return this.pooledAt;
        }

        long getKeepAlivedAt() {
            return this.keepAlivedAt;
        }

        void setKeepAlivedAt(long keepAlivedAt) {
            this.keepAlivedAt = keepAlivedAt;
        }
    }
}

