/*
 * Decompiled with CFR 0.152.
 */
package com.yandex.ydb.table;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.MoreExecutors;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.core.UnexpectedResultException;
import com.yandex.ydb.core.utils.Async;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.SessionSupplier;
import io.netty.util.Timeout;
import io.netty.util.TimerTask;
import java.time.Duration;
import java.util.EnumSet;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import java.util.function.Function;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

@ParametersAreNonnullByDefault
public class SessionRetryContext {
    private static final EnumSet<StatusCode> RETRYABLE_STATUSES = EnumSet.of(StatusCode.ABORTED, new StatusCode[]{StatusCode.UNAVAILABLE, StatusCode.OVERLOADED, StatusCode.CLIENT_RESOURCE_EXHAUSTED, StatusCode.BAD_SESSION, StatusCode.SESSION_BUSY});
    private final SessionSupplier sessionSupplier;
    private final Executor executor;
    private final int maxRetries;
    private final long backoffSlotMillis;
    private final int backoffCeiling;
    private final long fastBackoffSlotMillis;
    private final int fastBackoffCeiling;
    private final Duration sessionSupplyTimeout;
    private final boolean retryNotFound;
    private final boolean idempotent;

    private SessionRetryContext(Builder b) {
        this.sessionSupplier = b.sessionSupplier;
        this.executor = b.executor;
        this.maxRetries = b.maxRetries;
        this.backoffSlotMillis = b.backoffSlotMillis;
        this.backoffCeiling = b.backoffCeiling;
        this.fastBackoffSlotMillis = b.fastBackoffSlotMillis;
        this.fastBackoffCeiling = b.fastBackoffCeiling;
        this.sessionSupplyTimeout = b.sessionSupplyTimeout;
        this.retryNotFound = b.retryNotFound;
        this.idempotent = b.idempotent;
    }

    public static Builder create(SessionSupplier sessionSupplier) {
        return new Builder(Objects.requireNonNull(sessionSupplier));
    }

    public <T> CompletableFuture<Result<T>> supplyResult(Function<Session, CompletableFuture<Result<T>>> fn) {
        RetryableResultTask<T> task = new RetryableResultTask<T>(fn);
        task.run();
        return task.getFuture();
    }

    public CompletableFuture<Status> supplyStatus(Function<Session, CompletableFuture<Status>> fn) {
        RetryableStatusTask task = new RetryableStatusTask(fn);
        task.run();
        return task.getFuture();
    }

    private boolean canRetry(Throwable t) {
        Throwable cause = Async.unwrapCompletionException((Throwable)t);
        if (cause instanceof UnexpectedResultException) {
            StatusCode statusCode = ((UnexpectedResultException)cause).getStatusCode();
            return this.canRetry(statusCode);
        }
        return false;
    }

    private boolean canRetry(StatusCode code) {
        if (RETRYABLE_STATUSES.contains(code)) {
            return true;
        }
        switch (code) {
            case NOT_FOUND: {
                return this.retryNotFound;
            }
            case CLIENT_CANCELLED: 
            case CLIENT_INTERNAL_ERROR: 
            case UNDETERMINED: 
            case TRANSPORT_UNAVAILABLE: {
                return this.idempotent;
            }
        }
        return false;
    }

    private long backoffTimeMillisInternal(int retryNumber, long backoffSlotMillis, int backoffCeiling) {
        int slots = 1 << Math.min(retryNumber, backoffCeiling);
        long maxDurationMillis = backoffSlotMillis * (long)slots;
        return backoffSlotMillis + ThreadLocalRandom.current().nextLong(maxDurationMillis);
    }

    private long slowBackoffTimeMillis(int retryNumber) {
        return this.backoffTimeMillisInternal(retryNumber, this.backoffSlotMillis, this.backoffCeiling);
    }

    private long fastBackoffTimeMillis(int retryNumber) {
        return this.backoffTimeMillisInternal(retryNumber, this.fastBackoffSlotMillis, this.fastBackoffCeiling);
    }

    private long backoffTimeMillis(StatusCode code, int retryNumber) {
        switch (code) {
            case BAD_SESSION: {
                return 0L;
            }
            case CLIENT_CANCELLED: 
            case CLIENT_INTERNAL_ERROR: 
            case UNDETERMINED: 
            case TRANSPORT_UNAVAILABLE: 
            case ABORTED: 
            case SESSION_BUSY: 
            case UNAVAILABLE: {
                return this.fastBackoffTimeMillis(retryNumber);
            }
        }
        return this.slowBackoffTimeMillis(retryNumber);
    }

    private long backoffTimeMillis(Throwable t, int retryNumber) {
        Throwable cause = Async.unwrapCompletionException((Throwable)t);
        if (cause instanceof UnexpectedResultException) {
            StatusCode statusCode = ((UnexpectedResultException)cause).getStatusCode();
            return this.backoffTimeMillis(statusCode, retryNumber);
        }
        return this.slowBackoffTimeMillis(retryNumber);
    }

    @ParametersAreNonnullByDefault
    public static final class Builder {
        private final SessionSupplier sessionSupplier;
        private Executor executor = MoreExecutors.directExecutor();
        private int maxRetries = 10;
        private long backoffSlotMillis = 1000L;
        private int backoffCeiling = 6;
        private long fastBackoffSlotMillis = 5L;
        private int fastBackoffCeiling = 10;
        private Duration sessionSupplyTimeout = Duration.ofSeconds(5L);
        private boolean retryNotFound = true;
        private boolean idempotent = false;

        public Builder(SessionSupplier sessionSupplier) {
            this.sessionSupplier = sessionSupplier;
        }

        public Builder executor(Executor executor) {
            this.executor = Objects.requireNonNull(executor);
            return this;
        }

        public Builder maxRetries(int maxRetries) {
            this.maxRetries = maxRetries;
            return this;
        }

        public Builder backoffSlot(Duration duration) {
            Preconditions.checkArgument((!duration.isNegative() ? 1 : 0) != 0, (String)"backoffSlot(%s) is negative", (Object)duration);
            this.backoffSlotMillis = duration.toMillis();
            return this;
        }

        public Builder backoffCeiling(int backoffCeiling) {
            this.backoffCeiling = backoffCeiling;
            return this;
        }

        public Builder fastBackoffSlot(Duration duration) {
            Preconditions.checkArgument((!duration.isNegative() ? 1 : 0) != 0, (String)"backoffSlot(%s) is negative", (Object)duration);
            this.fastBackoffSlotMillis = duration.toMillis();
            return this;
        }

        public Builder fastBackoffCeiling(int backoffCeiling) {
            this.fastBackoffCeiling = backoffCeiling;
            return this;
        }

        public Builder sessionSupplyTimeout(Duration duration) {
            Preconditions.checkArgument((!duration.isNegative() ? 1 : 0) != 0, (String)"sessionSupplyTimeout(%s) is negative", (Object)duration);
            this.sessionSupplyTimeout = duration;
            return this;
        }

        public Builder retryNotFound(boolean retryNotFound) {
            this.retryNotFound = retryNotFound;
            return this;
        }

        public Builder idempotent(boolean idempotent) {
            this.idempotent = idempotent;
            return this;
        }

        public SessionRetryContext build() {
            return new SessionRetryContext(this);
        }
    }

    private final class RetryableStatusTask
    extends BaseRetryableTask<Status> {
        RetryableStatusTask(Function<Session, CompletableFuture<Status>> fn) {
            super(fn);
        }

        @Override
        StatusCode toStatusCode(Status status) {
            return status.getCode();
        }

        @Override
        Status toFailedResult(Result<Session> sessionResult) {
            return sessionResult.toStatus();
        }
    }

    private final class RetryableResultTask<T>
    extends BaseRetryableTask<Result<T>> {
        RetryableResultTask(Function<Session, CompletableFuture<Result<T>>> fn) {
            super(fn);
        }

        @Override
        StatusCode toStatusCode(Result<T> result) {
            return result.getCode();
        }

        @Override
        Result<T> toFailedResult(Result<Session> sessionResult) {
            return sessionResult.cast();
        }
    }

    private abstract class BaseRetryableTask<R>
    implements TimerTask,
    BiConsumer<Result<Session>, Throwable> {
        private final CompletableFuture<R> promise = new CompletableFuture();
        private final AtomicInteger retryNumber = new AtomicInteger();
        private final Function<Session, CompletableFuture<R>> fn;

        BaseRetryableTask(Function<Session, CompletableFuture<R>> fn) {
            this.fn = fn;
        }

        CompletableFuture<R> getFuture() {
            return this.promise;
        }

        abstract StatusCode toStatusCode(R var1);

        abstract R toFailedResult(Result<Session> var1);

        public void run(Timeout timeout) {
            if (this.promise.isCancelled()) {
                return;
            }
            this.retryNumber.incrementAndGet();
            SessionRetryContext.this.executor.execute(this::run);
        }

        public void run() {
            CompletableFuture<Result<Session>> sessionFuture = SessionRetryContext.this.sessionSupplier.getOrCreateSession(SessionRetryContext.this.sessionSupplyTimeout);
            if (sessionFuture.isDone() && !sessionFuture.isCompletedExceptionally()) {
                this.accept((Result<Session>)((Result)sessionFuture.getNow(null)), null);
            } else {
                sessionFuture.whenCompleteAsync((BiConsumer)this, SessionRetryContext.this.executor);
            }
        }

        @Override
        public void accept(Result<Session> sessionResult, Throwable sessionException) {
            assert (sessionResult == null != (sessionException == null));
            if (sessionException != null) {
                this.retryIfPossible(null, null, sessionException);
                return;
            }
            if (!sessionResult.isSuccess()) {
                this.retryIfPossible(sessionResult.getCode(), this.toFailedResult(sessionResult), null);
                return;
            }
            Session session = (Session)sessionResult.expect("session must present");
            Async.safeCall((Object)session, this.fn).whenComplete((fnResult, fnException) -> {
                try {
                    session.release();
                    if (fnException != null) {
                        this.retryIfPossible(null, null, (Throwable)fnException);
                        return;
                    }
                    StatusCode statusCode = this.toStatusCode(fnResult);
                    if (statusCode == StatusCode.SUCCESS) {
                        this.promise.complete(fnResult);
                    } else {
                        this.retryIfPossible(statusCode, fnResult, null);
                    }
                }
                catch (Throwable unexpected) {
                    this.promise.completeExceptionally(unexpected);
                }
            });
        }

        private void scheduleNext(long delayMillis) {
            if (this.promise.isCancelled()) {
                return;
            }
            Async.runAfter((TimerTask)this, (long)delayMillis, (TimeUnit)TimeUnit.MILLISECONDS);
        }

        private void retryIfPossible(@Nullable StatusCode code, @Nullable R result, @Nullable Throwable ex) {
            assert (result == null != (ex == null));
            assert (code == null == (result == null));
            int retry = this.retryNumber.incrementAndGet();
            if (ex != null) {
                if (retry <= SessionRetryContext.this.maxRetries && SessionRetryContext.this.canRetry(ex)) {
                    this.scheduleNext(SessionRetryContext.this.backoffTimeMillis(ex, retry));
                } else {
                    this.promise.completeExceptionally(ex);
                }
            } else if (retry <= SessionRetryContext.this.maxRetries && SessionRetryContext.this.canRetry(code)) {
                this.scheduleNext(SessionRetryContext.this.backoffTimeMillis(code, retry));
            } else {
                this.promise.complete(result);
            }
        }
    }
}

