/*
 * Decompiled with CFR 0.152.
 */
package com.yandex.ydb.table.impl.pool;

import com.yandex.ydb.core.utils.Async;
import com.yandex.ydb.table.impl.pool.AsyncPool;
import com.yandex.ydb.table.impl.pool.PooledObjectHandler;
import io.netty.util.Timeout;
import io.netty.util.TimerTask;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Deque;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FixedAsyncPool<T>
implements AsyncPool<T> {
    private static final Logger logger = LoggerFactory.getLogger(FixedAsyncPool.class);
    private final Deque<PooledObject<T>> objects = new LinkedList<PooledObject<T>>();
    private final AtomicInteger acquiredObjectsCount = new AtomicInteger(0);
    private final Queue<PendingAcquireTask> pendingAcquireTasks = new ConcurrentLinkedQueue<PendingAcquireTask>();
    private final AtomicInteger pendingAcquireCount = new AtomicInteger(0);
    private final PooledObjectHandler<T> handler;
    private final KeepAliveTask keepAliveTask;
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private final int minSize;
    private final int maxSize;
    private final int waitQueueMaxSize;

    public FixedAsyncPool(PooledObjectHandler<T> handler, int minSize, int maxSize, int waitQueueMaxSize, long keepAliveTimeMillis, long maxIdleTimeMillis) {
        this.handler = handler;
        this.minSize = minSize;
        this.maxSize = maxSize;
        this.waitQueueMaxSize = waitQueueMaxSize;
        int keepAliveBatchSize = Math.max(2, maxSize / 10);
        this.keepAliveTask = new KeepAliveTask(keepAliveTimeMillis, maxIdleTimeMillis, keepAliveBatchSize);
        this.keepAliveTask.scheduleNext();
    }

    public int getMinSize() {
        return this.minSize;
    }

    public int getMaxSize() {
        return this.maxSize;
    }

    @Override
    public int getAcquiredCount() {
        return this.acquiredObjectsCount.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int getIdleCount() {
        Deque<PooledObject<T>> deque = this.objects;
        synchronized (deque) {
            return this.objects.size();
        }
    }

    @Override
    public int getPendingAcquireCount() {
        return this.pendingAcquireCount.get();
    }

    @Override
    public CompletableFuture<T> acquire(Duration timeout) {
        CompletableFuture promise = new CompletableFuture();
        try {
            if (this.closed.get()) {
                promise.completeExceptionally(new IllegalStateException("pool was closed"));
                return promise;
            }
            long timeoutNanos = timeout.toNanos();
            long deadlineAfter = System.nanoTime() + timeoutNanos;
            int count = this.acquiredObjectsCount.get();
            while (count < this.maxSize) {
                if (!this.acquiredObjectsCount.compareAndSet(count, count + 1)) {
                    count = this.acquiredObjectsCount.get();
                    continue;
                }
                assert (count >= 0);
                this.doAcquireOrCreate(promise, deadlineAfter);
                logger.debug("Acquiring object, current acquired objects count: {}", (Object)this.acquiredObjectsCount.get());
                return promise;
            }
            if (timeoutNanos <= 0L) {
                promise.completeExceptionally(new IllegalStateException("too many acquired objects"));
            } else {
                if (this.pendingAcquireCount.getAndIncrement() < this.waitQueueMaxSize) {
                    this.pendingAcquireTasks.offer(new PendingAcquireTask(promise, timeoutNanos, deadlineAfter));
                    this.runPendingAcquireTasks();
                } else {
                    this.pendingAcquireCount.decrementAndGet();
                    promise.completeExceptionally(new IllegalStateException("too many outstanding acquire operations"));
                }
                logger.debug("Acquire: current pending acquire count: {}", (Object)this.pendingAcquireCount.get());
            }
        }
        catch (Throwable cause) {
            promise.completeExceptionally(cause);
        }
        return promise;
    }

    @Override
    public void release(T object) {
        if (this.closed.get()) {
            logger.debug("Destroy {} because pool already closed", object);
            this.handler.destroy(object);
            throw new IllegalStateException("pool was closed");
        }
        if (this.handler.isValid(object)) {
            if (this.tryToMoveObjectToPendingTask(object)) {
                return;
            }
            this.offerObject(new PooledObject<T>(object, System.currentTimeMillis()));
            this.acquiredObjectsCount.decrementAndGet();
        } else {
            this.acquiredObjectsCount.decrementAndGet();
            logger.debug("Destroy {} because invalid state", object);
            this.handler.destroy(object);
        }
        logger.debug("Object released, current acquired objects count: {}", (Object)this.acquiredObjectsCount.get());
        this.runPendingAcquireTasks();
    }

    void fakeRelease() {
        this.acquiredObjectsCount.decrementAndGet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void offerOrDestroy(T object) {
        if (this.closed.get()) {
            logger.debug("Destroy {} because pool already closed", object);
            this.handler.destroy(object);
            throw new IllegalStateException("pool was closed");
        }
        PooledObject<T> po = new PooledObject<T>(object, System.currentTimeMillis());
        Deque<PooledObject<T>> deque = this.objects;
        synchronized (deque) {
            if (this.acquiredObjectsCount.get() + this.objects.size() < this.maxSize) {
                this.objects.offerLast(po);
                return;
            }
        }
        logger.debug("Destroy {} because max pool size already reached", object);
        this.handler.destroy(object);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private PooledObject<T> pollObject() {
        Deque<PooledObject<T>> deque = this.objects;
        synchronized (deque) {
            return this.objects.pollLast();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void offerObject(PooledObject<T> object) {
        Deque<PooledObject<T>> deque = this.objects;
        synchronized (deque) {
            this.objects.offerLast(object);
        }
    }

    private void doAcquireOrCreate(CompletableFuture<T> promise, long deadlineAfter) {
        assert (this.acquiredObjectsCount.get() > 0);
        try {
            PooledObject<T> object = this.pollObject();
            if (object != null) {
                this.onAcquire(promise, object.getValue(), null);
                return;
            }
            CompletionStage future = this.handler.create(deadlineAfter).thenApply(o -> {
                logger.debug("Created {}", o);
                return o;
            });
            if (((CompletableFuture)future).isDone() && !((CompletableFuture)future).isCompletedExceptionally()) {
                this.onAcquire(promise, ((CompletableFuture)future).getNow(null), null);
                return;
            }
            ((CompletableFuture)future).whenCompleteAsync((o, ex) -> {
                if (ex != null) {
                    this.acquiredObjectsCount.decrementAndGet();
                    this.onAcquire(promise, null, (Throwable)ex);
                } else if (!promise.complete(o)) {
                    this.release(o);
                }
            });
        }
        catch (Throwable t) {
            this.acquiredObjectsCount.decrementAndGet();
            this.onAcquire(promise, null, t);
        }
    }

    private void onAcquire(CompletableFuture<T> promise, T object, Throwable error) {
        assert (object == null != (error == null));
        if (this.closed.get()) {
            if (error == null) {
                logger.debug("Destroy {} because pool already closed", object);
                this.handler.destroy(object);
            }
            promise.completeExceptionally(new IllegalStateException("pool was closed"));
        } else if (error == null) {
            promise.complete(object);
        } else {
            promise.completeExceptionally(error);
            this.runPendingAcquireTasks();
        }
    }

    private boolean tryToMoveObjectToPendingTask(T object) {
        PendingAcquireTask task = this.pendingAcquireTasks.poll();
        if (task != null && task.timeout.cancel()) {
            this.pendingAcquireCount.decrementAndGet();
            logger.debug("Move object to pending task: current pending acquire count: {}", (Object)this.pendingAcquireCount.get());
            this.onAcquire(task.promise, object, null);
            return true;
        }
        return false;
    }

    private void runPendingAcquireTasks() {
        int count;
        while ((count = this.acquiredObjectsCount.get()) < this.maxSize) {
            if (!this.acquiredObjectsCount.compareAndSet(count, count + 1)) continue;
            PendingAcquireTask task = this.pendingAcquireTasks.poll();
            if (task != null && task.timeout.cancel()) {
                this.pendingAcquireCount.decrementAndGet();
                this.doAcquireOrCreate(task.promise, task.deadlineAfter);
                continue;
            }
            this.acquiredObjectsCount.decrementAndGet();
            break;
        }
        logger.debug("Run pending: current pending/acquired count: {}/{}", (Object)this.pendingAcquireCount.get(), (Object)this.acquiredObjectsCount.get());
        assert (this.pendingAcquireCount.get() >= 0);
        assert (this.acquiredObjectsCount.get() >= 0);
    }

    @Override
    public void close() {
        PooledObject<T> object;
        PendingAcquireTask task;
        if (!this.closed.compareAndSet(false, true)) {
            return;
        }
        this.keepAliveTask.stop();
        IllegalStateException ex = new IllegalStateException("pool was closed");
        while ((task = this.pendingAcquireTasks.poll()) != null) {
            task.promise.completeExceptionally(ex);
            task.timeout.cancel();
        }
        while ((object = this.pollObject()) != null) {
            logger.debug("Destroy {} because pool is closed", object);
            CompletableFuture<Void> future = this.handler.destroy(object.getValue());
            try {
                future.get(3L, TimeUnit.SECONDS);
            }
            catch (TimeoutException timeoutException) {
            }
            catch (Exception e) {
                throw new RuntimeException("cannot destroy " + object, e);
            }
        }
        this.acquiredObjectsCount.set(0);
        this.pendingAcquireCount.set(0);
    }

    private final class KeepAliveTask
    implements TimerTask {
        private final long keepAliveTimeMillis;
        private final long maxIdleTimeMillis;
        private final int batchSize;
        private Timeout scheduledHandle = null;
        private volatile boolean stoped = false;

        KeepAliveTask(long keepAliveTimeMillis, long maxIdleTimeMillis, int batchSize) {
            this.keepAliveTimeMillis = keepAliveTimeMillis;
            this.maxIdleTimeMillis = maxIdleTimeMillis;
            this.batchSize = batchSize;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void run(Timeout timeout) {
            if (this.stoped) {
                return;
            }
            long nowMillis = System.currentTimeMillis();
            ArrayList toDestroy = new ArrayList();
            ArrayList toKeepAlive = new ArrayList();
            Deque deque = FixedAsyncPool.this.objects;
            synchronized (deque) {
                Iterator it = FixedAsyncPool.this.objects.iterator();
                while (it.hasNext()) {
                    PooledObject o = (PooledObject)it.next();
                    if (o.getNeedsToBeDestroyed().booleanValue()) {
                        if (logger.isDebugEnabled()) {
                            logger.debug("Destroy {} because it was marked for destruction during previous keep alive task", o.getValue());
                        }
                        toDestroy.add(o);
                        it.remove();
                        continue;
                    }
                    long idleTime = nowMillis - o.getPooledAt();
                    if (idleTime >= this.maxIdleTimeMillis && FixedAsyncPool.this.objects.size() > FixedAsyncPool.this.minSize) {
                        if (logger.isDebugEnabled()) {
                            logger.debug("Destroy {} because idle time {} >= max idle time {}", new Object[]{o.getValue(), idleTime, this.maxIdleTimeMillis});
                        }
                        toDestroy.add(o);
                        it.remove();
                        continue;
                    }
                    if (nowMillis - o.getKeepAlivedAt() < this.keepAliveTimeMillis) continue;
                    toKeepAlive.add(o);
                }
            }
            CompletableFuture<Void> destroy = this.destroy(toDestroy);
            CompletableFuture<Void> keepAlive = this.keepAlive(toKeepAlive);
            CompletableFuture.allOf(destroy, keepAlive).whenComplete((aVoid, throwable) -> {
                if (!this.stoped) {
                    this.scheduleNext();
                }
            });
        }

        private CompletableFuture<Void> keepAlive(List<PooledObject<T>> objects) {
            if (objects.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            PooledObject[] objectsArr = objects.toArray(new PooledObject[0]);
            Arrays.sort(objectsArr, Comparator.comparing(PooledObject::getKeepAlivedAt));
            int size = Math.min(objectsArr.length, this.batchSize);
            CompletableFuture[] futures = new CompletableFuture[size];
            for (int i = 0; i < size; ++i) {
                PooledObject pooledObject = objectsArr[i];
                futures[i] = FixedAsyncPool.this.handler.keepAlive(pooledObject.getValue()).whenComplete((result, throwable) -> {
                    pooledObject.setKeepAlivedAt(System.currentTimeMillis());
                    if (throwable != null) {
                        pooledObject.setNeedsToBeDestroyed(true);
                        logger.warn("Keep alive for " + pooledObject.getValue() + " failed with exception. Marking it for destruction.", throwable);
                    } else {
                        switch (result.getCode()) {
                            case BAD_SESSION: 
                            case SESSION_BUSY: 
                            case INTERNAL_ERROR: {
                                logger.debug("Keep alive for " + pooledObject.getValue() + " failed with code " + result.getCode().toString() + ". Marking it for destruction.");
                                pooledObject.setNeedsToBeDestroyed(true);
                                break;
                            }
                        }
                    }
                });
            }
            return CompletableFuture.allOf(futures);
        }

        private CompletableFuture<Void> destroy(List<PooledObject<T>> objects) {
            if (objects.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            CompletableFuture[] futures = new CompletableFuture[objects.size()];
            for (int i = 0; i < objects.size(); ++i) {
                futures[i] = FixedAsyncPool.this.handler.destroy(objects.get(i).getValue());
            }
            return CompletableFuture.allOf(futures);
        }

        void scheduleNext() {
            long delayMillis = Math.min(1000L, this.keepAliveTimeMillis / 2L);
            this.scheduledHandle = Async.runAfter((TimerTask)this, (long)delayMillis, (TimeUnit)TimeUnit.MILLISECONDS);
        }

        void stop() {
            this.stoped = true;
            if (this.scheduledHandle != null) {
                this.scheduledHandle.cancel();
                this.scheduledHandle = null;
            }
        }
    }

    private final class PendingAcquireTask
    implements TimerTask {
        final CompletableFuture<T> promise;
        final long timeoutNanos;
        final long deadlineAfter;
        final Timeout timeout;

        PendingAcquireTask(CompletableFuture<T> promise, long timeoutNanos, long deadlineAfter) {
            this.promise = promise;
            this.timeoutNanos = timeoutNanos;
            this.deadlineAfter = deadlineAfter;
            this.timeout = Async.runAfter((TimerTask)this, (long)timeoutNanos, (TimeUnit)TimeUnit.NANOSECONDS);
        }

        public void run(Timeout timeout) {
            int count = FixedAsyncPool.this.pendingAcquireCount.decrementAndGet();
            assert (count >= 0);
            FixedAsyncPool.this.pendingAcquireTasks.remove(this);
            String msg = "cannot acquire object within " + TimeUnit.NANOSECONDS.toMillis(this.timeoutNanos) + "ms";
            FixedAsyncPool.this.onAcquire(this.promise, null, new TimeoutException(msg));
        }
    }

    private static final class PooledObject<T> {
        private final T value;
        private final long pooledAt;
        private volatile long keepAlivedAt;
        private volatile Boolean needsToBeDestroyed;

        PooledObject(T value, long pooledAt) {
            this.value = value;
            this.pooledAt = pooledAt;
            this.keepAlivedAt = pooledAt;
            this.needsToBeDestroyed = false;
        }

        T getValue() {
            return this.value;
        }

        long getPooledAt() {
            return this.pooledAt;
        }

        long getKeepAlivedAt() {
            return this.keepAlivedAt;
        }

        void setKeepAlivedAt(long keepAlivedAt) {
            this.keepAlivedAt = keepAlivedAt;
        }

        Boolean getNeedsToBeDestroyed() {
            return this.needsToBeDestroyed;
        }

        void setNeedsToBeDestroyed(Boolean needsToBeDestroyed) {
            this.needsToBeDestroyed = needsToBeDestroyed;
        }
    }
}

