package com.yodo1.mas.mediation.admob;

import android.app.Activity;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.view.Display;
import android.view.View;

import androidx.annotation.NonNull;

import com.google.ads.mediation.admob.AdMobAdapter;
import com.google.android.gms.ads.AdListener;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.AdSize;
import com.google.android.gms.ads.AdValue;
import com.google.android.gms.ads.AdView;
import com.google.android.gms.ads.LoadAdError;
import com.google.android.gms.ads.OnPaidEventListener;
import com.yodo1.mas.Yodo1MasLog;
import com.yodo1.mas.banner.Yodo1MasBannerAdSize;
import com.yodo1.mas.banner.Yodo1MasBannerAdapterBase;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;

import java.util.List;

public class Yodo1MasAdMobBannerAdapter extends Yodo1MasBannerAdapterBase {

    private AdView bannerAd;
    private boolean isMax;
    private double price;
    private final AdListener bannerListener = new AdListener() {

        @Override
        public void onAdOpened() {
            String message = Yodo1MasAdMobUtils.getFormatString("onAdOpened", "banner", bannerAd.getAdUnitId(), bannerAd.getResponseInfo().getMediationAdapterClassName());
            Yodo1MasLog.d(TAG, message);
            bannerState = Yodo1MasAdapterBase.AdvertState.NONE;
            callback(Yodo1MasAdEvent.CODE_OPENED, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdClicked() {
            String message = Yodo1MasAdMobUtils.getFormatString("onAdClicked", "banner", bannerAd.getAdUnitId(), bannerAd.getResponseInfo().getMediationAdapterClassName());
            Yodo1MasLog.d(TAG, message);
        }

        @Override
        public void onAdClosed() {
            String message = Yodo1MasAdMobUtils.getFormatString("onAdClosed", "banner", bannerAd.getAdUnitId(), bannerAd.getResponseInfo().getMediationAdapterClassName());
            Yodo1MasLog.d(TAG, message);
            bannerState = Yodo1MasAdapterBase.AdvertState.NONE;
            callback(Yodo1MasAdEvent.CODE_CLOSED, TAG + ":{" + message + "}");
            loadBannerAdvert();
            adSource = "";
        }

        @Override
        public void onAdLoaded() {
            super.onAdLoaded();
            String message = Yodo1MasAdMobUtils.getFormatString("onAdLoaded", "banner", bannerAd.getAdUnitId(), bannerAd.getResponseInfo().getMediationAdapterClassName());
            Yodo1MasLog.d(TAG, message);
            bannerAd.setOnPaidEventListener(new OnPaidEventListener() {
                @Override
                public void onPaidEvent(@NonNull AdValue adValue) {
                    price = (double) (adValue.getValueMicros() / 1000000);
                }
            });
            bannerState = Yodo1MasAdapterBase.AdvertState.LOADED;
            advertAdapterState = AdvertAdapterState.LOADED;
            adSource = Yodo1MasAdMobUtils.getAdSource(bannerAd.getResponseInfo().getMediationAdapterClassName());
            trackAdRequestSuccessed();
            callback(Yodo1MasAdEvent.CODE_LOADED, TAG + ":{" + message + "}");
            callbackAdapterState();
        }

        @Override
        public void onAdFailedToLoad(@NonNull LoadAdError adError) {
            String message = Yodo1MasAdMobUtils.getAdErrorFormatString("onAdFailedToLoad", "banner", adError);
            Yodo1MasLog.d(TAG, message);
            adSource = "";
            trackAdRequestFailed(adError.getCode(), adError.getMessage());

            bannerState = Yodo1MasAdapterBase.AdvertState.NONE;
            if (isMax()) {
                handleLoadFailedCallback(message);
                nextBanner();
                loadBannerAdvertDelayed();
            } else if (isLastPlacement()) {
                handleLoadFailedCallback(message);
            } else {
                nextBanner();
                loadBannerAdvertDelayed();
            }
        }
    };

    public Yodo1MasAdMobBannerAdapter(List<Yodo1MasAdapterBase.AdId> adIds) {
        super(adIds);
    }

    @Override
    public boolean isMax() {
        return isMax;
    }

    public void setMax(boolean max) {
        isMax = max;
    }

    @Override
    public boolean isBannerAdvertLoaded() {
        return bannerAd != null && !TextUtils.isEmpty(bannerAd.getAdUnitId()) && bannerState == Yodo1MasAdapterBase.AdvertState.LOADED;
    }

    @Override
    public void loadBannerAdvert(@NonNull Activity activity) {
        super.loadBannerAdvert(activity);

        Yodo1MasAdapterBase.AdId adId = getBannerAdId();
        if (!isMax() && adId != null && adId.object != null && !isPriceValuable(adId)) {
            handlePricyNotAbaliable();
            return;
        }
        if (adId != null && !TextUtils.isEmpty(adId.adId)) {
            if (bannerAd == null || !adId.adId.equals(bannerAd.getAdUnitId())) {
                bannerAd = new AdView(activity);
                bannerAd.setAdListener(bannerListener);
                setBannerSize(activity, bannerAd);
                bannerAd.setAdUnitId(adId.adId);
            }
            if (bannerAd != null && bannerState != Yodo1MasAdapterBase.AdvertState.LOADING) {
                String message = "method: loadBannerAdvert, loading banner ad...";
                Yodo1MasLog.d(TAG, message);
                bannerAd.loadAd(buildRequest());
                bannerState = Yodo1MasAdapterBase.AdvertState.LOADING;
                advertAdapterState = AdvertAdapterState.LOADING;
                logAdIdInfo(adId);
            }
        }
    }

    @Override
    public View getBannerView() {
        return bannerAd;
    }

    @Override
    public void dismissBannerAdvert(boolean destroy) {
        super.dismissBannerAdvert(destroy);
        if (bannerAd != null) {
            if (destroy) {
                bannerAd.destroy();
                bannerAd = null;
                bannerState = Yodo1MasAdapterBase.AdvertState.NONE;
                advertAdapterState = AdvertAdapterState.NONE;
            }
        }
    }

    @Override
    public boolean isSupportMultipleInstance() {
        return true;
    }

    private AdRequest buildRequest() {
        final AdRequest.Builder builder = new AdRequest.Builder();
        final Bundle extras = new Bundle();
        if (!Yodo1MasHelper.getInstance().isGDPRUserConsent()) {
            extras.putString("npa", "1");
        }
//        if (Yodo1MasHelper.getInstance().isCOPPAAgeRestricted()) {
//            builder.tagForChildDirectedTreatment(true);
//        }
        if (Yodo1MasHelper.getInstance().isCCPADoNotSell()) {
            extras.putInt("rdp", 1);
        }
        return builder.addNetworkExtrasBundle(AdMobAdapter.class, extras).build();
    }

    private void setBannerSize(Activity activity, AdView bannerAd) {
        Yodo1MasBannerAdSize size = bannerConfig.size;
        Yodo1MasLog.d(TAG, "setBannerSize: size: " + size.name());

        if (Yodo1MasBannerAdSize.Banner == size) {
            bannerAd.setAdSize(AdSize.BANNER);
        } else if (Yodo1MasBannerAdSize.LargeBanner == size) {
            bannerAd.setAdSize(AdSize.LARGE_BANNER);
        } else if (Yodo1MasBannerAdSize.IABMediumRectangle == size) {
            bannerAd.setAdSize(AdSize.MEDIUM_RECTANGLE);
        } else if (Yodo1MasBannerAdSize.SmartBanner == size) {
            bannerAd.setAdSize(AdSize.SMART_BANNER);
        } else if (Yodo1MasBannerAdSize.AdaptiveBanner == size) {
            bannerAd.setAdSize(getAdSize(activity));
        }

    }

    private AdSize getAdSize(Activity activity) {
        // Step 2 - Determine the screen width (less decorations) to use for the ad width.
        Display display = activity.getWindowManager().getDefaultDisplay();
        DisplayMetrics outMetrics = new DisplayMetrics();
        display.getMetrics(outMetrics);

        float widthPixels = outMetrics.widthPixels;
        float density = outMetrics.density;

        int adWidth = (int) (widthPixels / density);

        // Step 3 - Get adaptive ad size and return for setting on the ad view.
        return AdSize.getCurrentOrientationAnchoredAdaptiveBannerAdSize(activity, adWidth);
    }

    private void handleLoadFailedCallback(String message) {
        Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
        callback(error);
        advertAdapterState = AdvertAdapterState.FAILED;
        callbackAdapterState();
    }

    @Override
    public double getBestPrice() {
        return price * 1000;
    }
}
