// Copyright 2019 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.yodo1.mas.mediation.admob.nativetemplate;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Paint;
import android.graphics.Rect;
import android.graphics.Typeface;
import android.graphics.drawable.Drawable;
import android.os.Bundle;
import android.text.TextPaint;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.RatingBar;
import android.widget.TextView;

import androidx.annotation.Nullable;
import androidx.constraintlayout.widget.ConstraintLayout;

import com.google.android.gms.ads.nativead.MediaView;
import com.google.android.gms.ads.nativead.NativeAd;
import com.google.android.gms.ads.nativead.NativeAdView;
import com.yodo1.mas.Yodo1MasLog;
import com.yodo1.mas.mediation.admob.R;
import com.yodo1.mas.nativeads.Yodo1MasNativeSize;
import com.yodo1.mas.utils.Yodo1MasNativeUtil;

/**
 * Base class for a template view. *
 */
public class Yodo1MasAdmobNativeTemplateView extends FrameLayout {
  private static final String TAG = "Yodo1MasAdmobNativeTemplateView";

  private int templateType;
  private Yodo1MasAdmobNativeTemplateStyle styles;
  private NativeAd nativeAd;
  private NativeAdView nativeAdView;

  private TextView primaryView;
  private TextView secondaryView;
  private RatingBar ratingBar;
  private TextView tertiaryView;
  private ImageView iconView;
  private MediaView mediaView;
  private Button callToActionView;
  private ConstraintLayout background;
  private String nativeSize;
  private int containerHeightPx;
  private int containerWidthPx;

  private static final String MEDIUM_TEMPLATE = "medium_template";
  private static final String SMALL_TEMPLATE = "small_template";

  public Yodo1MasAdmobNativeTemplateView(Context context) {
    super(context);
  }

  public Yodo1MasAdmobNativeTemplateView(Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
    initView(context, attrs, null);
  }

  public Yodo1MasAdmobNativeTemplateView(Context context, @Nullable AttributeSet attrs, Bundle extra) {
    super(context, attrs);
    if(extra.containsKey("nativeSize")) {
      nativeSize = extra.getString("nativeSize");
    }
    if(extra.containsKey("containerHeightPx")) {
      containerHeightPx = extra.getInt("containerHeightPx", 0);
    }
    if(extra.containsKey("containerWidthPx")) {
      containerWidthPx = extra.getInt("containerWidthPx", 0);
    }
    Yodo1MasLog.d(TAG, "In constructor method nativeSize: " + nativeSize + " containerHeightPx: " + containerHeightPx + " containerWidthPx: " + containerWidthPx);
    initView(context, attrs, nativeSize);
  }

  public Yodo1MasAdmobNativeTemplateView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    initView(context, attrs, null);
  }

  public Yodo1MasAdmobNativeTemplateView(Context context, AttributeSet attrs, int defStyleAttr, int defStyleRes) {
    super(context, attrs, defStyleAttr, defStyleRes);
    initView(context, attrs, null);
  }

  public void setStyles(Yodo1MasAdmobNativeTemplateStyle styles) {
    this.styles = styles;
    this.applyStyles();
  }

  public NativeAdView getNativeAdView() {
    return nativeAdView;
  }

  private void applyStyles() {

    Drawable mainBackground = styles.getMainBackgroundColor();
    if (mainBackground != null) {
      background.setBackground(mainBackground);
      if (primaryView != null) {
        primaryView.setBackground(mainBackground);
      }
      if (secondaryView != null) {
        secondaryView.setBackground(mainBackground);
      }
      if (tertiaryView != null) {
        tertiaryView.setBackground(mainBackground);
      }
    }

    Typeface primary = styles.getPrimaryTextTypeface();
    if (primary != null && primaryView != null) {
      primaryView.setTypeface(primary);
    }

    Typeface secondary = styles.getSecondaryTextTypeface();
    if (secondary != null && secondaryView != null) {
      secondaryView.setTypeface(secondary);
    }

    Typeface tertiary = styles.getTertiaryTextTypeface();
    if (tertiary != null && tertiaryView != null) {
      tertiaryView.setTypeface(tertiary);
    }

    Typeface ctaTypeface = styles.getCallToActionTextTypeface();
    if (ctaTypeface != null && callToActionView != null) {
      callToActionView.setTypeface(ctaTypeface);
    }

    if (styles.getPrimaryTextTypefaceColor() != null && primaryView != null) {
      primaryView.setTextColor(styles.getPrimaryTextTypefaceColor());
    }

    if (styles.getSecondaryTextTypefaceColor() != null && secondaryView != null) {
      secondaryView.setTextColor(styles.getSecondaryTextTypefaceColor());
    }

    if (styles.getTertiaryTextTypefaceColor() != null && tertiaryView != null) {
      tertiaryView.setTextColor(styles.getTertiaryTextTypefaceColor());
    }

    if (styles.getCallToActionTypefaceColor() != null && callToActionView != null) {
      callToActionView.setTextColor(styles.getCallToActionTypefaceColor());
    }

    float ctaTextSize = styles.getCallToActionTextSize();
    if (ctaTextSize > 0 && callToActionView != null) {
      callToActionView.setTextSize(ctaTextSize);
    }

    float primaryTextSize = styles.getPrimaryTextSize();
    if (primaryTextSize > 0 && primaryView != null) {
      primaryView.setTextSize(primaryTextSize);
    }

    float secondaryTextSize = styles.getSecondaryTextSize();
    if (secondaryTextSize > 0 && secondaryView != null) {
      secondaryView.setTextSize(secondaryTextSize);
    }

    float tertiaryTextSize = styles.getTertiaryTextSize();
    if (tertiaryTextSize > 0 && tertiaryView != null) {
      tertiaryView.setTextSize(tertiaryTextSize);
    }

    Drawable ctaBackground = styles.getCallToActionBackgroundColor();
    if (ctaBackground != null && callToActionView != null) {
      callToActionView.setBackground(ctaBackground);
    }

    Drawable primaryBackground = styles.getPrimaryTextBackgroundColor();
    if (primaryBackground != null && primaryView != null) {
      primaryView.setBackground(primaryBackground);
    }

    Drawable secondaryBackground = styles.getSecondaryTextBackgroundColor();
    if (secondaryBackground != null && secondaryView != null) {
      secondaryView.setBackground(secondaryBackground);
    }

    Drawable tertiaryBackground = styles.getTertiaryTextBackgroundColor();
    if (tertiaryBackground != null && tertiaryView != null) {
      tertiaryView.setBackground(tertiaryBackground);
    }

    invalidate();
    requestLayout();
  }

  private boolean adHasOnlyStore(NativeAd nativeAd) {
    String store = nativeAd.getStore();
    String advertiser = nativeAd.getAdvertiser();
    return !TextUtils.isEmpty(store) && TextUtils.isEmpty(advertiser);
  }

  public void setNativeAd(NativeAd nativeAd) {
    this.nativeAd = nativeAd;

    String store = nativeAd.getStore();
    String advertiser = nativeAd.getAdvertiser();
    String headline = nativeAd.getHeadline();
    String body = nativeAd.getBody();
    String cta = nativeAd.getCallToAction();
    Double starRating = nativeAd.getStarRating();
    NativeAd.Image icon = nativeAd.getIcon();

    String secondaryText;

    nativeAdView.setCallToActionView(callToActionView);
    nativeAdView.setHeadlineView(primaryView);
    nativeAdView.setMediaView(mediaView);
    secondaryView.setVisibility(VISIBLE);
    if (adHasOnlyStore(nativeAd)) {
      nativeAdView.setStoreView(secondaryView);
      secondaryText = store;
    } else if (!TextUtils.isEmpty(advertiser)) {
      nativeAdView.setAdvertiserView(secondaryView);
      secondaryText = advertiser;
    } else {
      secondaryText = "";
    }

    primaryView.setText(headline);
    callToActionView.setText(cta);

    //  Set the secondary view to be the star rating if available.
    if (starRating != null && starRating > 0) {
      secondaryView.setVisibility(GONE);
      ratingBar.setVisibility(VISIBLE);
      ratingBar.setRating(starRating.floatValue());

      nativeAdView.setStarRatingView(ratingBar);
    } else {
      secondaryView.setText(secondaryText);
      secondaryView.setVisibility(VISIBLE);
      ratingBar.setVisibility(GONE);
    }

    if (icon != null) {
      iconView.setVisibility(VISIBLE);
      iconView.setImageDrawable(icon.getDrawable());
      nativeAdView.setIconView(iconView);
    } else {
      iconView.setVisibility(GONE);
    }

    if (tertiaryView != null) {
      tertiaryView.setText(body);
      nativeAdView.setBodyView(tertiaryView);
    }

    nativeAdView.setNativeAd(nativeAd);
    customSubViews();
  }

  /**
   * To prevent memory leaks, make sure to destroy your ad when you don't need it anymore. This
   * method does not destroy the template view.
   * https://developers.google.com/admob/android/native-unified#destroy_ad
   */
  public void destroyNativeAd() {
    nativeAd.destroy();
  }

  public String getTemplateTypeName() {
    if (templateType == R.layout.yodo1_mas_gnt_medium_template_view) {
      return MEDIUM_TEMPLATE;
    } else if (templateType == R.layout.yodo1_mas_gnt_small_template_view) {
      return SMALL_TEMPLATE;
    }
    return "";
  }

  private void initView(Context context, AttributeSet attributeSet, String size) {
    TypedArray attributes =
        context.getTheme().obtainStyledAttributes(attributeSet, R.styleable.TemplateView, 0, 0);

    if(TextUtils.equals(size, Yodo1MasNativeSize.SMALL.getValue())) {
      templateType = R.layout.yodo1_mas_gnt_small_template_view;
    }
    else {
      templateType = R.layout.yodo1_mas_gnt_medium_template_view;
    }
    LayoutInflater inflater =
        (LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
    inflater.inflate(templateType, this);
    inflateView();
  }

  @Override
  public void onFinishInflate() {
    super.onFinishInflate();
  }

  private void inflateView() {
    nativeAdView = (NativeAdView) findViewById(R.id.native_ad_view);
    primaryView = (TextView) findViewById(R.id.primary);
    secondaryView = (TextView) findViewById(R.id.secondary);
    tertiaryView = (TextView) findViewById(R.id.body);

    ratingBar = (RatingBar) findViewById(R.id.rating_bar);
    ratingBar.setEnabled(false);

    callToActionView = (Button) findViewById(R.id.cta);
    iconView = (ImageView) findViewById(R.id.icon);
    mediaView = (MediaView) findViewById(R.id.media_view);
    background = (ConstraintLayout) findViewById(R.id.background);
  }

  /**
   * Call this method after setNativeAd to custom some sub UI view in native ad
   */
  private void customSubViews() {
    //给 primaryView 设置跑马灯特效
    primaryView.setSingleLine(true);
    primaryView.setEllipsize(TextUtils.TruncateAt.MARQUEE);
    primaryView.setMarqueeRepeatLimit(-1);
    primaryView.setFocusable(true);
    primaryView.setFocusableInTouchMode(true);
    primaryView.requestFocus();

    customeMediaViewSize();
  }

  /**
   * Handle the mediaview size by the container size
   */
  private void customeMediaViewSize() {
    if(TextUtils.equals(nativeSize, Yodo1MasNativeSize.SMALL.getValue()) || mediaView == null) {
      return;
    }
    if(containerHeightPx <= 0) {
      Yodo1MasLog.d(TAG, "customeMediaViewSize containerHeightPx value <= 0, the method customeMediaViewSize will not work");
      return;
    }

    // 计算可用高度（减去其他固定高度部分）
    Context context = this.getContext();
    int middleHeight = Yodo1MasNativeUtil.dp2px(context, 60);
    int bodyHeight = getBodyTextViewHegiht(context);
    int ctaHeight = Yodo1MasNativeUtil.dp2px(context, 35);
    int spaceHeight = Yodo1MasNativeUtil.dp2px(context, 70); // 容器总高减去各个组件高度计算得出

    int containerHeight = containerHeightPx;// container.height; 当容器高度为 385dp 时，mediaView 的高度才能到 admob demo 中的默认值 200dp
    int availableHeight = containerHeight - middleHeight - bodyHeight - ctaHeight - spaceHeight;
    Yodo1MasLog.d(TAG, "customeMediaViewSize: containerHeightPx is: " + containerHeight
            + " availableHeight px = " + availableHeight);
    if (availableHeight < Yodo1MasNativeUtil.dp2px(context, 100)) { // 空间不足，隐藏 MediaView
      mediaView.setVisibility(View.GONE);
    } else {
      mediaView.setVisibility(View.VISIBLE);
//      Yodo1MasLog.d(TAG, "customeMediaViewSize: mediaViewHeightPx new value is: " + availableHeight);
      ViewGroup.LayoutParams params = mediaView.getLayoutParams();
      if(mediaView.getLayoutParams() instanceof ConstraintLayout.LayoutParams) {
        params.height = availableHeight;
        mediaView.setLayoutParams(params);
      }
    }

  }

  private int getBodyTextViewHegiht(Context context) {
    int bodyHeight = Yodo1MasNativeUtil.dp2px(context, 20);
    // 下面是计算 body 将要显示的文本的高度值,暂时先注释掉跟ios显示逻辑保存一致
//    if(containerWidthPx > 0) {
//      int padding1 = Yodo1MasNativeUtil.dp2px(context, 2) * 2;//硬编码获取 nativeAdView 的 background 设置的stroke width 值
//      ConstraintLayout bgView = (ConstraintLayout)this.findViewById(R.id.background);
//      int padding2 = bgView.getPaddingLeft() + bgView.getPaddingRight();
//      Log.d(TAG, "getBodyTextViewHegiht: padding1Px = " + padding1 + " padding2Px= " + padding2);
//      int paddingSpacePx = padding1 - padding2;
//      int availableWidth = containerWidthPx - paddingSpacePx;//Yodo1MasNativeUtil.dp2px(Ycontext, (320-2-5-5-2)); //
//      bodyHeight = calculateTextViewHeight(tertiaryView,availableWidth );
//    }
    return bodyHeight;
  }

  /**
     * 此方法为估算值，实际显示高度可能因字体、系统渲染等因素略有差异
     */
  private int calculateTextViewHeight(TextView textView, int availableWidth) {
    String text = textView.getText().toString();
    // 1. 获取 TextView 的文本样式属性
    TextPaint textPaint = textView.getPaint();
    float textSize = textView.getTextSize(); // 单位为 px
    int maxLines = textView.getMaxLines();
    boolean includePadding = true; // 是否包含 padding（根据需求调整）

    // 2. 计算文本的边界
    Rect bounds = new Rect();
    textPaint.getTextBounds(text, 0, text.length(), bounds);
    Paint.FontMetrics fm = textPaint.getFontMetrics();
    float textHeight = fm.descent - fm.ascent; // 单行高度PX（含字体内置间距）

    // 3. 计算多行文本的总高度
    if (availableWidth > 0) {
      // 计算文本实际占用的宽度（考虑换行）
      int textWidth = (int) textPaint.measureText(text);
      int lineCount = (int) Math.ceil((float) textWidth / availableWidth);

      // 限制最大行数
      if (maxLines > 0) {
        lineCount = Math.min(lineCount, maxLines);
      }

      // 总高度 = 行数 × 单行高度 + 行间距（如有）
      textHeight = lineCount * textHeight;
    }

    // 4. 添加 TextView 的 padding（如果需要）
    if (includePadding) {
      textHeight += textView.getPaddingTop() + textView.getPaddingBottom();
    }

//    Yodo1MasLog.d(TAG, "calculateTextViewHeight textHeightPx = " + textHeight + " heightDP = " + Yodo1MasNativeUtil.px2dp(Yodo1MasAdmobNativeTemplateView.this.getContext(), textHeight));
    return (int) textHeight;
  }
}
