package com.yodo1.mas.mediation.admob;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.Display;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.ads.mediation.admob.AdMobAdapter;
import com.google.android.gms.ads.AdError;
import com.google.android.gms.ads.AdListener;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.AdSize;
import com.google.android.gms.ads.AdView;
import com.google.android.gms.ads.FullScreenContentCallback;
import com.google.android.gms.ads.LoadAdError;
import com.google.android.gms.ads.MobileAds;
import com.google.android.gms.ads.OnUserEarnedRewardListener;
import com.google.android.gms.ads.RequestConfiguration;
import com.google.android.gms.ads.initialization.AdapterStatus;
import com.google.android.gms.ads.interstitial.InterstitialAd;
import com.google.android.gms.ads.interstitial.InterstitialAdLoadCallback;
import com.google.android.gms.ads.rewarded.RewardItem;
import com.google.android.gms.ads.rewarded.RewardedAd;
import com.google.android.gms.ads.rewarded.RewardedAdLoadCallback;
import com.yodo1.mas.BuildConfig;
import com.yodo1.mas.Yodo1Mas;
import com.yodo1.mas.analytics.Yodo1MasSensorHelper;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.helper.Yodo1MasBanner;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.helper.model.Yodo1MasAdBuildConfig;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;

import org.json.JSONObject;

public class Yodo1MasAdMobAdapter extends Yodo1MasAdapterBase {

    private RewardedAd rewardAd;
    private InterstitialAd interstitialAd;
    private AdView bannerAd;
    private AdvertState interstitialState = AdvertState.NONE;
    private AdvertState rewardState = AdvertState.NONE;
    private boolean isRewarded = false;

    @Override
    public String getAdvertCode() {
        return "admob";
    }

    @Override
    public String getSDKVersion() {
        return MobileAds.getVersionString();
    }

    @Override
    public String getMediationVersion() {
        return BuildConfig.MAS_VERSION_NAME;
    }

    @Override
    public void initSDK(@NonNull Activity activity, @NonNull Config config, @Nullable InitCallback callback) {
        super.initSDK(activity, config, callback);
        if (!isInitSDK()) {
            init = true;
            if (!isMax()) {
                MobileAds.disableMediationAdapterInitialization(currentActivity);
            }
            preInitialize();
            MobileAds.initialize(currentActivity, initializationStatus -> {
                String message = "method: onInitializationComplete, status: " + initializationStatus.getAdapterStatusMap();
                Log.d(TAG, message);
                updatePrivacy();
                loadRewardAdvert();
                loadInterstitialAdvert();
                loadBannerAdvert();

                if (callback != null) {
                    callback.onAdapterInitSuccessful(getAdvertCode());
                }
            });
        } else {
            if (callback != null) {
                callback.onAdapterInitSuccessful(getAdvertCode());
            }
        }
    }

    public boolean isMax() {
        return false;
    }

    @Override
    public boolean isInitSDK() {
        boolean init = super.isInitSDK();
        if (init) {
            String key = MobileAds.class.getName();
            AdapterStatus status = MobileAds.getInitializationStatus().getAdapterStatusMap().get(key);
            init = status != null && status.getInitializationState() == AdapterStatus.State.READY;
        }
        return init;
    }

    @Override
    public void updatePrivacy() {
        super.updatePrivacy();
        SharedPreferences sharedPref = currentActivity.getPreferences(Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();

        editor.putInt("gad_npa", Yodo1MasHelper.getInstance().isGDPRUserConsent() ? 0 : 1);
        editor.putInt("gad_rdp", Yodo1MasHelper.getInstance().isCCPADoNotSell() ? 1 : 0);
        editor.apply();

        Yodo1MasAdaptersPrivacy.setAdColonyPrivacy();
        Yodo1MasAdaptersPrivacy.setAppLovinPrivacy(currentActivity);
        Yodo1MasAdaptersPrivacy.setFacebookPrivacy();
        Yodo1MasAdaptersPrivacy.setFyberPrivacy();
        Yodo1MasAdaptersPrivacy.setInMobiPrivacy();
        Yodo1MasAdaptersPrivacy.setIronSourcePrivacy();
        Yodo1MasAdaptersPrivacy.setMyTargetPrivacy();
        Yodo1MasAdaptersPrivacy.setTapjoyPrivacy();
        Yodo1MasAdaptersPrivacy.setUnityAdsPrivacy(currentActivity);
        Yodo1MasAdaptersPrivacy.setVunglePrivacy();
    }

    /**
     * According to google's answer
     * MobileAds.setRequestConfiguration should be call before MobileAds.initialize
     * https://groups.google.com/g/google-admob-ads-sdk/c/17oVu0sABjs?pli=1
     */
    private void preInitialize() {
        RequestConfiguration.Builder builder = MobileAds.getRequestConfiguration()
                .toBuilder();
        if (Yodo1MasHelper.getInstance().isCOPPAAgeRestricted()) {
            builder.setTagForChildDirectedTreatment(RequestConfiguration.TAG_FOR_CHILD_DIRECTED_TREATMENT_TRUE);
        }
        RequestConfiguration requestConfiguration = builder.build();
        MobileAds.setRequestConfiguration(requestConfiguration);
    }

    @Override
    public boolean isRewardAdvertLoaded() {
        super.isRewardAdvertLoaded();
        return rewardAd != null && rewardState == AdvertState.LOADED;
    }

    @Override
    public void loadRewardAdvert(@NonNull Activity activity) {
        super.loadRewardAdvert(activity);
        if (!isInitSDK()) {
            return;
        }

        AdId adId = getRewardAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId) && rewardState != AdvertState.LOADING) {
            String message = "method: loadRewardAdvert, loading reward ad...";
            Log.d(TAG, message);
            RewardedAd.load(activity, adId.adId, buildRequest(), rewardLoadListener);
            rewardState = AdvertState.LOADING;
        }
    }

    @Override
    public void showRewardAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showRewardAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Reward, callback)) {
            String message = "method: showRewardAdvert, show reward ad...";
            Log.d(TAG, message);
            rewardAd.show(activity, userEarnedRewardListener);
            rewardAd = null;
            rewardState = AdvertState.NONE;
        }
    }

    @Override
    public void dismissRewardAdvert() {
        super.dismissRewardAdvert();
        rewardAd = null;
    }

    @Override
    public boolean isInterstitialAdvertLoaded() {
        super.isInterstitialAdvertLoaded();
        return interstitialAd != null && interstitialState == AdvertState.LOADED;
    }

    @Override
    public void loadInterstitialAdvert(@NonNull Activity activity) {
        super.loadInterstitialAdvert(activity);
        if (!isInitSDK()) {
            return;
        }
        AdId adId = getInterstitialAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId) && interstitialState != AdvertState.LOADING) {
            String message = "method: loadInterstitialAdvert, loading interstitial ad...";
            Log.d(TAG, message);
            InterstitialAd.load(activity, adId.adId, buildRequest(), interstitialLoadListener);
            interstitialState = AdvertState.LOADING;
        }
    }

    @Override
    public void showInterstitialAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showInterstitialAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Interstitial, callback)) {
            String message = "method: showInterstitialAdvert, show interstitial ad...";
            Log.d(TAG, message);
            interstitialAd.show(activity);
            interstitialState = AdvertState.NONE;
        }
    }

    @Override
    public void dismissInterstitialAdvert() {
        super.dismissInterstitialAdvert();
        interstitialAd = null;
    }

    @Override
    public boolean isBannerAdvertLoaded() {
        super.isBannerAdvertLoaded();
        return bannerAd != null && !TextUtils.isEmpty(bannerAd.getAdUnitId()) && bannerState == AdvertState.LOADED;
    }

    @Override
    public void loadBannerAdvert(@NonNull Activity activity) {
        super.loadBannerAdvert(activity);

        AdId adId = getBannerAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId)) {
            if (bannerAd == null || !adId.adId.equals(bannerAd.getAdUnitId())) {
                bannerAd = new AdView(activity);
                bannerAd.setAdListener(bannerListener);
                Yodo1MasAdBuildConfig adBuildConfig = Yodo1MasHelper.getInstance().getAdBuildConfig();
                if (adBuildConfig != null && adBuildConfig.isEnableAdaptiveBanner()) {
                    bannerAd.setAdSize(getAdSize());
                } else {
                    bannerAd.setAdSize(AdSize.BANNER);
                }
                bannerAd.setAdUnitId(adId.adId);
            }
            if (bannerAd != null && bannerState != AdvertState.LOADING) {
                String message = "method: loadBannerAdvert, loading banner ad...";
                Log.d(TAG, message);
                bannerAd.loadAd(buildRequest());
                bannerState = AdvertState.LOADING;
            }
        }
    }

    private AdSize getAdSize() {
        // Step 2 - Determine the screen width (less decorations) to use for the ad width.
        Display display = currentActivity.getWindowManager().getDefaultDisplay();
        DisplayMetrics outMetrics = new DisplayMetrics();
        display.getMetrics(outMetrics);

        float widthPixels = outMetrics.widthPixels;
        float density = outMetrics.density;

        int adWidth = (int) (widthPixels / density);

        // Step 3 - Get adaptive ad size and return for setting on the ad view.
        return AdSize.getCurrentOrientationAnchoredAdaptiveBannerAdSize(currentActivity, adWidth);
    }

    @Override
    public void showBannerAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showBannerAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Banner, callback)) {
            String message = "method: showBannerAdvert, show banner ad...";
            Log.d(TAG, message);
            Yodo1MasBanner.showBanner(activity, bannerAd, object);
        }
    }

    @Override
    public void dismissBannerAdvert(boolean destroy) {
        super.dismissBannerAdvert(destroy);
        if (bannerAd != null) {
            Yodo1MasBanner.removeBanner(bannerAd);
            if (destroy) {
                bannerAd.destroy();
                bannerAd = null;
                bannerState = AdvertState.NONE;
                loadBannerAdvert();
            }
        }
    }

    private AdRequest buildRequest() {
        final AdRequest.Builder builder = new AdRequest.Builder();
        final Bundle extras = new Bundle();
        if (!Yodo1MasHelper.getInstance().isGDPRUserConsent()) {
            extras.putString("npa", "1");
        }
//        if (Yodo1MasHelper.getInstance().isCOPPAAgeRestricted()) {
//            builder.tagForChildDirectedTreatment(true);
//        }
        if (Yodo1MasHelper.getInstance().isCCPADoNotSell()) {
            extras.putInt("rdp", 1);
        }
        return builder.addNetworkExtrasBundle(AdMobAdapter.class, extras).build();
    }

    private final RewardedAdLoadCallback rewardLoadListener = new RewardedAdLoadCallback() {
        @Override
        public void onAdLoaded(@NonNull RewardedAd rewardedAd) {
            super.onAdLoaded(rewardedAd);
            String message = "method: onAdLoaded, reward, Mediation Adapter Class Name: " + rewardedAd.getResponseInfo().getMediationAdapterClassName();
            Log.d(TAG, message);
            rewardState = AdvertState.LOADED;
            trackAdRequest(Yodo1Mas.AdType.Reward, Yodo1MasSensorHelper.AdResult.SUCCESS);

            Yodo1MasAdMobAdapter.this.rewardAd = rewardedAd;
            rewardedAd.setFullScreenContentCallback(rewardShowListener);
        }

        @Override
        public void onAdFailedToLoad(@NonNull LoadAdError loadAdError) {
            super.onAdFailedToLoad(loadAdError);
            String message = "method: onAdFailedToLoad, reward error: " + loadAdError.toString();
            Log.d(TAG, message);

            rewardState = AdvertState.NONE;
            trackAdRequest(Yodo1Mas.AdType.Reward, Yodo1MasSensorHelper.AdResult.FAIL);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Reward);

            loadRewardAdvertDelayed();
        }
    };

    private final OnUserEarnedRewardListener userEarnedRewardListener = new OnUserEarnedRewardListener() {
        @Override
        public void onUserEarnedReward(@NonNull RewardItem rewardItem) {
            String message = "method: onUserEarnedReward, reward item: " + rewardItem.getType();
            Log.d(TAG, message);

            isRewarded = true;
//            callback(Yodo1MasAdEvent.CODE_REWARD_EARNED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
        }
    };

    private final FullScreenContentCallback rewardShowListener = new FullScreenContentCallback() {
        @Override
        public void onAdFailedToShowFullScreenContent(@NonNull AdError adError) {
            super.onAdFailedToShowFullScreenContent(adError);
            String message = "method: onAdFailedToShowFullScreenContent, reward, error: " + adError.toString();
            Log.d(TAG, message);

            rewardState = AdvertState.NONE;
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Reward);
            nextReward();
            loadRewardAdvert();
        }

        @Override
        public void onAdShowedFullScreenContent() {
            super.onAdShowedFullScreenContent();
            String message = "method: onAdShowedFullScreenContent, reward";
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdDismissedFullScreenContent() {
            super.onAdDismissedFullScreenContent();
            String message = "method: onAdDismissedFullScreenContent, reward";
            Log.d(TAG, message);
            if (isRewarded) {
                callback(Yodo1MasAdEvent.CODE_REWARD_EARNED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
                isRewarded = false;
            }
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
            loadRewardAdvert();
        }

        @Override
        public void onAdImpression() {
            super.onAdImpression();
            String message = "method: onAdImpression, reward";
            Log.d(TAG, message);
        }
    };

    private final InterstitialAdLoadCallback interstitialLoadListener = new InterstitialAdLoadCallback() {
        @Override
        public void onAdLoaded(@NonNull InterstitialAd interstitialAd) {
            super.onAdLoaded(interstitialAd);
            String message = "method: onAdLoaded, interstitial, Mediation Adapter Class Name: " + interstitialAd.getResponseInfo().getMediationAdapterClassName();
            Log.d(TAG, message);
            interstitialState = AdvertState.LOADED;
            trackAdRequest(Yodo1Mas.AdType.Interstitial, Yodo1MasSensorHelper.AdResult.SUCCESS);

            Yodo1MasAdMobAdapter.this.interstitialAd = interstitialAd;
            interstitialAd.setFullScreenContentCallback(interstitialShowListener);
        }

        @Override
        public void onAdFailedToLoad(@NonNull LoadAdError loadAdError) {
            super.onAdFailedToLoad(loadAdError);
            String message = "method: onAdFailedToLoad, interstitial, error: " + loadAdError.toString();
            Log.d(TAG, message);
            interstitialState = AdvertState.NONE;

            trackAdRequest(Yodo1Mas.AdType.Interstitial, Yodo1MasSensorHelper.AdResult.FAIL);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Interstitial);

            nextInterstitial();
            loadInterstitialAdvertDelayed();
        }
    };

    private final FullScreenContentCallback interstitialShowListener = new FullScreenContentCallback() {
        @Override
        public void onAdFailedToShowFullScreenContent(@NonNull AdError adError) {
            super.onAdFailedToShowFullScreenContent(adError);

            String message = "method: onAdFailedToShowFullScreenContent, error: " + adError.toString();
            Log.d(TAG, message);
            interstitialState = AdvertState.NONE;

            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Interstitial);
            nextInterstitial();
            loadInterstitialAdvert();
        }

        @Override
        public void onAdShowedFullScreenContent() {
            super.onAdShowedFullScreenContent();
            String message = "method: onAdShowedFullScreenContent, interstitial";
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdDismissedFullScreenContent() {
            super.onAdDismissedFullScreenContent();
            String message = "method: onAdDismissedFullScreenContent, interstitial";
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
            loadInterstitialAdvert();
        }

        @Override
        public void onAdImpression() {
            super.onAdImpression();
            String message = "method: onAdImpression, interstitial";
            Log.d(TAG, message);
        }
    };

    private final AdListener bannerListener = new AdListener() {

        @Override
        public void onAdOpened() {
            String message = "method: onAdOpened, banner";
            Log.d(TAG, message);
            bannerState = AdvertState.NONE;
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Banner, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdClicked() {
            String message = "method: onAdClicked, banner";
            Log.d(TAG, message);
        }

        @Override
        public void onAdClosed() {
            String message = "method: onAdClosed, banner";
            Log.d(TAG, message);
            bannerState = AdvertState.NONE;
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Banner, TAG + ":{" + message + "}");
            loadBannerAdvert();
        }

        @Override
        public void onAdLoaded() {
            super.onAdLoaded();
            String message = "method: onAdLoaded, banner, Mediation Adapter Class Name: " + (bannerAd == null ? "" : bannerAd.getResponseInfo().getMediationAdapterClassName());
            Log.d(TAG, message);
            bannerState = AdvertState.LOADED;
            trackAdRequest(Yodo1Mas.AdType.Banner, Yodo1MasSensorHelper.AdResult.SUCCESS);
            callback(Yodo1MasAdEvent.CODE_LOADED, Yodo1Mas.AdType.Banner, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdFailedToLoad(LoadAdError adError) {
            String message = "method: onAdFailedToLoad, banner, error: " + adError.toString();
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Banner, Yodo1MasSensorHelper.AdResult.FAIL);

            bannerState = AdvertState.NONE;
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Banner);
            nextBanner();
            loadBannerAdvertDelayed();
        }
    };
}
