package com.yodo1.mas.mediation.admob;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.ads.mediation.admob.AdMobAdapter;
import com.google.android.gms.ads.AdError;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.FullScreenContentCallback;
import com.google.android.gms.ads.LoadAdError;
import com.google.android.gms.ads.MobileAds;
import com.google.android.gms.ads.OnUserEarnedRewardListener;
import com.google.android.gms.ads.RequestConfiguration;
import com.google.android.gms.ads.initialization.AdapterStatus;
import com.google.android.gms.ads.interstitial.InterstitialAd;
import com.google.android.gms.ads.interstitial.InterstitialAdLoadCallback;
import com.google.android.gms.ads.rewarded.RewardItem;
import com.google.android.gms.ads.rewarded.RewardedAd;
import com.google.android.gms.ads.rewarded.RewardedAdLoadCallback;
import com.yodo1.mas.BuildConfig;
import com.yodo1.mas.Yodo1Mas;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;
import com.yodo1.mas.mediation.Yodo1MasBannerAdapterBase;

import org.json.JSONObject;

import java.util.Map;

public class Yodo1MasAdMobAdapter extends Yodo1MasAdapterBase {

    private RewardedAd rewardAd;
    private InterstitialAd interstitialAd;
    private AdvertState interstitialState = AdvertState.NONE;
    private final FullScreenContentCallback interstitialShowListener = new FullScreenContentCallback() {
        @Override
        public void onAdFailedToShowFullScreenContent(@NonNull AdError adError) {
            super.onAdFailedToShowFullScreenContent(adError);

            String message = "method: onAdFailedToShowFullScreenContent, error: " + adError.toString();
            Log.d(TAG, message);
            interstitialState = AdvertState.NONE;

            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            callbackDisplayError(error, Yodo1Mas.AdType.Interstitial, adError.getCode(), adError.getMessage());
            nextInterstitial();
            loadInterstitialAdvert();
        }

        @Override
        public void onAdShowedFullScreenContent() {
            super.onAdShowedFullScreenContent();
            String message = "method: onAdShowedFullScreenContent, interstitial";
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdDismissedFullScreenContent() {
            super.onAdDismissedFullScreenContent();
            String message = "method: onAdDismissedFullScreenContent, interstitial";
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
            loadInterstitialAdvert();
        }

        @Override
        public void onAdImpression() {
            super.onAdImpression();
            String message = "method: onAdImpression, interstitial";
            Log.d(TAG, message);
        }
    };
    private final InterstitialAdLoadCallback interstitialLoadListener = new InterstitialAdLoadCallback() {
        @Override
        public void onAdLoaded(@NonNull InterstitialAd interstitialAd) {
            super.onAdLoaded(interstitialAd);
            String message = "method: onAdLoaded, interstitial, Mediation Adapter Class Name: " + interstitialAd.getResponseInfo().getMediationAdapterClassName();
            Log.d(TAG, message);
            interstitialState = AdvertState.LOADED;
            trackAdRequestSuccessed(Yodo1Mas.AdType.Interstitial);

            Yodo1MasAdMobAdapter.this.interstitialAd = interstitialAd;
            interstitialAd.setFullScreenContentCallback(interstitialShowListener);
        }

        @Override
        public void onAdFailedToLoad(@NonNull LoadAdError loadAdError) {
            super.onAdFailedToLoad(loadAdError);
            String message = "method: onAdFailedToLoad, interstitial, error: " + loadAdError.toString();
            Log.d(TAG, message);
            interstitialState = AdvertState.NONE;

            trackAdRequestFailed(Yodo1Mas.AdType.Interstitial, loadAdError.getCode(), loadAdError.getMessage());
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Interstitial);

            nextInterstitial();
            loadInterstitialAdvertDelayed();
        }
    };
    private AdvertState rewardState = AdvertState.NONE;
    private boolean isRewarded = false;
    private final OnUserEarnedRewardListener userEarnedRewardListener = new OnUserEarnedRewardListener() {
        @Override
        public void onUserEarnedReward(@NonNull RewardItem rewardItem) {
            String message = "method: onUserEarnedReward, reward item: " + rewardItem.getType();
            Log.d(TAG, message);

            isRewarded = true;
        }
    };
    private final FullScreenContentCallback rewardShowListener = new FullScreenContentCallback() {
        @Override
        public void onAdFailedToShowFullScreenContent(@NonNull AdError adError) {
            super.onAdFailedToShowFullScreenContent(adError);
            String message = "method: onAdFailedToShowFullScreenContent, reward, error: " + adError.toString();
            Log.d(TAG, message);

            rewardState = AdvertState.NONE;
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            callbackDisplayError(error, Yodo1Mas.AdType.Reward, adError.getCode(), adError.getMessage());
            nextReward();
            loadRewardAdvert();
        }

        @Override
        public void onAdShowedFullScreenContent() {
            super.onAdShowedFullScreenContent();
            String message = "method: onAdShowedFullScreenContent, reward";
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdDismissedFullScreenContent() {
            super.onAdDismissedFullScreenContent();
            String message = "method: onAdDismissedFullScreenContent, reward";
            Log.d(TAG, message);
            if (isRewarded) {
                callback(Yodo1MasAdEvent.CODE_REWARD_EARNED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
                isRewarded = false;
            }
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
            loadRewardAdvert();
        }

        @Override
        public void onAdImpression() {
            super.onAdImpression();
            String message = "method: onAdImpression, reward";
            Log.d(TAG, message);
        }
    };
    private final RewardedAdLoadCallback rewardLoadListener = new RewardedAdLoadCallback() {
        @Override
        public void onAdLoaded(@NonNull RewardedAd rewardedAd) {
            super.onAdLoaded(rewardedAd);
            String message = "method: onAdLoaded, reward, Mediation Adapter Class Name: " + rewardedAd.getResponseInfo().getMediationAdapterClassName();
            Log.d(TAG, message);
            rewardState = AdvertState.LOADED;
            trackAdRequestSuccessed(Yodo1Mas.AdType.Reward);

            Yodo1MasAdMobAdapter.this.rewardAd = rewardedAd;
            rewardedAd.setFullScreenContentCallback(rewardShowListener);
        }

        @Override
        public void onAdFailedToLoad(@NonNull LoadAdError loadAdError) {
            super.onAdFailedToLoad(loadAdError);
            String message = "method: onAdFailedToLoad, reward error: " + loadAdError.toString();
            Log.d(TAG, message);

            rewardState = AdvertState.NONE;
            trackAdRequestFailed(Yodo1Mas.AdType.Reward, loadAdError.getCode(), loadAdError.getMessage());
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Reward);

            loadRewardAdvertDelayed();
        }
    };

    @Override
    public String getAdvertCode() {
        return "admob";
    }

    @Override
    public String getSDKVersion() {
        return MobileAds.getVersionString();
    }

    @Override
    public String getMediationVersion() {
        return BuildConfig.MAS_VERSION_NAME;
    }

    @Override
    public void initSDK(@NonNull Activity activity, @NonNull Config config, @Nullable InitCallback callback) {
        super.initSDK(activity, config, callback);
        if (!isInitSDK()) {
            init = true;
            if (!isMax()) {
                MobileAds.disableMediationAdapterInitialization(currentActivity);
            }
            preInitialize();
            MobileAds.initialize(currentActivity, initializationStatus -> {
                Map<String, AdapterStatus> statusMap = initializationStatus.getAdapterStatusMap();
                for (String adapterClass : statusMap.keySet()) {
                    AdapterStatus status = statusMap.get(adapterClass);
                    Log.d(TAG, String.format(
                            "Adapter name: %s, Description: %s, Latency: %d",
                            adapterClass, status.getDescription(), status.getLatency()));
                }

                String message = "method: onInitializationComplete";
                Log.d(TAG, message);
                updatePrivacy();
                loadRewardAdvert();
                loadInterstitialAdvert();

                if (callback != null) {
                    callback.onAdapterInitSuccessful(getAdvertCode());
                }
            });
        } else {
            if (callback != null) {
                callback.onAdapterInitSuccessful(getAdvertCode());
            }
        }
    }

    public boolean isMax() {
        return false;
    }

    @Override
    public boolean isInitSDK() {
        boolean init = super.isInitSDK();
        if (init) {
            String key = MobileAds.class.getName();
            AdapterStatus status = MobileAds.getInitializationStatus().getAdapterStatusMap().get(key);
            init = status != null && status.getInitializationState() == AdapterStatus.State.READY;
        }
        return init;
    }

    @Override
    public void updatePrivacy() {
        super.updatePrivacy();
        SharedPreferences sharedPref = currentActivity.getPreferences(Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();

        editor.putInt("gad_npa", Yodo1MasHelper.getInstance().isGDPRUserConsent() ? 0 : 1);
        editor.putInt("gad_rdp", Yodo1MasHelper.getInstance().isCCPADoNotSell() ? 1 : 0);
        editor.apply();

        Yodo1MasAdaptersPrivacy.setAdColonyPrivacy();
        Yodo1MasAdaptersPrivacy.setAppLovinPrivacy(currentActivity);
        Yodo1MasAdaptersPrivacy.setFacebookPrivacy();
        Yodo1MasAdaptersPrivacy.setFyberPrivacy();
        Yodo1MasAdaptersPrivacy.setInMobiPrivacy();
        Yodo1MasAdaptersPrivacy.setIronSourcePrivacy(currentActivity);
        Yodo1MasAdaptersPrivacy.setMyTargetPrivacy();
        Yodo1MasAdaptersPrivacy.setTapjoyPrivacy();
        Yodo1MasAdaptersPrivacy.setUnityAdsPrivacy(currentActivity);
        Yodo1MasAdaptersPrivacy.setVunglePrivacy();

    }

    /**
     * According to google's answer
     * MobileAds.setRequestConfiguration should be call before MobileAds.initialize
     * https://groups.google.com/g/google-admob-ads-sdk/c/17oVu0sABjs?pli=1
     */
    private void preInitialize() {
        RequestConfiguration.Builder builder = MobileAds.getRequestConfiguration()
                .toBuilder();

        int ageRestrictedUserTag = Yodo1MasHelper.getInstance().isCOPPAAgeRestricted() ? RequestConfiguration.TAG_FOR_CHILD_DIRECTED_TREATMENT_TRUE : RequestConfiguration.TAG_FOR_CHILD_DIRECTED_TREATMENT_FALSE;
        builder.setTagForChildDirectedTreatment(ageRestrictedUserTag);

        RequestConfiguration requestConfiguration = builder.build();
        MobileAds.setRequestConfiguration(requestConfiguration);
    }

    @Override
    public boolean isRewardAdvertLoaded() {
        super.isRewardAdvertLoaded();
        return rewardAd != null && rewardState == AdvertState.LOADED;
    }

    @Override
    public void loadRewardAdvert(@NonNull Activity activity) {
        super.loadRewardAdvert(activity);
        if (!isInitSDK()) {
            return;
        }

        AdId adId = getRewardAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId) && rewardState != AdvertState.LOADING) {
            String message = "method: loadRewardAdvert, loading reward ad...";
            Log.d(TAG, message);
            RewardedAd.load(activity, adId.adId, buildRequest(), rewardLoadListener);
            rewardState = AdvertState.LOADING;
        }
    }

    @Override
    public void showRewardAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showRewardAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Reward, callback)) {
            String message = "method: showRewardAdvert, show reward ad...";
            Log.d(TAG, message);
            rewardAd.show(activity, userEarnedRewardListener);
            rewardAd = null;
            rewardState = AdvertState.NONE;
        }
    }

    @Override
    public void dismissRewardAdvert() {
        super.dismissRewardAdvert();
        rewardAd = null;
        rewardState = AdvertState.NONE;
    }

    @Override
    public boolean isInterstitialAdvertLoaded() {
        super.isInterstitialAdvertLoaded();
        return interstitialAd != null && interstitialState == AdvertState.LOADED;
    }

    @Override
    public void loadInterstitialAdvert(@NonNull Activity activity) {
        super.loadInterstitialAdvert(activity);
        if (!isInitSDK()) {
            return;
        }
        AdId adId = getInterstitialAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId) && interstitialState != AdvertState.LOADING) {
            String message = "method: loadInterstitialAdvert, loading interstitial ad...";
            Log.d(TAG, message);
            InterstitialAd.load(activity, adId.adId, buildRequest(), interstitialLoadListener);
            interstitialState = AdvertState.LOADING;
        }
    }

    @Override
    public void showInterstitialAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showInterstitialAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Interstitial, callback)) {
            String message = "method: showInterstitialAdvert, show interstitial ad...";
            Log.d(TAG, message);
            interstitialAd.show(activity);
            interstitialState = AdvertState.NONE;
        }
    }

    @Override
    public void dismissInterstitialAdvert() {
        super.dismissInterstitialAdvert();
        interstitialAd = null;
        interstitialState = AdvertState.NONE;
    }

    @Override
    public Yodo1MasBannerAdapterBase getYodo1MasBannerAdapter() {
        Yodo1MasAdMobBannerAdapter bannerAdapter = new Yodo1MasAdMobBannerAdapter();
        bannerAdapter.bannerAdIds.clear();
        bannerAdapter.bannerAdIds.addAll(bannerAdIds);
        bannerAdapter.relateAdapter = this;
        bannerAdapter.advertCode = getAdvertCode();
        bannerAdapter.sdkVersion = getSDKVersion();
        bannerAdapter.mediationVersion = getMediationVersion();
        if (isMax()) {
            bannerAdapter.setMax(true);
        }
        return bannerAdapter;
    }

    private AdRequest buildRequest() {
        final AdRequest.Builder builder = new AdRequest.Builder();
        final Bundle extras = new Bundle();
        if (!Yodo1MasHelper.getInstance().isGDPRUserConsent()) {
            extras.putString("npa", "1");
        }
        if (Yodo1MasHelper.getInstance().isCCPADoNotSell()) {
            extras.putInt("rdp", 1);
        }
        return builder.addNetworkExtrasBundle(AdMobAdapter.class, extras).build();
    }

}
