package com.yodo1.mas.mediation.admob;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.ads.mediation.admob.AdMobAdapter;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.MobileAds;
import com.google.android.gms.ads.RequestConfiguration;
import com.google.android.gms.ads.initialization.AdapterStatus;
import com.yodo1.mas.BuildConfig;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.interstitial.Yodo1MasInterstitialAdapterBase;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;
import com.yodo1.mas.banner.Yodo1MasBannerAdapterBase;
import com.yodo1.mas.nativeads.Yodo1MasNativeAdapterBase;
import com.yodo1.mas.reward.Yodo1MasRewardAdapterBase;

import java.util.Map;

public class Yodo1MasAdMobAdapter extends Yodo1MasAdapterBase {

    @Override
    public String getAdvertCode() {
        return "admob";
    }

    @Override
    public String getSDKVersion() {
        return MobileAds.getVersionString();
    }

    @Override
    public String getMediationVersion() {
        return BuildConfig.MAS_VERSION_NAME;
    }

    @Override
    public void initSDK(@NonNull Activity activity, @NonNull Config config, @Nullable InitCallback callback) {
        super.initSDK(activity, config, callback);
        if (!isInitSDK()) {
            init = true;
            if (!isMax()) {
                MobileAds.disableMediationAdapterInitialization(currentActivity);
            }
            preInitialize();
            MobileAds.initialize(currentActivity, initializationStatus -> {
                Map<String, AdapterStatus> statusMap = initializationStatus.getAdapterStatusMap();
                for (String adapterClass : statusMap.keySet()) {
                    AdapterStatus status = statusMap.get(adapterClass);
                    Log.d(TAG, String.format(
                            "Adapter name: %s, Description: %s, Latency: %d",
                            adapterClass, status.getDescription(), status.getLatency()));
                }

                String message = "method: onInitializationComplete";
                Log.d(TAG, message);
                updatePrivacy();
                if (callback != null) {
                    callback.onAdapterInitSuccessful(getAdvertCode());
                }
            });
        } else {
            if (callback != null) {
                callback.onAdapterInitSuccessful(getAdvertCode());
            }
        }
    }

    public boolean isMax() {
        return false;
    }

    @Override
    public boolean isInitSDK() {
        boolean init = super.isInitSDK();
        if (init) {
            String key = MobileAds.class.getName();
            AdapterStatus status = MobileAds.getInitializationStatus().getAdapterStatusMap().get(key);
            init = status != null && status.getInitializationState() == AdapterStatus.State.READY;
        }
        return init;
    }

    @Override
    public void updatePrivacy() {
        super.updatePrivacy();
        SharedPreferences sharedPref = currentActivity.getPreferences(Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();

        editor.putInt("gad_npa", Yodo1MasHelper.getInstance().isGDPRUserConsent() ? 0 : 1);
        editor.putInt("gad_rdp", Yodo1MasHelper.getInstance().isCCPADoNotSell() ? 1 : 0);
        editor.apply();

        Yodo1MasAdaptersPrivacy.setAdColonyPrivacy();
        Yodo1MasAdaptersPrivacy.setAppLovinPrivacy(currentActivity);
        Yodo1MasAdaptersPrivacy.setFacebookPrivacy();
        Yodo1MasAdaptersPrivacy.setFyberPrivacy();
        Yodo1MasAdaptersPrivacy.setInMobiPrivacy();
        Yodo1MasAdaptersPrivacy.setIronSourcePrivacy(currentActivity);
        Yodo1MasAdaptersPrivacy.setMyTargetPrivacy();
        Yodo1MasAdaptersPrivacy.setTapjoyPrivacy();
        Yodo1MasAdaptersPrivacy.setUnityAdsPrivacy(currentActivity);
        Yodo1MasAdaptersPrivacy.setVunglePrivacy();

    }

    /**
     * According to google's answer
     * MobileAds.setRequestConfiguration should be call before MobileAds.initialize
     * https://groups.google.com/g/google-admob-ads-sdk/c/17oVu0sABjs?pli=1
     */
    private void preInitialize() {
        RequestConfiguration.Builder builder = MobileAds.getRequestConfiguration()
                .toBuilder();

        int ageRestrictedUserTag = Yodo1MasHelper.getInstance().isCOPPAAgeRestricted() ? RequestConfiguration.TAG_FOR_CHILD_DIRECTED_TREATMENT_TRUE : RequestConfiguration.TAG_FOR_CHILD_DIRECTED_TREATMENT_FALSE;
        builder.setTagForChildDirectedTreatment(ageRestrictedUserTag);

        RequestConfiguration requestConfiguration = builder.build();
        MobileAds.setRequestConfiguration(requestConfiguration);
    }

    // Reward
    @Override
    protected Class<? extends Yodo1MasRewardAdapterBase> rewardAdapterClass() {
        return Yodo1MasAdMobRewardAdapter.class;
    }

    // Interstitial
    @Override
    protected Class<? extends Yodo1MasInterstitialAdapterBase> interstitialAdapterClass() {
        return Yodo1MasAdMobInterstitialAdapter.class;
    }

    // Native
    @Override
    public Class<? extends Yodo1MasNativeAdapterBase> nativeAdapterClass() {
        return null;
    }

    // Banner
    @Override
    public Class<? extends Yodo1MasBannerAdapterBase> bannerAdapterClass() {
        return Yodo1MasAdMobBannerAdapter.class;
    }

    @Override
    public Yodo1MasBannerAdapterBase getBannerAdapter() {
        Yodo1MasAdMobBannerAdapter bannerAdapter = (Yodo1MasAdMobBannerAdapter)super.getBannerAdapter();
        bannerAdapter.setMax(isMax());
        return bannerAdapter;
    }

    public static AdRequest buildRequest() {
        final AdRequest.Builder builder = new AdRequest.Builder();
        final Bundle extras = new Bundle();
        if (!Yodo1MasHelper.getInstance().isGDPRUserConsent()) {
            extras.putString("npa", "1");
        }
        if (Yodo1MasHelper.getInstance().isCCPADoNotSell()) {
            extras.putInt("rdp", 1);
        }
        return builder.addNetworkExtrasBundle(AdMobAdapter.class, extras).build();
    }
}
