package com.yodo1.mas.mediation.applovin;

import android.app.Activity;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.applovin.mediation.MaxAd;
import com.applovin.mediation.MaxAdListener;
import com.applovin.mediation.MaxAdViewAdListener;
import com.applovin.mediation.MaxReward;
import com.applovin.mediation.MaxRewardedAdListener;
import com.applovin.mediation.ads.MaxAdView;
import com.applovin.mediation.ads.MaxInterstitialAd;
import com.applovin.mediation.ads.MaxRewardedAd;
import com.applovin.sdk.AppLovinPrivacySettings;
import com.applovin.sdk.AppLovinSdk;
import com.yodo1.mas.Yodo1Mas;
import com.yodo1.mas.analytics.Yodo1MasSensorHelper;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.helper.Yodo1MasBanner;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;

import org.json.JSONException;
import org.json.JSONObject;

public class Yodo1MasAppLovinMaxAdapter extends Yodo1MasAdapterBase {

    private MaxRewardedAd rewardAd;
    private MaxInterstitialAd interstitialAd;
    private MaxAdView bannerAd;
    private String currBannerUnitId;

    @Override
    public String getAdvertCode() {
        return "APPLOVIN";
    }

    @Override
    public String getSDKVersion() {
        return AppLovinSdk.VERSION;
    }

    @Override
    public String getMediationVersion() {
        return BuildConfig.MAS_VERSION_NAME;
    }

    @Override
    public void initSDK(@NonNull Activity activity, @NonNull Config config, @Nullable InitCallback callback) {
        super.initSDK(activity, config, callback);
        if (!isInitSDK()) {
            init = true;
            if (isMax()) {
                AppLovinSdk.getInstance(currentActivity).setMediationProvider("max");
            }
            AppLovinSdk.initializeSdk(currentActivity, configuration -> {
                String message = "method: onSdkInitialized, configuration: " + configuration.getConsentDialogState();
                Log.d(TAG, message);

                updatePrivacy();
                loadRewardAdvert();
                loadInterstitialAdvert();
                loadBannerAdvert();

                if (callback != null) {
                    callback.onAdapterInitSuccessful(getAdvertCode());
                }
            });
        } else {
            if (callback != null) {
                callback.onAdapterInitSuccessful(getAdvertCode());
            }
        }
    }

    @Override
    public void updatePrivacy() {
        super.updatePrivacy();
        if (currentActivity != null) {
            AppLovinPrivacySettings.setHasUserConsent(Yodo1MasHelper.getInstance().isGDPRUserConsent(), currentActivity);
            AppLovinPrivacySettings.setIsAgeRestrictedUser(Yodo1MasHelper.getInstance().isCOPPAAgeRestricted(), currentActivity);
            AppLovinPrivacySettings.setDoNotSell(Yodo1MasHelper.getInstance().isCCPADoNotSell(), currentActivity);
        }
    }

    public boolean isMax() {
        return true;
    }

    @Override
    public boolean isRewardAdvertLoaded() {
        super.isRewardAdvertLoaded();
        return rewardAd != null && rewardAd.isReady();
    }

    @Override
    public void loadRewardAdvert(@NonNull Activity activity) {
        super.loadRewardAdvert(activity);
        if (!isInitSDK()) return;
        AdId adId = getRewardAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId)) {
            rewardAd = MaxRewardedAd.getInstance(adId.adId, activity);
            rewardAd.setListener(rewardListener);
        }
        if (rewardAd != null) {
            String message = "method: loadRewardAdvert, loading reward ad...";
            Log.d(TAG, message);
            rewardAd.loadAd();
        }
    }

    @Override
    public void showRewardAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showRewardAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Reward, callback)) {
            String message = "method: showRewardAdvert, show reward ad...";
            Log.d(TAG, message);

            String placement = null;
            if (object != null && object.has(Yodo1MasAdapterBase.KEY_ARGUMENT_PLACEMENT)) {
                try {
                    placement = object.getString(Yodo1MasAdapterBase.KEY_ARGUMENT_PLACEMENT);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
            if (isMax() && !TextUtils.isEmpty(placement)) {
                rewardAd.showAd(placement);
            } else {
                rewardAd.showAd();
            }
        }
    }

    @Override
    public void dismissRewardAdvert() {
        super.dismissRewardAdvert();
        if (rewardAd != null) {
            rewardAd.destroy();
        }
        rewardAd = null;
    }

    @Override
    public boolean isInterstitialAdvertLoaded() {
        super.isInterstitialAdvertLoaded();
        return interstitialAd != null && interstitialAd.isReady();
    }

    @Override
    public void loadInterstitialAdvert(@NonNull Activity activity) {
        super.loadInterstitialAdvert(activity);
        if (!isInitSDK()) return;

        AdId adId = getInterstitialAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId)) {
            interstitialAd = new MaxInterstitialAd(adId.adId, activity);
            interstitialAd.setListener(interstitialListener);
        }
        if (interstitialAd != null) {
            String message = "method: loadRewardAdvert, loading interstitial ad...";
            Log.d(TAG, message);
            interstitialAd.loadAd();
        }
    }

    @Override
    public void showInterstitialAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showInterstitialAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Interstitial, callback)) {
            String message = "method: showInterstitialAdvert, show interstitial ad...";
            Log.d(TAG, message);

            String placement = null;
            if (object != null && object.has(Yodo1MasAdapterBase.KEY_ARGUMENT_PLACEMENT)) {
                try {
                    placement = object.getString(Yodo1MasAdapterBase.KEY_ARGUMENT_PLACEMENT);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
            if (isMax() && !TextUtils.isEmpty(placement)) {
                interstitialAd.showAd(placement);
            } else {
                interstitialAd.showAd();
            }
        }
    }

    @Override
    public void dismissInterstitialAdvert() {
        super.dismissInterstitialAdvert();
        if (interstitialAd != null) {
            interstitialAd.destroy();
        }
        interstitialAd = null;
    }

    @Override
    public boolean isBannerAdvertLoaded() {
        super.isBannerAdvertLoaded();
        return bannerAd != null && getBannerAdId() != null && bannerState == AdvertState.LOADED;
    }

    @Override
    public void loadBannerAdvert(@NonNull Activity activity) {
        super.loadBannerAdvert(activity);
        if (!isInitSDK()) return;

        AdId adId = getBannerAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId) && (TextUtils.isEmpty(currBannerUnitId) || !adId.adId.equals(currBannerUnitId))) {
            bannerAd = new MaxAdView(adId.adId, activity);
            bannerAd.setListener(bannerListener);
            currBannerUnitId = adId.adId;
        }
        if (bannerAd != null && bannerState != AdvertState.LOADING) {
            String message = "method: loadBannerAdvert, load banner ad...";
            Log.d(TAG, message);
            bannerAd.loadAd();
            bannerState = AdvertState.LOADING;
        }
    }

    @Override
    public void showBannerAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showBannerAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Banner, callback)) {
            String message = "method: showBannerAdvert, show banner ad...";
            Log.d(TAG, message);

            String placement = null;
            if (object != null && object.has(Yodo1MasAdapterBase.KEY_ARGUMENT_PLACEMENT)) {
                try {
                    placement = object.getString(Yodo1MasAdapterBase.KEY_ARGUMENT_PLACEMENT);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
            if (!TextUtils.isEmpty(placement)) {
                bannerAd.setPlacement(placement);
            }
            Yodo1MasBanner.showBanner(activity, bannerAd, object);
        }
    }

    @Override
    public void dismissBannerAdvert(boolean destroy) {
        super.dismissBannerAdvert(destroy);
        if (bannerAd != null) {
            Yodo1MasBanner.removeBanner(bannerAd);
            if (destroy) {
                bannerAd.destroy();
                bannerAd = null;
                currBannerUnitId = null;
                bannerState = AdvertState.NONE;
                loadBannerAdvert();
            }
        }
    }

    private final MaxRewardedAdListener rewardListener = new MaxRewardedAdListener() {

        @Override
        public void onAdLoaded(MaxAd ad) {
            String message = "method: onAdLoaded, reward: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Reward, Yodo1MasSensorHelper.AdResult.SUCCESS);
        }

        @Override
        public void onAdLoadFailed(String adUnitId, int errorCode) {
            String message = "method: onAdLoadFailed, placementId: " + adUnitId + ", error: " + errorCode;
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Reward, Yodo1MasSensorHelper.AdResult.FAIL);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Reward);

            nextReward();
            loadRewardAdvertDelayed();
        }

        @Override
        public void onAdDisplayed(MaxAd ad) {
            String message = "method: onAdDisplayed, reward: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdDisplayFailed(MaxAd ad, int errorCode) {
            String message = "method: onAdDisplayFailed, reward:" + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId() + ", error: " + errorCode;
            Log.d(TAG, message);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Reward);

            nextReward();
            loadRewardAdvert();
        }

        @Override
        public void onAdHidden(MaxAd ad) {
            String message = "method: onAdHidden, reward: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
            loadRewardAdvert();
        }

        @Override
        public void onAdClicked(MaxAd ad) {
            String message = "method: onAdClicked, reward: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
        }

        @Override
        public void onRewardedVideoStarted(MaxAd ad) {
            String message = "method: onRewardedVideoStarted, reward: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
        }

        @Override
        public void onRewardedVideoCompleted(MaxAd ad) {
            String message = "method: onRewardedVideoCompleted, reward: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
        }

        @Override
        public void onUserRewarded(MaxAd ad, MaxReward reward) {
            String message = "method: onUserRewarded, reward: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_REWARD_EARNED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
        }
    };

    private final MaxAdListener interstitialListener = new MaxAdListener() {

        @Override
        public void onAdLoaded(MaxAd ad) {
            String message = "method: onAdLoaded, interstitial: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Interstitial, Yodo1MasSensorHelper.AdResult.SUCCESS);
        }

        @Override
        public void onAdLoadFailed(String adUnitId, int errorCode) {
            String message = "method: onAdLoadFailed, placementId: " + adUnitId + ", error:" + errorCode;
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Interstitial, Yodo1MasSensorHelper.AdResult.FAIL);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Interstitial);

            nextInterstitial();
            loadInterstitialAdvertDelayed();
        }

        @Override
        public void onAdDisplayed(MaxAd ad) {
            String message = "method: onAdDisplayed, interstitial: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdDisplayFailed(MaxAd ad, int errorCode) {
            String message = "method: onAdDisplayFailed, reward:" + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId() + ", error: " + errorCode;
            Log.d(TAG, message);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Interstitial);

            nextInterstitial();
            loadInterstitialAdvert();
        }

        @Override
        public void onAdHidden(MaxAd ad) {
            String message = "method: onAdHidden, interstitial: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
            loadInterstitialAdvert();
        }

        @Override
        public void onAdClicked(MaxAd ad) {
            String message = "method: onAdClicked, interstitial: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
        }
    };

    private final MaxAdViewAdListener bannerListener = new MaxAdViewAdListener() {
        @Override
        public void onAdExpanded(MaxAd ad) {
            String message = "method: onAdExpanded, banner: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
        }

        @Override
        public void onAdCollapsed(MaxAd ad) {
            String message = "method: onAdCollapsed, banner: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Banner, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdLoaded(MaxAd ad) {
            String message = "method: onAdLoaded, banner: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            bannerState = AdvertState.LOADED;
            trackAdRequest(Yodo1Mas.AdType.Banner, Yodo1MasSensorHelper.AdResult.SUCCESS);
        }

        @Override
        public void onAdLoadFailed(String adUnitId, int errorCode) {
            String message = "method: onAdLoadFailed, placementId: " + adUnitId + ", error:" + errorCode;
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Banner, Yodo1MasSensorHelper.AdResult.FAIL);

            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Banner);
            bannerState = AdvertState.NONE;
            nextBanner();
            loadBannerAdvertDelayed();
        }

        @Override
        public void onAdDisplayed(MaxAd ad) {
            String message = "method: onAdDisplayed, banner: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Banner, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdDisplayFailed(MaxAd ad, int errorCode) {
            String message = "method: onAdDisplayFailed, reward:" + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId() + ", error: " + errorCode;
            Log.d(TAG, message);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Banner);
            bannerState = AdvertState.NONE;
            nextBanner();
            loadBannerAdvert();
        }

        @Override
        public void onAdHidden(MaxAd ad) {
            String message = "method: onAdHidden, banner: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            bannerState = AdvertState.NONE;
            loadBannerAdvert();
        }

        @Override
        public void onAdClicked(MaxAd ad) {
            String message = "method: onAdClicked, banner: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
        }
    };
}
