package com.yodo1.mas.mediation.applovin;

import android.app.Activity;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;

import androidx.annotation.NonNull;

import com.amazon.device.ads.AdError;
import com.amazon.device.ads.AdRegistration;
import com.amazon.device.ads.DTBAdCallback;
import com.amazon.device.ads.DTBAdRequest;
import com.amazon.device.ads.DTBAdResponse;
import com.amazon.device.ads.DTBAdSize;
import com.applovin.mediation.MaxAd;
import com.applovin.mediation.MaxAdFormat;
import com.applovin.mediation.MaxAdRevenueListener;
import com.applovin.mediation.MaxAdViewAdListener;
import com.applovin.mediation.MaxError;
import com.applovin.mediation.ads.MaxAdView;
import com.applovin.sdk.AppLovinSdkUtils;
import com.yodo1.mas.Yodo1MasLog;
import com.yodo1.mas.banner.Yodo1MasBannerAdSize;
import com.yodo1.mas.banner.Yodo1MasBannerAdapterBase;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;

import java.util.List;

public class Yodo1MasAppLovinMaxBannerAdapter extends Yodo1MasBannerAdapterBase {

    private final MaxAdViewAdListener bannerListener = new MaxAdViewAdListener() {
        @Override
        public void onAdExpanded(MaxAd ad) {
            String message = "method: onAdExpanded, banner, AdUnitId: " + ad.getAdUnitId() + ", NetworkName: " + ad.getNetworkName();
            Yodo1MasLog.d(TAG, message);

            callback(Yodo1MasAdEvent.CODE_OPENED, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdCollapsed(MaxAd ad) {
            String message = "method: onAdCollapsed, banner, AdUnitId: " + ad.getAdUnitId() + ", NetworkName: " + ad.getNetworkName();
            Yodo1MasLog.d(TAG, message);

            callback(Yodo1MasAdEvent.CODE_CLOSED, TAG + ":{" + message + "}");

            bannerState = Yodo1MasAdapterBase.AdvertState.NONE;
            loadBannerAdvert();
        }

        @Override
        public void onAdLoaded(MaxAd ad) {
            String message = "method: onAdLoaded, banner, AdUnitId: " + ad.getAdUnitId() + ", NetworkName: " + ad.getNetworkName();
            Yodo1MasLog.d(TAG, message);
            bannerState = Yodo1MasAdapterBase.AdvertState.LOADED;
            adSource = ad.getNetworkName();
            price = ad.getRevenue();
            trackAdRequestSuccessed();
            callback(Yodo1MasAdEvent.CODE_LOADED, TAG + ":{" + message + "}");
            advertAdapterState = AdvertAdapterState.LOADED;
            callbackAdapterState();
        }

        @Override
        public void onAdLoadFailed(String adUnitId, MaxError maxError) {
            String message = "method: onAdLoadFailed, banner, AdUnitId: " + adUnitId + ", error: " + maxError.getWaterfall();
            Yodo1MasLog.d(TAG, message);

            adSource = "";
            trackAdRequestFailed(maxError.getCode(), maxError.getMessage());

            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error);
            bannerState = Yodo1MasAdapterBase.AdvertState.NONE;
            advertAdapterState = AdvertAdapterState.FAILED;
            callbackAdapterState();
            nextBanner();
            loadBannerAdvertDelayed();
        }

        @Override
        public void onAdDisplayed(MaxAd ad) {
        }

        @Override
        public void onAdDisplayFailed(MaxAd ad, MaxError maxError) {
            String message = "method: onAdDisplayFailed, banner, AdUnitId: " + ad.getAdUnitId() + ", NetworkName: " + ad.getNetworkName() + ", error: " + maxError.getWaterfall();
            Yodo1MasLog.d(TAG, message);

            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            callbackDisplayError(error, maxError.getCode(), maxError.getMessage());
            bannerState = Yodo1MasAdapterBase.AdvertState.NONE;

            advertAdapterState = AdvertAdapterState.FAILED;
            callbackAdapterState();
            nextBanner();
            loadBannerAdvert();
        }

        @Override
        public void onAdHidden(MaxAd ad) {
        }

        @Override
        public void onAdClicked(MaxAd ad) {
            String message = "method: onAdClicked, banner, AdUnitId: " + ad.getAdUnitId() + ", NetworkName: " + ad.getNetworkName();
            Yodo1MasLog.d(TAG, message);
        }
    };
    private MaxAdView bannerAd;
    private String currBannerUnitId;
    private MaxAdRevenueListener revenueListener;
    private double price;

    public Yodo1MasAppLovinMaxBannerAdapter(List<Yodo1MasAdapterBase.AdId> adIds) {
        super(adIds);
    }

    @Override
    public boolean isMax() {
        return true;
    }

    @Override
    public boolean isBannerAdvertLoaded() {
        return bannerAd != null && getBannerAdId() != null && bannerState == Yodo1MasAdapterBase.AdvertState.LOADED;
    }

    @Override
    public void loadBannerAdvert(@NonNull Activity activity) {
        super.loadBannerAdvert(activity);
        if (relateAdapter == null || !relateAdapter.isInitSDK()) {
            return;
        }

        Yodo1MasAdapterBase.AdId adId = getBannerAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId) && (TextUtils.isEmpty(currBannerUnitId) || !adId.adId.equals(currBannerUnitId))) {
            bannerAd = new MaxAdView(adId.adId, activity);
            bannerAd.setListener(bannerListener);
            if (this.revenueListener != null) {
                bannerAd.setRevenueListener(revenueListener);
            }
            currBannerUnitId = adId.adId;
        }
        if (bannerAd != null && bannerState != Yodo1MasAdapterBase.AdvertState.LOADING) {
            String message = "method: loadBannerAdvert, load banner ad...";
            Yodo1MasLog.d(TAG, message);

            setBannerSize(activity, bannerAd);
            if (TextUtils.isEmpty((bannerConfig.placement))) {
                bannerAd.setPlacement((bannerConfig.placement));
            }

            String amazonUnitId = Yodo1MasAppLovinMaxUtil.getAmazonUnitId(adId);
            if (!TextUtils.isEmpty(amazonUnitId) && AdRegistration.isInitialized()) {
                loadAmazonAdfirst(amazonUnitId, bannerAd);
            } else {
                bannerAd.loadAd();
            }
            bannerState = Yodo1MasAdapterBase.AdvertState.LOADING;
            advertAdapterState = AdvertAdapterState.LOADING;
            logAdIdInfo(adId);
        }
    }

    private void loadAmazonAdfirst(String amazonUnitId, MaxAdView bannerAd) {
        String amazonAdSlotId;
        MaxAdFormat adFormat;
        amazonAdSlotId = amazonUnitId;
        adFormat = MaxAdFormat.BANNER;

        AppLovinSdkUtils.Size rawSize = adFormat.getSize();
        DTBAdSize size = new DTBAdSize(rawSize.getWidth(), rawSize.getHeight(), amazonAdSlotId);

        DTBAdRequest adLoader = new DTBAdRequest();
        adLoader.setSizes(size);
        adLoader.loadAd(new DTBAdCallback() {
            @Override
            public void onFailure(@NonNull AdError adError) {
                bannerAd.setLocalExtraParameter("amazon_ad_error", adError);
                bannerAd.loadAd();
            }

            @Override
            public void onSuccess(@NonNull DTBAdResponse dtbAdResponse) {
                bannerAd.setLocalExtraParameter("amazon_ad_response", dtbAdResponse);
                bannerAd.loadAd();
            }
        });
    }

    private void setBannerSize(Activity activity, MaxAdView bannerAd) {
        Yodo1MasBannerAdSize size = bannerConfig.size;
        Yodo1MasLog.d(TAG, "setBannerSize: size: " + size.name());
        if (Yodo1MasBannerAdSize.AdaptiveBanner == size) {
            // Stretch to the width of the screen for banners to be fully functional
            int width = ViewGroup.LayoutParams.MATCH_PARENT;
            // Get the adaptive banner height.
            int heightDp = MaxAdFormat.BANNER.getAdaptiveSize(activity).getHeight();
            int heightPx = AppLovinSdkUtils.dpToPx(activity, heightDp);
            bannerAd.setLayoutParams(new FrameLayout.LayoutParams(width, heightPx));
            bannerAd.setExtraParameter("adaptive_banner", "true");
        }
    }

    @Override
    public View getBannerView() {
        return bannerAd;
    }

    @Override
    public void dismissBannerAdvert(boolean destroy) {
        super.dismissBannerAdvert(destroy);
        if (bannerAd != null) {
            if (destroy) {
                bannerAd.destroy();
                bannerAd = null;
                currBannerUnitId = null;
                bannerState = Yodo1MasAdapterBase.AdvertState.NONE;
                advertAdapterState = AdvertAdapterState.NONE;
            }
        }
    }

    @Override
    public boolean isSupportMultipleInstance() {
        return false;
    }

    public void setRevenueListener(MaxAdRevenueListener revenueListener) {
        this.revenueListener = revenueListener;
    }

    @Override
    public double getBestPrice() {
        return price * 1000;
    }
}
