package com.yodo1.mas.mediation.applovin;

import android.app.Activity;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.applovin.adview.AppLovinAdView;
import com.applovin.adview.AppLovinAdViewDisplayErrorCode;
import com.applovin.adview.AppLovinAdViewEventListener;
import com.applovin.adview.AppLovinIncentivizedInterstitial;
import com.applovin.adview.AppLovinInterstitialAd;
import com.applovin.adview.AppLovinInterstitialAdDialog;
import com.applovin.sdk.AppLovinAd;
import com.applovin.sdk.AppLovinAdClickListener;
import com.applovin.sdk.AppLovinAdDisplayListener;
import com.applovin.sdk.AppLovinAdLoadListener;
import com.applovin.sdk.AppLovinAdRewardListener;
import com.applovin.sdk.AppLovinAdSize;
import com.applovin.sdk.AppLovinAdVideoPlaybackListener;
import com.applovin.sdk.AppLovinPrivacySettings;
import com.applovin.sdk.AppLovinSdk;
import com.yodo1.mas.Yodo1Mas;
import com.yodo1.mas.analytics.Yodo1MasSensorHelper;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.helper.Yodo1MasBanner;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;

import org.json.JSONObject;

import java.util.Map;

public class Yodo1MasAppLovinAdapter extends Yodo1MasAdapterBase {

    private AppLovinIncentivizedInterstitial rewardAd;
    private AppLovinAd interstitialAd;
    private AppLovinAdView bannerAd;

    @Override
    public String getAdvertCode() {
        return "applovin";
    }

    @Override
    public String getSDKVersion() {
        return AppLovinSdk.VERSION;
    }

    @Override
    public String getMediationVersion() {
        return BuildConfig.MAS_VERSION_NAME;
    }

    @Override
    public void initSDK(@NonNull Activity activity, @NonNull Config config, @Nullable InitCallback callback) {
        super.initSDK(activity, config, callback);
        if (!isInitSDK()) {
            init = true;
            AppLovinSdk.initializeSdk(currentActivity, configuration -> {
                String message = "method: onSdkInitialized, configuration: " + configuration.getConsentDialogState();
                Log.d(TAG, message);

                updatePrivacy();
                loadRewardAdvert();
                loadInterstitialAdvert();
                loadBannerAdvert();

                if (callback != null) {
                    callback.onAdapterInitSuccessful(getAdvertCode());
                }
            });
        } else {
            if (callback != null) {
                callback.onAdapterInitSuccessful(getAdvertCode());
            }
        }
    }

    @Override
    public void updatePrivacy() {
        super.updatePrivacy();
        setAppLoinPrivacy(currentActivity);
    }

    public static void setAppLoinPrivacy(Activity activity) {
        if (activity == null) {
            return;
        }
        AppLovinPrivacySettings.setHasUserConsent(Yodo1MasHelper.getInstance().isGDPRUserConsent(), activity);
        AppLovinPrivacySettings.setIsAgeRestrictedUser(Yodo1MasHelper.getInstance().isCOPPAAgeRestricted(), activity);
        AppLovinPrivacySettings.setDoNotSell(Yodo1MasHelper.getInstance().isCCPADoNotSell(), activity);
    }

    @Override
    public boolean isRewardAdvertLoaded() {
        super.isRewardAdvertLoaded();
        return rewardAd != null && rewardAd.isAdReadyToDisplay();
    }

    @Override
    public void loadRewardAdvert(@NonNull Activity activity) {
        super.loadRewardAdvert(activity);
        if (!isInitSDK()) {
            return;
        }
        if (rewardAd == null) {
            rewardAd = AppLovinIncentivizedInterstitial.create(currentActivity);
        }
        if (rewardAd != null) {
            String message = "method: loadRewardAdvert, loading reward ad...";
            Log.d(TAG, message);
            rewardAd.preload(rewardLoadListener);
        }
    }

    @Override
    public void showRewardAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showRewardAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Reward, callback)) {
            String message = "method: showRewardAdvert, show reward ad...";
            Log.d(TAG, message);
            rewardAd.show(activity, rewardedListener, rewardPlayListener, rewardDisplayListener, rewardClickListener);
        }
    }

    @Override
    public void dismissRewardAdvert() {
        super.dismissRewardAdvert();
        if (rewardAd != null) {
            rewardAd.dismiss();
        }
        rewardAd = null;
    }

    @Override
    public boolean isInterstitialAdvertLoaded() {
        super.isInterstitialAdvertLoaded();
        return interstitialAd != null;
    }

    @Override
    public void loadInterstitialAdvert(@NonNull Activity activity) {
        super.loadInterstitialAdvert(activity);
        if (!isInitSDK()) {
            return;
        }
        AppLovinSdk.getInstance(activity.getApplicationContext()).getAdService().loadNextAd(AppLovinAdSize.INTERSTITIAL, interstitialLoadListener);
    }

    @Override
    public void showInterstitialAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showInterstitialAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Interstitial, callback)) {
            String message = "method: showInterstitialAdvert, show interstitial ad...";
            Log.d(TAG, message);

            if (interstitialAd != null) {
                final AppLovinSdk sdk = AppLovinSdk.getInstance(activity);

                final AppLovinInterstitialAdDialog interstitialAdDialog = AppLovinInterstitialAd.create(sdk, activity);
                interstitialAdDialog.setAdDisplayListener(interstitialDisplayListener);
                interstitialAdDialog.setAdClickListener(interstitialClickListener);
                interstitialAdDialog.setAdVideoPlaybackListener(interstitialPlayListener);
                interstitialAdDialog.showAndRender(interstitialAd);
            }
        }
    }

    @Override
    public void dismissInterstitialAdvert() {
        super.dismissInterstitialAdvert();
//        if (interstitialAd != null) {
//            interstitialAd.dismiss();
//        }
//        interstitialAd = null;
    }

    @Override
    public boolean isBannerAdvertLoaded() {
        super.isBannerAdvertLoaded();
        return bannerAd != null && bannerState == AdvertState.LOADED;
    }

    @Override
    public void loadBannerAdvert(@NonNull Activity activity) {
        super.loadBannerAdvert(activity);
        if (!isInitSDK()) {
            return;
        }

        if (bannerAd == null) {
            bannerAd = new AppLovinAdView(AppLovinAdSize.BANNER, activity);
            bannerAd.setAdLoadListener(bannerLoadListener);
            bannerAd.setAdDisplayListener(bannerDisplayListener);
            bannerAd.setAdViewEventListener(bannerPlayListener);
            bannerAd.setAdClickListener(bannerClickListener);
        }
        if (bannerAd != null && bannerState != AdvertState.LOADING) {
            String message = "method: loadBannerAdvert, load banner ad...";
            Log.d(TAG, message);
            bannerAd.loadNextAd();
            bannerState = AdvertState.LOADING;
        }
    }

    @Override
    public void showBannerAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showBannerAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Banner, callback)) {
            String message = "method: showBannerAdvert, show banner ad...";
            Log.d(TAG, message);
            Yodo1MasBanner.showBanner(activity, bannerAd, object);
        }
    }

    private final AppLovinAdLoadListener rewardLoadListener = new AppLovinAdLoadListener() {

        @Override
        public void adReceived(AppLovinAd appLovinAd) {
            String message = "method: adReceived, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Reward, Yodo1MasSensorHelper.AdResult.SUCCESS);
        }

        @Override
        public void failedToReceiveAd(int i) {
            String message = "method: failedToReceiveAd, error: " + i;
            Log.d(TAG, message);


            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Reward);

//            loadRewardAdvertDelayed();
            loadRewardAdvert();
            trackAdRequest(Yodo1Mas.AdType.Reward, Yodo1MasSensorHelper.AdResult.FAIL);
        }
    };

    private final AppLovinAdVideoPlaybackListener rewardPlayListener = new AppLovinAdVideoPlaybackListener() {
        @Override
        public void videoPlaybackBegan(AppLovinAd appLovinAd) {
            String message = "method: videoPlaybackBegan, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }

        @Override
        public void videoPlaybackEnded(AppLovinAd appLovinAd, double v, boolean b) {
            String message = "method: videoPlaybackEnded, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_REWARD_EARNED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
        }
    };

    private final AppLovinAdDisplayListener rewardDisplayListener = new AppLovinAdDisplayListener() {
        @Override
        public void adDisplayed(AppLovinAd appLovinAd) {
            String message = "method: adDisplayed, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
        }

        @Override
        public void adHidden(AppLovinAd appLovinAd) {
            String message = "method: adHidden, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
            loadRewardAdvert();
        }
    };

    private final AppLovinAdClickListener rewardClickListener = appLovinAd -> {
        String message = "method: adClicked, reward: " + appLovinAd.getAdIdNumber();
        Log.d(TAG, message);
    };

    private final AppLovinAdRewardListener rewardedListener = new AppLovinAdRewardListener() {
        @Override
        public void userRewardVerified(AppLovinAd appLovinAd, Map<String, String> map) {
            String message = "method: userRewardVerified, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }

        @Override
        public void userOverQuota(AppLovinAd appLovinAd, Map<String, String> map) {
            String message = "method: userOverQuota, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }

        @Override
        public void userRewardRejected(AppLovinAd appLovinAd, Map<String, String> map) {
            String message = "method: userRewardRejected, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }

        @Override
        public void validationRequestFailed(AppLovinAd appLovinAd, int i) {
            String message = "method: validationRequestFailed, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }

        @Override
        public void userDeclinedToViewAd(AppLovinAd appLovinAd) {
            String message = "method: userDeclinedToViewAd, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }
    };

    private final AppLovinAdLoadListener interstitialLoadListener = new AppLovinAdLoadListener() {
        @Override
        public void adReceived(AppLovinAd appLovinAd) {
            String message = "method: adReceived, interstitial: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            interstitialAd = appLovinAd;
            trackAdRequest(Yodo1Mas.AdType.Interstitial, Yodo1MasSensorHelper.AdResult.SUCCESS);
        }

        @Override
        public void failedToReceiveAd(int i) {
            String message = "method: failedToReceiveAd, error: " + i;
            Log.d(TAG, message);

            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Interstitial);

//            loadInterstitialAdvertDelayed();
            loadInterstitialAdvert();
            trackAdRequest(Yodo1Mas.AdType.Interstitial, Yodo1MasSensorHelper.AdResult.FAIL);
        }
    };

    private final AppLovinAdVideoPlaybackListener interstitialPlayListener = new AppLovinAdVideoPlaybackListener() {
        @Override
        public void videoPlaybackBegan(AppLovinAd appLovinAd) {
            String message = "method: videoPlaybackBegan, interstitial: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }

        @Override
        public void videoPlaybackEnded(AppLovinAd appLovinAd, double v, boolean b) {
            String message = "method: videoPlaybackEnded, interstitial: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }
    };

    private final AppLovinAdDisplayListener interstitialDisplayListener = new AppLovinAdDisplayListener() {
        @Override
        public void adDisplayed(AppLovinAd appLovinAd) {
            String message = "method: adDisplayed, interstitial: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
        }

        @Override
        public void adHidden(AppLovinAd appLovinAd) {
            String message = "method: adHidden, interstitial: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
            loadInterstitialAdvert();
        }
    };

    private final AppLovinAdClickListener interstitialClickListener = new AppLovinAdClickListener() {
        @Override
        public void adClicked(AppLovinAd appLovinAd) {
            String message = "method: adClicked, interstitial: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }
    };


    // Banner
    private final AppLovinAdLoadListener bannerLoadListener = new AppLovinAdLoadListener() {
        @Override
        public void adReceived(AppLovinAd appLovinAd) {
            String message = "method: adReceived, banner: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            bannerState = AdvertState.LOADED;
            trackAdRequest(Yodo1Mas.AdType.Banner, Yodo1MasSensorHelper.AdResult.SUCCESS);
            callback(Yodo1MasAdEvent.CODE_LOADED, Yodo1Mas.AdType.Banner, TAG + ":{" + message + "}");
        }

        @Override
        public void failedToReceiveAd(int i) {
            String message = "method: failedToReceiveAd, error: " + i;
            Log.d(TAG, message);

            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Banner);

            bannerState = AdvertState.NONE;
            loadBannerAdvert();
            //loadBannerAdvertDelayed();
            trackAdRequest(Yodo1Mas.AdType.Banner, Yodo1MasSensorHelper.AdResult.FAIL);
        }
    };

    private final AppLovinAdViewEventListener bannerPlayListener = new AppLovinAdViewEventListener() {
        @Override
        public void adOpenedFullscreen(AppLovinAd appLovinAd, AppLovinAdView appLovinAdView) {
            String message = "method: adOpenedFullscreen, banner: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }

        @Override
        public void adClosedFullscreen(AppLovinAd appLovinAd, AppLovinAdView appLovinAdView) {
            String message = "method: adClosedFullscreen, banner: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }

        @Override
        public void adLeftApplication(AppLovinAd appLovinAd, AppLovinAdView appLovinAdView) {
            String message = "method: adLeftApplication, banner: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }

        @Override
        public void adFailedToDisplay(AppLovinAd appLovinAd, AppLovinAdView appLovinAdView, AppLovinAdViewDisplayErrorCode appLovinAdViewDisplayErrorCode) {
            String message = "method: adFailedToDisplay, banner: " + appLovinAd.getAdIdNumber() + ", error:" + appLovinAdViewDisplayErrorCode;
            Log.d(TAG, message);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Banner);
        }
    };

    private final AppLovinAdDisplayListener bannerDisplayListener = new AppLovinAdDisplayListener() {
        @Override
        public void adDisplayed(AppLovinAd appLovinAd) {
            String message = "method: adDisplayed, banner: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Banner, TAG + ":{" + message + "}");
        }

        @Override
        public void adHidden(AppLovinAd appLovinAd) {
            String message = "method: adHidden, banner: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Banner, TAG + ":{" + message + "}");

            bannerState = AdvertState.NONE;
            loadBannerAdvert();
        }
    };

    private final AppLovinAdClickListener bannerClickListener = appLovinAd -> {
        String message = "method: adClicked, banner: " + appLovinAd.getAdIdNumber();
        Log.d(TAG, message);
    };
}
