package com.yodo1.mas.mediation.applovin;

import android.app.Activity;
import android.text.TextUtils;
import android.util.Log;

import com.applovin.mediation.MaxAd;
import com.applovin.mediation.MaxAdListener;
import com.applovin.mediation.MaxError;
import com.applovin.mediation.MaxReward;
import com.applovin.mediation.MaxRewardedAdListener;
import com.applovin.mediation.ads.MaxInterstitialAd;
import com.applovin.mediation.ads.MaxRewardedAd;
import com.applovin.sdk.AppLovinPrivacySettings;
import com.applovin.sdk.AppLovinSdk;
import com.yodo1.mas.Yodo1Mas;
import com.yodo1.mas.analytics.Yodo1MasSensorHelper;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;
import com.yodo1.mas.mediation.Yodo1MasBannerAdapterBase;
import com.yodo1.mas.utils.Yodo1MasUtils;

import org.json.JSONException;
import org.json.JSONObject;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

public class Yodo1MasAppLovinMaxAdapter extends Yodo1MasAdapterBase {

    private MaxRewardedAd rewardAd;
    private MaxInterstitialAd interstitialAd;

    @Override
    public String getAdvertCode() {
        return "APPLOVIN";
    }

    @Override
    public String getSDKVersion() {
        return AppLovinSdk.VERSION;
    }

    @Override
    public String getMediationVersion() {
        return BuildConfig.MAS_VERSION_NAME;
    }

    @Override
    public void initSDK(@NonNull Activity activity, @NonNull Config config, @Nullable InitCallback callback) {
        super.initSDK(activity, config, callback);
        if (!isInitSDK()) {
            init = true;
            if (isMax()) {
                AppLovinSdk.getInstance(currentActivity).setMediationProvider("max");
            }
            AppLovinSdk.initializeSdk(currentActivity, configuration -> {
                String message = "method: onSdkInitialized, configuration: " + configuration.getConsentDialogState();
                Log.d(TAG, message);

                updatePrivacy();
                loadRewardAdvert();
                loadInterstitialAdvert();
//                loadBannerAdvert();

                if (callback != null) {
                    callback.onAdapterInitSuccessful(getAdvertCode());
                }
                if (Yodo1MasUtils.getMetaData(activity, "com.yodo1.mas.EnableAppLovinMediationDebugger", false)) {
                    AppLovinSdk.getInstance(activity).showMediationDebugger();
                }
            });
        } else {
            if (callback != null) {
                callback.onAdapterInitSuccessful(getAdvertCode());
            }
        }
    }

    @Override
    public void updatePrivacy() {
        super.updatePrivacy();
        if (currentActivity != null) {
            AppLovinPrivacySettings.setHasUserConsent(Yodo1MasHelper.getInstance().isGDPRUserConsent(), currentActivity);
            AppLovinPrivacySettings.setIsAgeRestrictedUser(Yodo1MasHelper.getInstance().isCOPPAAgeRestricted(), currentActivity);
            AppLovinPrivacySettings.setDoNotSell(Yodo1MasHelper.getInstance().isCCPADoNotSell(), currentActivity);
        }
    }

    public boolean isMax() {
        return true;
    }

    @Override
    public boolean isRewardAdvertLoaded() {
        super.isRewardAdvertLoaded();
        return rewardAd != null && rewardAd.isReady();
    }

    @Override
    public void loadRewardAdvert(@NonNull Activity activity) {
        super.loadRewardAdvert(activity);
        if (!isInitSDK()) {
            return;
        }
        AdId adId = getRewardAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId)) {
            rewardAd = MaxRewardedAd.getInstance(adId.adId, activity);
            rewardAd.setListener(rewardListener);
        }
        if (rewardAd != null) {
            String message = "method: loadRewardAdvert, loading reward ad...";
            Log.d(TAG, message);
            rewardAd.loadAd();
        }
    }

    @Override
    public void showRewardAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showRewardAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Reward, callback)) {
            String message = "method: showRewardAdvert, show reward ad...";
            Log.d(TAG, message);

            String placement = null;
            if (object != null && object.has(Yodo1MasAdapterBase.KEY_ARGUMENT_PLACEMENT)) {
                try {
                    placement = object.getString(Yodo1MasAdapterBase.KEY_ARGUMENT_PLACEMENT);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
            if (isMax() && !TextUtils.isEmpty(placement)) {
                rewardAd.showAd(placement);
            } else {
                rewardAd.showAd();
            }
        }
    }

    @Override
    public void dismissRewardAdvert() {
        super.dismissRewardAdvert();
        if (rewardAd != null) {
            rewardAd.destroy();
        }
        rewardAd = null;
    }

    @Override
    public boolean isInterstitialAdvertLoaded() {
        super.isInterstitialAdvertLoaded();
        return interstitialAd != null && interstitialAd.isReady();
    }

    @Override
    public void loadInterstitialAdvert(@NonNull Activity activity) {
        super.loadInterstitialAdvert(activity);
        if (!isInitSDK()) {
            return;
        }

        AdId adId = getInterstitialAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId)) {
            interstitialAd = new MaxInterstitialAd(adId.adId, activity);
            interstitialAd.setListener(interstitialListener);
        }
        if (interstitialAd != null) {
            String message = "method: loadInterstitialAdvert, loading interstitial ad...";
            Log.d(TAG, message);
            interstitialAd.loadAd();
        }
    }

    @Override
    public void showInterstitialAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showInterstitialAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Interstitial, callback)) {
            String message = "method: showInterstitialAdvert, show interstitial ad...";
            Log.d(TAG, message);

            String placement = null;
            if (object != null && object.has(Yodo1MasAdapterBase.KEY_ARGUMENT_PLACEMENT)) {
                try {
                    placement = object.getString(Yodo1MasAdapterBase.KEY_ARGUMENT_PLACEMENT);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
            if (isMax() && !TextUtils.isEmpty(placement)) {
                interstitialAd.showAd(placement);
            } else {
                interstitialAd.showAd();
            }
        }
    }

    @Override
    public void dismissInterstitialAdvert() {
        super.dismissInterstitialAdvert();
        if (interstitialAd != null) {
            interstitialAd.destroy();
        }
        interstitialAd = null;
    }

    @Override
    public Yodo1MasBannerAdapterBase getYodo1MasBannerAdapter() {
        Yodo1MasAppLovinMaxBannerAdapter bannerAdapter = new Yodo1MasAppLovinMaxBannerAdapter();
        bannerAdapter.bannerAdIds.clear();
        bannerAdapter.bannerAdIds.addAll(bannerAdIds);
        bannerAdapter.relateAdapter = this;
        bannerAdapter.advertCode = getAdvertCode();
        bannerAdapter.sDKVersion = getSDKVersion();
        bannerAdapter.mediationVersion = getMediationVersion();
        return bannerAdapter;
    }

    private final MaxRewardedAdListener rewardListener = new MaxRewardedAdListener() {

        @Override
        public void onAdLoaded(MaxAd ad) {
            String message = "method: onAdLoaded, reward: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Reward, Yodo1MasSensorHelper.AdResult.SUCCESS);
        }

        @Override
        public void onAdLoadFailed(String adUnitId, MaxError maxError) {
            String message = "method: onAdLoadFailed, placementId: " + adUnitId + ", error: " + maxError.getAdLoadFailureInfo();
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Reward, Yodo1MasSensorHelper.AdResult.FAIL);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Reward);

            nextReward();
            loadRewardAdvertDelayed();
        }

        @Override
        public void onAdDisplayed(MaxAd ad) {
            String message = "method: onAdDisplayed, reward: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdDisplayFailed(MaxAd ad, MaxError maxError) {
            String message = "method: onAdDisplayFailed, reward:" + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId() + ", error: " + maxError.getAdLoadFailureInfo();
            Log.d(TAG, message);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Reward);

            nextReward();
            loadRewardAdvert();
        }

        @Override
        public void onAdHidden(MaxAd ad) {
            String message = "method: onAdHidden, reward: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
            loadRewardAdvert();
        }

        @Override
        public void onAdClicked(MaxAd ad) {
            String message = "method: onAdClicked, reward: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
        }

        @Override
        public void onRewardedVideoStarted(MaxAd ad) {
            String message = "method: onRewardedVideoStarted, reward: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
        }

        @Override
        public void onRewardedVideoCompleted(MaxAd ad) {
            String message = "method: onRewardedVideoCompleted, reward: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
        }

        @Override
        public void onUserRewarded(MaxAd ad, MaxReward reward) {
            String message = "method: onUserRewarded, reward: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_REWARD_EARNED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
        }
    };

    private final MaxAdListener interstitialListener = new MaxAdListener() {

        @Override
        public void onAdLoaded(MaxAd ad) {
            String message = "method: onAdLoaded, interstitial: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Interstitial, Yodo1MasSensorHelper.AdResult.SUCCESS);
        }

        @Override
        public void onAdLoadFailed(String adUnitId, MaxError maxError) {
            String message = "method: onAdLoadFailed, placementId: " + adUnitId + ", error:" + maxError.getAdLoadFailureInfo();
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Interstitial, Yodo1MasSensorHelper.AdResult.FAIL);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Interstitial);

            nextInterstitial();
            loadInterstitialAdvertDelayed();
        }

        @Override
        public void onAdDisplayed(MaxAd ad) {
            String message = "method: onAdDisplayed, interstitial: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdDisplayFailed(MaxAd ad, MaxError maxError) {
            String message = "method: onAdDisplayFailed, reward:" + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId() + ", error: " + maxError.getAdLoadFailureInfo();
            Log.d(TAG, message);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Interstitial);

            nextInterstitial();
            loadInterstitialAdvert();
        }

        @Override
        public void onAdHidden(MaxAd ad) {
            String message = "method: onAdHidden, interstitial: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
            loadInterstitialAdvert();
        }

        @Override
        public void onAdClicked(MaxAd ad) {
            String message = "method: onAdClicked, interstitial: " + ad.getNetworkName() + ", placementId: " + ad.getAdUnitId();
            Log.d(TAG, message);
        }
    };

}
