package com.yodo1.mas.mediation.applovin;

import android.app.Activity;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.applovin.adview.AppLovinIncentivizedInterstitial;
import com.applovin.adview.AppLovinInterstitialAd;
import com.applovin.adview.AppLovinInterstitialAdDialog;
import com.applovin.sdk.AppLovinAd;
import com.applovin.sdk.AppLovinAdClickListener;
import com.applovin.sdk.AppLovinAdDisplayListener;
import com.applovin.sdk.AppLovinAdLoadListener;
import com.applovin.sdk.AppLovinAdRewardListener;
import com.applovin.sdk.AppLovinAdSize;
import com.applovin.sdk.AppLovinAdVideoPlaybackListener;
import com.applovin.sdk.AppLovinPrivacySettings;
import com.applovin.sdk.AppLovinSdk;
import com.yodo1.mas.Yodo1Mas;
import com.yodo1.mas.analytics.Yodo1MasSensorHelper;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;
import com.yodo1.mas.mediation.Yodo1MasBannerAdapterBase;

import org.json.JSONObject;

import java.util.Map;

public class Yodo1MasAppLovinAdapter extends Yodo1MasAdapterBase {

    private AppLovinIncentivizedInterstitial rewardAd;
    private AppLovinAd interstitialAd;

    @Override
    public String getAdvertCode() {
        return "applovin";
    }

    @Override
    public String getSDKVersion() {
        return AppLovinSdk.VERSION;
    }

    @Override
    public String getMediationVersion() {
        return BuildConfig.MAS_VERSION_NAME;
    }

    @Override
    public void initSDK(@NonNull Activity activity, @NonNull Config config, @Nullable InitCallback callback) {
        super.initSDK(activity, config, callback);
        if (!isInitSDK()) {
            init = true;
            AppLovinSdk.initializeSdk(currentActivity, configuration -> {
                String message = "method: onSdkInitialized, configuration: " + configuration.getConsentDialogState();
                Log.d(TAG, message);

                updatePrivacy();
                loadRewardAdvert();
                loadInterstitialAdvert();

                if (callback != null) {
                    callback.onAdapterInitSuccessful(getAdvertCode());
                }
            });
        } else {
            if (callback != null) {
                callback.onAdapterInitSuccessful(getAdvertCode());
            }
        }
    }

    @Override
    public void updatePrivacy() {
        super.updatePrivacy();
        setAppLoinPrivacy(currentActivity);
    }

    public static void setAppLoinPrivacy(Activity activity) {
        if (activity == null) {
            return;
        }
        AppLovinPrivacySettings.setHasUserConsent(Yodo1MasHelper.getInstance().isGDPRUserConsent(), activity);
        AppLovinPrivacySettings.setIsAgeRestrictedUser(Yodo1MasHelper.getInstance().isCOPPAAgeRestricted(), activity);
        AppLovinPrivacySettings.setDoNotSell(Yodo1MasHelper.getInstance().isCCPADoNotSell(), activity);
    }

    @Override
    public boolean isRewardAdvertLoaded() {
        super.isRewardAdvertLoaded();
        return rewardAd != null && rewardAd.isAdReadyToDisplay();
    }

    @Override
    public void loadRewardAdvert(@NonNull Activity activity) {
        super.loadRewardAdvert(activity);
        if (!isInitSDK()) {
            return;
        }
        if (rewardAd == null) {
            rewardAd = AppLovinIncentivizedInterstitial.create(currentActivity);
        }
        if (rewardAd != null) {
            String message = "method: loadRewardAdvert, loading reward ad...";
            Log.d(TAG, message);
            rewardAd.preload(rewardLoadListener);
        }
    }

    @Override
    public void showRewardAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showRewardAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Reward, callback)) {
            String message = "method: showRewardAdvert, show reward ad...";
            Log.d(TAG, message);
            rewardAd.show(activity, rewardedListener, rewardPlayListener, rewardDisplayListener, rewardClickListener);
        }
    }

    @Override
    public void dismissRewardAdvert() {
        super.dismissRewardAdvert();
        rewardAd = null;
    }

    @Override
    public boolean isInterstitialAdvertLoaded() {
        super.isInterstitialAdvertLoaded();
        return interstitialAd != null;
    }

    @Override
    public void loadInterstitialAdvert(@NonNull Activity activity) {
        super.loadInterstitialAdvert(activity);
        if (!isInitSDK()) {
            return;
        }
        AppLovinSdk.getInstance(activity.getApplicationContext()).getAdService().loadNextAd(AppLovinAdSize.INTERSTITIAL, interstitialLoadListener);
    }

    @Override
    public void showInterstitialAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showInterstitialAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Interstitial, callback)) {
            String message = "method: showInterstitialAdvert, show interstitial ad...";
            Log.d(TAG, message);

            if (interstitialAd != null) {
                final AppLovinSdk sdk = AppLovinSdk.getInstance(activity);

                final AppLovinInterstitialAdDialog interstitialAdDialog = AppLovinInterstitialAd.create(sdk, activity);
                interstitialAdDialog.setAdDisplayListener(interstitialDisplayListener);
                interstitialAdDialog.setAdClickListener(interstitialClickListener);
                interstitialAdDialog.setAdVideoPlaybackListener(interstitialPlayListener);
                interstitialAdDialog.showAndRender(interstitialAd);
            }
        }
    }

    @Override
    public void dismissInterstitialAdvert() {
        super.dismissInterstitialAdvert();
//        if (interstitialAd != null) {
//            interstitialAd.dismiss();
//        }
//        interstitialAd = null;
    }

    @Override
    public Yodo1MasBannerAdapterBase getYodo1MasBannerAdapter() {
        Yodo1MasAppLovinBannerAdapter bannerAdapter = new Yodo1MasAppLovinBannerAdapter();
        bannerAdapter.bannerAdIds.clear();
        bannerAdapter.bannerAdIds.addAll(bannerAdIds);
        bannerAdapter.relateAdapter = this;
        bannerAdapter.advertCode = getAdvertCode();
        bannerAdapter.sdkVersion = getSDKVersion();
        bannerAdapter.mediationVersion = getMediationVersion();
        return bannerAdapter;
    }

    private final AppLovinAdLoadListener rewardLoadListener = new AppLovinAdLoadListener() {

        @Override
        public void adReceived(AppLovinAd appLovinAd) {
            String message = "method: adReceived, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            trackAdRequestSuccessed(Yodo1Mas.AdType.Reward);
        }

        @Override
        public void failedToReceiveAd(int i) {
            String message = "method: failedToReceiveAd, error: " + i;
            Log.d(TAG, message);


            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Reward);

//            loadRewardAdvertDelayed();
            loadRewardAdvert();
            trackAdRequestFailed(Yodo1Mas.AdType.Reward, i, null);
        }
    };

    private final AppLovinAdVideoPlaybackListener rewardPlayListener = new AppLovinAdVideoPlaybackListener() {
        @Override
        public void videoPlaybackBegan(AppLovinAd appLovinAd) {
            String message = "method: videoPlaybackBegan, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }

        @Override
        public void videoPlaybackEnded(AppLovinAd appLovinAd, double v, boolean b) {
            String message = "method: videoPlaybackEnded, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_REWARD_EARNED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
        }
    };

    private final AppLovinAdDisplayListener rewardDisplayListener = new AppLovinAdDisplayListener() {
        @Override
        public void adDisplayed(AppLovinAd appLovinAd) {
            String message = "method: adDisplayed, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
        }

        @Override
        public void adHidden(AppLovinAd appLovinAd) {
            String message = "method: adHidden, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
            loadRewardAdvert();
        }
    };

    private final AppLovinAdClickListener rewardClickListener = appLovinAd -> {
        String message = "method: adClicked, reward: " + appLovinAd.getAdIdNumber();
        Log.d(TAG, message);
    };

    private final AppLovinAdRewardListener rewardedListener = new AppLovinAdRewardListener() {
        @Override
        public void userRewardVerified(AppLovinAd appLovinAd, Map<String, String> map) {
            String message = "method: userRewardVerified, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }

        @Override
        public void userOverQuota(AppLovinAd appLovinAd, Map<String, String> map) {
            String message = "method: userOverQuota, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }

        @Override
        public void userRewardRejected(AppLovinAd appLovinAd, Map<String, String> map) {
            String message = "method: userRewardRejected, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }

        @Override
        public void validationRequestFailed(AppLovinAd appLovinAd, int i) {
            String message = "method: validationRequestFailed, reward: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }
    };

    private final AppLovinAdLoadListener interstitialLoadListener = new AppLovinAdLoadListener() {
        @Override
        public void adReceived(AppLovinAd appLovinAd) {
            String message = "method: adReceived, interstitial: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            interstitialAd = appLovinAd;
            trackAdRequestSuccessed(Yodo1Mas.AdType.Interstitial);
        }

        @Override
        public void failedToReceiveAd(int i) {
            String message = "method: failedToReceiveAd, error: " + i;
            Log.d(TAG, message);

            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Interstitial);

//            loadInterstitialAdvertDelayed();
            loadInterstitialAdvert();
            trackAdRequestFailed(Yodo1Mas.AdType.Interstitial, i, null);
        }
    };

    private final AppLovinAdVideoPlaybackListener interstitialPlayListener = new AppLovinAdVideoPlaybackListener() {
        @Override
        public void videoPlaybackBegan(AppLovinAd appLovinAd) {
            String message = "method: videoPlaybackBegan, interstitial: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }

        @Override
        public void videoPlaybackEnded(AppLovinAd appLovinAd, double v, boolean b) {
            String message = "method: videoPlaybackEnded, interstitial: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }
    };

    private final AppLovinAdDisplayListener interstitialDisplayListener = new AppLovinAdDisplayListener() {
        @Override
        public void adDisplayed(AppLovinAd appLovinAd) {
            String message = "method: adDisplayed, interstitial: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
        }

        @Override
        public void adHidden(AppLovinAd appLovinAd) {
            String message = "method: adHidden, interstitial: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
            loadInterstitialAdvert();
        }
    };

    private final AppLovinAdClickListener interstitialClickListener = new AppLovinAdClickListener() {
        @Override
        public void adClicked(AppLovinAd appLovinAd) {
            String message = "method: adClicked, interstitial: " + appLovinAd.getAdIdNumber();
            Log.d(TAG, message);
        }
    };

}
