package com.yodo1.mas.mediation.facebook;

import android.app.Activity;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.facebook.ads.Ad;
import com.facebook.ads.AdError;
import com.facebook.ads.AdListener;
import com.facebook.ads.AdSettings;
import com.facebook.ads.AdSize;
import com.facebook.ads.AdView;
import com.facebook.ads.AudienceNetworkAds;
import com.facebook.ads.InterstitialAd;
import com.facebook.ads.InterstitialAdListener;
import com.facebook.ads.RewardedVideoAd;
import com.facebook.ads.RewardedVideoAdListener;
import com.yodo1.mas.Yodo1Mas;
import com.yodo1.mas.analytics.Yodo1MasSensorHelper;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.helper.Yodo1MasBanner;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;

import org.json.JSONObject;

public class Yodo1MasFacebookAdapter extends Yodo1MasAdapterBase {

    private RewardedVideoAd rewardAd;
    private InterstitialAd interstitialAd;
    private AdView bannerAd;

    @Override
    public String getAdvertCode() {
        return "facebook";
    }

    @Override
    public String getSDKVersion() {
        return BuildConfig.SDK_VERSION_NAME;
    }

    @Override
    public String getMediationVersion() {
        return BuildConfig.MAS_VERSION_NAME;
    }

    @Override
    public void initSDK(@NonNull Activity activity, @NonNull Config config, @Nullable InitCallback callback) {
        super.initSDK(activity, config, callback);
        if (!isInitSDK()) {
            init = true;
            if (Yodo1MasHelper.getInstance().isDebug()) {
                AdSettings.turnOnSDKDebugger(applicationContext);
            }
            if (!AudienceNetworkAds.isInitialized(applicationContext)) {
                AudienceNetworkAds.buildInitSettings(applicationContext).withInitListener(result -> {
                    String message = "method: onInitialized, result: " + result;
                    Log.d(TAG, message);

                    if (result.isSuccess()) {
                        updatePrivacy();
                        loadRewardAdvert();
                        loadInterstitialAdvert();
                        loadBannerAdvert();

                        if (callback != null) {
                            callback.onAdapterInitSuccessful(getAdvertCode());
                        }
                    } else {
                        if (callback != null) {
                            callback.onAdapterInitFailed(getAdvertCode(), new Yodo1MasError(Yodo1MasError.CODE_ADVERT_UNINITIALIZED, result.getMessage()));
                        }
                    }
                }).initialize();
            }
        } else {
            if (callback != null) {
                callback.onAdapterInitSuccessful(getAdvertCode());
            }
        }
    }

    @Override
    public boolean isInitSDK() {
        return super.isInitSDK() && AudienceNetworkAds.isInitialized(applicationContext);
    }

    @Override
    public void updatePrivacy() {
        super.updatePrivacy();
        setFacebookPrivacy();
    }

    public static void setFacebookPrivacy() {
        AdSettings.setMixedAudience(Yodo1MasHelper.getInstance().isCOPPAAgeRestricted());
    }

    @Override
    public boolean isRewardAdvertLoaded() {
        super.isRewardAdvertLoaded();
        return rewardAd != null && rewardAd.isAdLoaded();
    }

    @Override
    public void loadRewardAdvert(@NonNull Activity activity) {
        super.loadRewardAdvert(activity);
        if (!isInitSDK()) {
            return;
        }

        AdId adId = getRewardAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId) && (rewardAd == null || !rewardAd.getPlacementId().equals(adId.adId))) {
            rewardAd = new RewardedVideoAd(activity, adId.adId);
        }
        if (rewardAd != null) {
            String message = "method: loadRewardAdvert, loading reward ad...";
            Log.d(TAG, message);
            rewardAd.loadAd(rewardAd.buildLoadAdConfig().withAdListener(rewardListener).build());
        }
    }

    @Override
    public void showRewardAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showRewardAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Reward, callback)) {
            String message = "method: showRewardAdvert, show reward ad...";
            Log.d(TAG, message);
            rewardAd.show();
        }
    }

    @Override
    public void dismissRewardAdvert() {
        super.dismissRewardAdvert();
        if (rewardAd != null) {
            rewardAd.destroy();
        }
        rewardAd = null;
    }

    @Override
    public boolean isInterstitialAdvertLoaded() {
        super.isInterstitialAdvertLoaded();
        return interstitialAd != null && interstitialAd.isAdLoaded();
    }

    @Override
    public void loadInterstitialAdvert(@NonNull Activity activity) {
        super.loadInterstitialAdvert(activity);
        if (!isInitSDK()) {
            return;
        }

        AdId adId = getInterstitialAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId) && (interstitialAd == null || !interstitialAd.getPlacementId().equals(adId.adId))) {
            interstitialAd = new InterstitialAd(activity, adId.adId);
        }
        if (interstitialAd != null) {
            String message = "method: loadInterstitialAdvert, loading interstitial ad...";
            Log.d(TAG, message);
            interstitialAd.loadAd(interstitialAd.buildLoadAdConfig().withAdListener(interstitialListener).build());
        }
    }

    @Override
    public void showInterstitialAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showInterstitialAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Interstitial, callback)) {
            String message = "method: showInterstitialAdvert, show interstitial ad...";
            Log.d(TAG, message);
            interstitialAd.show();
        }
    }

    @Override
    public void dismissInterstitialAdvert() {
        super.dismissInterstitialAdvert();
        if (interstitialAd != null) {
            interstitialAd.destroy();
        }
        interstitialAd = null;
    }

    @Override
    public boolean isBannerAdvertLoaded() {
        super.isBannerAdvertLoaded();
        return bannerAd != null && !bannerAd.isAdInvalidated() && getBannerAdId() != null && bannerState == AdvertState.LOADED;
    }

    @Override
    public void loadBannerAdvert(@NonNull Activity activity) {
        super.loadBannerAdvert(activity);
        if (!isInitSDK()) {
            return;
        }

        AdId adId = getBannerAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId) && (bannerAd == null || !bannerAd.getPlacementId().equals(adId.adId))) {
            bannerAd = new AdView(activity, adId.adId, AdSize.BANNER_320_50);
        }
        if (bannerAd != null && bannerState != AdvertState.LOADING) {
            String message = "method: loadBannerAdvert, loading banner ad...";
            Log.d(TAG, message);
            bannerAd.loadAd(bannerAd.buildLoadAdConfig().withAdListener(bannerListener).build());
            bannerState = AdvertState.LOADING;
        }
    }

    @Override
    public void showBannerAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showBannerAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Banner, callback)) {
            String message = "method: showBannerAdvert, show banner ad...";
            Log.d(TAG, message);
            Yodo1MasBanner.showBanner(activity, bannerAd, object);
        }
    }

    @Override
    public void dismissBannerAdvert(boolean destroy) {
        super.dismissBannerAdvert(destroy);
        ;
        if (bannerAd != null) {
            Yodo1MasBanner.removeBanner(bannerAd);
            if (destroy) {
                bannerAd.destroy();
                bannerAd = null;
                bannerState = AdvertState.NONE;
                loadBannerAdvert();
            }
        }
    }

    private final RewardedVideoAdListener rewardListener = new RewardedVideoAdListener() {
        @Override
        public void onAdLoaded(Ad ad) {
            String message = "method: onAdLoaded, reward: " + ad.getPlacementId();
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Reward, Yodo1MasSensorHelper.AdResult.SUCCESS);
        }

        @Override
        public void onError(Ad ad, AdError adError) {
            String message = "method: onError, reward: " + ad.getPlacementId() + ", error: " + adError.getErrorMessage();
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Reward, Yodo1MasSensorHelper.AdResult.FAIL);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Reward);

            nextReward();
            loadRewardAdvertDelayed();
        }

        @Override
        public void onAdClicked(Ad ad) {
            String message = "method: onAdClicked, reward: " + ad.getPlacementId();
            Log.d(TAG, message);
        }

        @Override
        public void onRewardedVideoCompleted() {
            String message = "method: onRewardedVideoCompleted";
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_REWARD_EARNED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
            loadRewardAdvert();
        }

        @Override
        public void onLoggingImpression(Ad ad) {
            String message = "method: onLoggingImpression, reward: " + ad.getPlacementId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
        }

        @Override
        public void onRewardedVideoClosed() {
            String message = "method: onRewardedVideoClosed";
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
            loadRewardAdvert();
        }
    };

    private final InterstitialAdListener interstitialListener = new InterstitialAdListener() {
        @Override
        public void onInterstitialDisplayed(Ad ad) {
            String message = "method: onInterstitialDisplayed, interstitial: " + ad.getPlacementId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
        }

        @Override
        public void onInterstitialDismissed(Ad ad) {
            String message = "method: onInterstitialDismissed, interstitial: " + ad.getPlacementId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
            loadInterstitialAdvert();
        }

        @Override
        public void onError(Ad ad, AdError adError) {
            String message = "method: onError, interstitial: " + ad.getPlacementId() + ", error: " + adError.getErrorMessage();
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Interstitial, Yodo1MasSensorHelper.AdResult.FAIL);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Interstitial);

            nextInterstitial();
            loadInterstitialAdvertDelayed();
        }

        @Override
        public void onAdLoaded(Ad ad) {
            String message = "method: onAdLoaded, interstitial: " + ad.getPlacementId();
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Interstitial, Yodo1MasSensorHelper.AdResult.SUCCESS);
        }

        @Override
        public void onAdClicked(Ad ad) {
            String message = "method: onAdClicked, interstitial: " + ad.getPlacementId();
            Log.d(TAG, message);
        }

        @Override
        public void onLoggingImpression(Ad ad) {
            String message = "method: onLoggingImpression, interstitial: " + ad.getPlacementId();
            Log.d(TAG, message);
        }
    };

    private final AdListener bannerListener = new AdListener() {
        @Override
        public void onError(Ad ad, AdError adError) {
            String message = "method: onError, banner: " + ad.getPlacementId() + ", error: " + adError.getErrorMessage();
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Banner, Yodo1MasSensorHelper.AdResult.FAIL);
            Yodo1MasError error = new Yodo1MasError(adError == AdError.SHOW_CALLED_BEFORE_LOAD_ERROR ? Yodo1MasError.CODE_ADVERT_SHOW_FAIL : Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Banner);

            nextBanner();
            bannerState = AdvertState.NONE;
            loadBannerAdvertDelayed();
        }

        @Override
        public void onAdLoaded(Ad ad) {
            String message = "method: onAdLoaded, banner: " + ad.getPlacementId();
            Log.d(TAG, message);
            bannerState = AdvertState.LOADED;
            trackAdRequest(Yodo1Mas.AdType.Banner, Yodo1MasSensorHelper.AdResult.SUCCESS);
            callback(Yodo1MasAdEvent.CODE_LOADED, Yodo1Mas.AdType.Banner, TAG + ":{" + message + "}");
        }

        @Override
        public void onAdClicked(Ad ad) {
            String message = "method: onAdClicked, banner: " + ad.getPlacementId();
            Log.d(TAG, message);
        }

        @Override
        public void onLoggingImpression(Ad ad) {
            String message = "method: onLoggingImpression, banner: " + ad.getPlacementId();
            Log.d(TAG, message);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Banner, TAG + ":{" + message + "}");
        }
    };
}
