package com.yodo1.mas.mediation.fyber;

import android.app.Activity;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.fyber.inneractive.sdk.external.InneractiveAdManager;
import com.fyber.inneractive.sdk.external.InneractiveAdRequest;
import com.fyber.inneractive.sdk.external.InneractiveAdSpot;
import com.fyber.inneractive.sdk.external.InneractiveAdSpotManager;
import com.fyber.inneractive.sdk.external.InneractiveErrorCode;
import com.fyber.inneractive.sdk.external.InneractiveFullscreenAdEventsListener;
import com.fyber.inneractive.sdk.external.InneractiveFullscreenUnitController;
import com.fyber.inneractive.sdk.external.InneractiveUnitController;
import com.fyber.inneractive.sdk.external.OnFyberMarketplaceInitializedListener;
import com.yodo1.mas.Yodo1Mas;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;
import com.yodo1.mas.mediation.Yodo1MasBannerAdapterBase;

import org.json.JSONObject;

public class Yodo1MasFyberAdapter extends Yodo1MasAdapterBase {

    private InneractiveAdSpot rewardAd;
    private InneractiveAdSpot interstitialAd;

    private AdvertState rewardState = AdvertState.NONE;
    private AdvertState interstitialState = AdvertState.NONE;

    @Override
    public String getAdvertCode() {
        return "fyber";
    }

    @Override
    public String getSDKVersion() {
        return BuildConfig.SDK_VERSION_NAME;
    }

    @Override
    public String getMediationVersion() {
        return BuildConfig.MAS_VERSION_NAME;
    }

    @Override
    public void initSDK(@NonNull Activity activity, @NonNull Config config, @Nullable InitCallback callback) {
        super.initSDK(activity, config, callback);
        if (isInitSDK()) {
            if (callback != null) {
                callback.onAdapterInitSuccessful(getAdvertCode());
            }
            return;
        }

        if (TextUtils.isEmpty(config.appId)) {
            if (callback != null) {
                callback.onAdapterInitFailed(getAdvertCode(), new Yodo1MasError(Yodo1MasError.CODE_ADVERT_UNINITIALIZED, "config.appId is null"));
            }
            return;
        }

        init = true;
        InneractiveAdManager.initialize(activity, config.appId);
        InneractiveAdManager.initialize(activity, config.appId, status -> {
            String message = "method: onFyberMarketplaceInitialized, status: " + status.name();
            Log.d(TAG, message);
            if (OnFyberMarketplaceInitializedListener.FyberInitStatus.SUCCESSFULLY == status) {
                updatePrivacy();
                loadRewardAdvert();
                loadInterstitialAdvert();
                if (callback != null) {
                    callback.onAdapterInitSuccessful(getAdvertCode());
                }
            }
        });
    }

    @Override
    public void updatePrivacy() {
        super.updatePrivacy();
        setFyberPrivacy();
    }

    public static void setFyberPrivacy() {
        InneractiveAdManager.setGdprConsent(Yodo1MasHelper.getInstance().isGDPRUserConsent());
        String ccpaContent = Yodo1MasHelper.getInstance().isCCPADoNotSell() ? "1YYN" : "1YNN";
        InneractiveAdManager.setUSPrivacyString(ccpaContent);
    }

    @Override
    public boolean isRewardAdvertLoaded() {
        return rewardAd != null && rewardAd.isReady();
    }

    @Override
    public void loadRewardAdvert(@NonNull Activity activity) {
        super.loadRewardAdvert(activity);
        if (!isInitSDK()) {
            return;
        }

        AdId adId = getRewardAdId();
        if (adId == null || TextUtils.isEmpty(adId.adId)) {
            return;
        }

        if (rewardAd == null) {
            rewardAd = InneractiveAdSpotManager.get().createSpot();
            InneractiveFullscreenUnitController controller = new InneractiveFullscreenUnitController();
            rewardAd.addUnitController(controller);
            rewardAd.setRequestListener(rewardListener);
        }
        if (rewardAd != null && rewardState != AdvertState.LOADING) {
            String message = "method: loadRewardAdvert, loading reward ad...";
            Log.d(TAG, message);

            InneractiveAdRequest adRequest = new InneractiveAdRequest("");
            rewardAd.requestAd(adRequest);
            rewardState = AdvertState.LOADING;
        }
    }

    @Override
    public void showRewardAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showRewardAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Reward, callback)) {
            String message = "method: showRewardAdvert, show reward ad...";
            Log.d(TAG, message);
            InneractiveFullscreenUnitController controller = (InneractiveFullscreenUnitController) rewardAd.getSelectedUnitController();
            controller.show(activity);
            rewardState = AdvertState.NONE;
        }
    }

    @Override
    public void dismissRewardAdvert() {
        super.dismissRewardAdvert();
        if (rewardAd != null) {
            rewardAd.destroy();
        }
        rewardAd = null;
        rewardState = AdvertState.NONE;
    }

    @Override
    public boolean isInterstitialAdvertLoaded() {
        return interstitialAd != null && interstitialAd.isReady();
    }

    @Override
    public void loadInterstitialAdvert(@NonNull Activity activity) {
        super.loadInterstitialAdvert(activity);
        if (!isInitSDK()) {
            return;
        }

        AdId adId = getInterstitialAdId();
        if (adId == null || TextUtils.isEmpty(adId.adId)) {
            return;
        }

        if (interstitialAd == null) {
            interstitialAd = InneractiveAdSpotManager.get().createSpot();
            InneractiveFullscreenUnitController controller = new InneractiveFullscreenUnitController();
            interstitialAd.addUnitController(controller);
            interstitialAd.setRequestListener(interstitialListener);
        }
        if (interstitialAd != null && interstitialState != AdvertState.LOADING) {
            String message = "method: loadInterstitialAdvert, loading interstitial ad...";
            Log.d(TAG, message);

            InneractiveAdRequest adRequest = new InneractiveAdRequest("");
            interstitialAd.requestAd(adRequest);
            interstitialState = AdvertState.LOADING;
        }
    }

    @Override
    public void showInterstitialAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showInterstitialAdvert(activity, object, callback);

        if (isCanShow(Yodo1Mas.AdType.Interstitial, callback)) {
            String message = "method: showInterstitialAdvert, show interstitial ad...";
            Log.d(TAG, message);

            InneractiveFullscreenUnitController controller = (InneractiveFullscreenUnitController) interstitialAd.getSelectedUnitController();
            controller.show(activity);
            interstitialState = AdvertState.NONE;
        }
    }

    @Override
    public void dismissInterstitialAdvert() {
        super.dismissInterstitialAdvert();
        if (interstitialAd != null) {
            interstitialAd.destroy();
        }
        interstitialAd = null;
        interstitialState = AdvertState.NONE;
    }

    @Override
    public Yodo1MasBannerAdapterBase getYodo1MasBannerAdapter() {
        Yodo1MasFyberBannerAdapter bannerAdapter = new Yodo1MasFyberBannerAdapter();
        bannerAdapter.bannerAdIds.clear();
        bannerAdapter.bannerAdIds.addAll(bannerAdIds);
        bannerAdapter.relateAdapter = this;
        bannerAdapter.advertCode = getAdvertCode();
        bannerAdapter.sdkVersion = getSDKVersion();
        bannerAdapter.mediationVersion = getMediationVersion();
        return bannerAdapter;
    }

    @Override
    public void onActivityDestroy(@NonNull Activity activity) {
        super.onActivityDestroy(activity);
        InneractiveAdManager.destroy();
    }

    private final InneractiveAdSpot.RequestListener rewardListener = new InneractiveAdSpot.RequestListener() {

        @Override
        public void onInneractiveSuccessfulAdRequest(InneractiveAdSpot inneractiveAdSpot) {

            String message = "method: onInneractiveSuccessfulAdRequest, reward: ";
            Log.d(TAG, message);
            rewardState = AdvertState.LOADED;

            InneractiveFullscreenUnitController controller =
                    (InneractiveFullscreenUnitController) rewardAd.getSelectedUnitController();
            controller.setRewardedListener(inneractiveAdSpot1 -> {
                String message1 = "method: onAdRewarded, reward: ";
                Log.d(TAG, message1);
                callback(Yodo1MasAdEvent.CODE_REWARD_EARNED, Yodo1Mas.AdType.Reward, TAG + ":{" + message1 + "}");
            });

            controller.setEventsListener(new InneractiveFullscreenAdEventsListener() {

                @Override
                public void onAdImpression(InneractiveAdSpot inneractiveAdSpot) {
                    String message = "method: onAdImpression, reward: ";
                    Log.d(TAG, message);
                    callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
                }

                @Override
                public void onAdClicked(InneractiveAdSpot inneractiveAdSpot) {
                    String message = "method: onAdClicked, reward: ";
                    Log.d(TAG, message);
                }

                @Override
                public void onAdWillOpenExternalApp(InneractiveAdSpot inneractiveAdSpot) {
                    String message = "method: onAdWillOpenExternalApp, reward: ";
                    Log.d(TAG, message);
                }

                @Override
                public void onAdEnteredErrorState(InneractiveAdSpot inneractiveAdSpot, InneractiveUnitController.AdDisplayError adDisplayError) {
                    String message = "method: onAdEnteredErrorState, reward: ";
                    Log.d(TAG, message);
                }

                @Override
                public void onAdWillCloseInternalBrowser(InneractiveAdSpot inneractiveAdSpot) {
                    String message = "method: onAdWillCloseInternalBrowser, reward: ";
                    Log.d(TAG, message);
                }

                @Override
                public void onAdDismissed(InneractiveAdSpot inneractiveAdSpot) {
                    String message = "method: onAdDismissed, reward: ";
                    Log.d(TAG, message);
                    callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
                    loadRewardAdvert();
                }
            });

        }

        @Override
        public void onInneractiveFailedAdRequest(InneractiveAdSpot inneractiveAdSpot, InneractiveErrorCode inneractiveErrorCode) {
            String message = "method: onInneractiveFailedAdRequest, reward errorCode:" + inneractiveErrorCode.name();
            Log.d(TAG, message);
            rewardState = AdvertState.NONE;

            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Reward);

            nextReward();
            loadRewardAdvertDelayed();
        }
    };

    private final InneractiveAdSpot.RequestListener interstitialListener = new InneractiveAdSpot.RequestListener() {

        @Override
        public void onInneractiveSuccessfulAdRequest(InneractiveAdSpot inneractiveAdSpot) {

            String message = "method: onInneractiveSuccessfulAdRequest, interstitial: ";
            Log.d(TAG, message);
            interstitialState = AdvertState.LOADED;

            InneractiveFullscreenUnitController controller =
                    (InneractiveFullscreenUnitController) interstitialAd.getSelectedUnitController();
            controller.setEventsListener(new InneractiveFullscreenAdEventsListener() {

                @Override
                public void onAdImpression(InneractiveAdSpot inneractiveAdSpot) {
                    String message = "method: onAdImpression, interstitial: ";
                    Log.d(TAG, message);
                    callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
                }

                @Override
                public void onAdClicked(InneractiveAdSpot inneractiveAdSpot) {
                    String message = "method: onAdClicked, interstitial: ";
                    Log.d(TAG, message);
                }

                @Override
                public void onAdWillOpenExternalApp(InneractiveAdSpot inneractiveAdSpot) {
                    String message = "method: onAdWillOpenExternalApp, interstitial: ";
                    Log.d(TAG, message);
                }

                @Override
                public void onAdEnteredErrorState(InneractiveAdSpot inneractiveAdSpot, InneractiveUnitController.AdDisplayError adDisplayError) {
                    String message = "method: onAdEnteredErrorState, interstitial: ";
                    Log.d(TAG, message);
                }

                @Override
                public void onAdWillCloseInternalBrowser(InneractiveAdSpot inneractiveAdSpot) {
                    String message = "method: onAdWillCloseInternalBrowser, interstitial: ";
                    Log.d(TAG, message);
                }

                @Override
                public void onAdDismissed(InneractiveAdSpot inneractiveAdSpot) {
                    String message = "method: onAdWillCloseInternalBrowser, interstitial: ";
                    Log.d(TAG, message);
                    callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
                    loadInterstitialAdvert();
                }
            });
        }

        @Override
        public void onInneractiveFailedAdRequest(InneractiveAdSpot inneractiveAdSpot, InneractiveErrorCode inneractiveErrorCode) {
            String message = "method: onInneractiveFailedAdRequest, interstitial: errorCode: " + inneractiveErrorCode.name();
            Log.d(TAG, message);
            interstitialState = AdvertState.NONE;

            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Interstitial);

            nextInterstitial();
            loadInterstitialAdvertDelayed();
        }
    };

}
