package com.yodo1.mas.mediation.ironsource;

import android.app.Activity;
import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.ironsource.mediationsdk.IronSource;
import com.ironsource.mediationsdk.demandOnly.ISDemandOnlyRewardedVideoListener;
import com.ironsource.mediationsdk.logger.IronSourceError;
import com.yodo1.mas.Yodo1Mas;
import com.yodo1.mas.Yodo1MasLog;
import com.yodo1.mas.analytics.model.Yodo1MasAdRequestResultInfo;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;
import com.yodo1.mas.reward.Yodo1MasRewardAdapterBase;
import com.yodo1.mas.utils.Yodo1MasAdRequestTrackUtil;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

public class Yodo1MasIronSourceRewardAdapter extends Yodo1MasRewardAdapterBase {

    private static final Map<String, Yodo1MasIronSourceRewardAdapter> map = new HashMap<>();

    private final ISDemandOnlyRewardedVideoListener rewardListener = new ISDemandOnlyRewardedVideoListener() {

        @Override
        public void onRewardedVideoAdLoadSuccess(String placementId) {
            String message = "method: onRewardedVideoAdLoadSuccess, placementId: " + placementId;
            Yodo1MasLog.d(TAG, message);
            Yodo1MasIronSourceRewardAdapter instance = getInstanceByPlacementId(placementId);
            if (instance != null) {
                Yodo1MasAdRequestResultInfo adRequestResult
                        = Yodo1MasAdRequestTrackUtil.getYodo1MediationAdRequestResult(Yodo1Mas.AdType.Reward
                        , advertCode, placementId, true, instance.getAdLoadDuration(), null, null);
                instance.callbackLoad(adRequestResult);
            }
        }

        @Override
        public void onRewardedVideoAdLoadFailed(String placementId, IronSourceError adError) {
            String message = "method: onRewardedVideoAdLoadFailed, placementId: " + placementId + ", error: " + adError.toString();
            Yodo1MasLog.d(TAG, message);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            Yodo1MasIronSourceRewardAdapter instance = getInstanceByPlacementId(placementId);
            if (instance != null) {
                Yodo1MasAdRequestResultInfo adRequestResult
                        = Yodo1MasAdRequestTrackUtil.getYodo1MediationAdRequestResult(Yodo1Mas.AdType.Reward
                        , advertCode, placementId, false, instance.getAdLoadDuration(), adError.getErrorCode() + "", adError.getErrorMessage());
                instance.callbackError(error, adError.getErrorCode(), adError.getErrorMessage(), adRequestResult);
            }
        }

        @Override
        public void onRewardedVideoAdOpened(String placementId) {
            String message = "method: onRewardedVideoAdOpened, placementId: " + placementId;
            Yodo1MasLog.d(TAG, message);
            callbackOpen();
        }

        @Override
        public void onRewardedVideoAdClosed(String placementId) {
            String message = "method: onRewardedVideoAdClosed, placementId: " + placementId;
            Yodo1MasLog.d(TAG, message);
            callbackClose();
        }

        @Override
        public void onRewardedVideoAdShowFailed(String placementId, IronSourceError adError) {
            String message = "method: onRewardedVideoAdShowFailed, placementId: " + placementId + ", error: " + adError.toString();
            Yodo1MasLog.d(TAG, message);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            callbackError(error, adError.getErrorCode(), adError.getErrorMessage(), null);
        }

        @Override
        public void onRewardedVideoAdClicked(String placementId) {
            String message = "method: onRewardedVideoAdClicked, placementId: " + placementId;
            Yodo1MasLog.d(TAG, message);
            callbackClick();
        }

        @Override
        public void onRewardedVideoAdRewarded(String placementId) {
            String message = "method: onRewardedVideoAdRewarded, placementId: " + placementId;
            Yodo1MasLog.d(TAG, message);
            callbackEarned();
        }
    };

    public Yodo1MasIronSourceRewardAdapter(Yodo1MasAdapterBase.AdId unitId) {
        super(unitId);
        IronSource.setISDemandOnlyRewardedVideoListener(rewardListener);
    }

    @Override
    public boolean isRewardAdLoaded() {
        String adId = getAdUnitId();
        return !TextUtils.isEmpty(adId) && IronSource.isISDemandOnlyRewardedVideoAvailable(adId);
    }

    @Override
    public void loadRewardAdvert(@NonNull Activity activity) {
        super.loadRewardAdvert(activity);
        String adId = getAdUnitId();
        if (TextUtils.isEmpty(adId)) return;
        if (rewardStatus == Yodo1MasAdapterBase.AdvertStatus.LOADING) return;

        String message = "method: loadRewardAdvert, loading reward ad...";
        Yodo1MasLog.d(TAG, message);
        rewardStatus = Yodo1MasAdapterBase.AdvertStatus.LOADING;
        adLoadStartTimeStamp = System.currentTimeMillis();
        map.put(adId, this);
        IronSource.loadISDemandOnlyRewardedVideo(activity, adId);
    }

    @Override
    public void showRewardAdvertFromActivity(@NonNull Activity activity) {
        super.showRewardAdvertFromActivity(activity);
        if (isRewardAdLoaded()) {
            String message = "method: showRewardAdvert, show reward ad...";
            Yodo1MasLog.d(TAG, message);
            String adId = getAdUnitId();
            IronSource.showISDemandOnlyRewardedVideo(adId);
        }
    }


    private Yodo1MasIronSourceRewardAdapter getInstanceByPlacementId(String placementId) {
        if (map != null && map.containsKey(placementId)) {
            return map.get(placementId);
        }
        return null;
    }

    @Override
    public void destroy() {
        super.destroy();
        Iterator<Map.Entry<String, Yodo1MasIronSourceRewardAdapter>> iterator = map.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, Yodo1MasIronSourceRewardAdapter> entry = iterator.next();
            if (this.equals(entry.getValue())) {
                iterator.remove();
            }
        }
    }
}
