package com.yodo1.mas.mediation.pangle;

import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.text.TextUtils;
import android.util.Log;

import com.bytedance.sdk.openadsdk.AdSlot;
import com.bytedance.sdk.openadsdk.TTAdConfig;
import com.bytedance.sdk.openadsdk.TTAdConstant;
import com.bytedance.sdk.openadsdk.TTAdNative;
import com.bytedance.sdk.openadsdk.TTAdSdk;
import com.bytedance.sdk.openadsdk.TTFullScreenVideoAd;
import com.bytedance.sdk.openadsdk.TTNativeExpressAd;
import com.bytedance.sdk.openadsdk.TTRewardVideoAd;
import com.yodo1.mas.Yodo1Mas;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;
import com.yodo1.mas.mediation.Yodo1MasBannerAdapterBase;
import com.yodo1.sdk.kit.SysUtils;
import com.yodo1.sdk.kit.YLog;

import org.json.JSONObject;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

public class Yodo1MasPangleAdapter extends Yodo1MasAdapterBase {

    private TTAdNative mTTAdNative;
    private TTRewardVideoAd mttRewardVideoAd;
    private TTFullScreenVideoAd mttFullVideoAd;
    private TTNativeExpressAd nativeExpressAd;
    private AdvertState rewardState = AdvertState.NONE;
    private AdvertState interstitialState = AdvertState.NONE;
    private String currentRewardId;
    private String currentInterstitialId;
    private Activity activity;

    @Override
    public String getAdvertCode() {
        return "pangle";
    }

    @Override
    public String getSDKVersion() {
        return BuildConfig.SDK_VERSION_NAME;
    }

    @Override
    public String getMediationVersion() {
        return BuildConfig.MAS_VERSION_NAME;
    }

    public TTAdNative getTTAdNative() {
        return mTTAdNative;
    }

    @Override
    public void initSDK(@NonNull Activity activity, @NonNull Config config, @Nullable InitCallback callback) {
        super.initSDK(activity, config, callback);
        if (isInitSDK()) {
            if (callback != null) {
                callback.onAdapterInitSuccessful(getAdvertCode());
            }
            return;
        }

        if (TextUtils.isEmpty(config.appId)) {
            if (callback != null) {
                callback.onAdapterInitFailed(getAdvertCode(), new Yodo1MasError(Yodo1MasError.CODE_ADVERT_UNINITIALIZED, "config.appId is null"));
            }
            return;
        }

        this.activity = activity;
        init = true;
        TTAdSdk.init(activity, buildConfig(activity, config.appId));
        //创建TTAdNative对象,用于调用广告请求接口
        mTTAdNative = TTAdSdk.getAdManager().createAdNative(activity);

        updatePrivacy();
        loadRewardAdvert();
        loadInterstitialAdvert();
        if (callback != null) {
            callback.onAdapterInitSuccessful(getAdvertCode());
        }
    }

    /**
     * Get TTAdConfig
     * useTextureView: 使用TextureView控件播放视频,默认为SurfaceView,当有SurfaceView冲突的场景，可以使用TextureView
     * allowShowPageWhenScreenLock: 是否在锁屏场景支持展示广告落地页
     * debug: 测试阶段打开，可以通过日志排查问题，上线时去除该调用
     * coppa: Fields to indicate whether you are a child or an adult ，0:adult ，1:child
     * setGDPR: Fields to indicate whether you are protected by GDPR,  the value of GDPR : 0 close GDRP Privacy protection ，1: open GDRP Privacy protection
     *
     * @param context Context
     * @param appId   App Id
     * @return TTAdConfig
     */
    private static TTAdConfig buildConfig(Context context, String appId) {
        return new TTAdConfig.Builder()
                .appId(appId)
                .useTextureView(true)
                .appName(SysUtils.getAppName(context))
                .titleBarTheme(TTAdConstant.TITLE_BAR_THEME_DARK)
                .allowShowPageWhenScreenLock(true)
                .debug(false)
                .coppa(Yodo1MasHelper.getInstance().isCOPPAAgeRestricted() ? 1 : 0)
                .setGDPR(Yodo1MasHelper.getInstance().isGDPRUserConsent() ? 1 : 0)
                .supportMultiProcess(false)
                .build();
    }


    @Override
    public void updatePrivacy() {
        super.updatePrivacy();
    }

    @Override
    public boolean isRewardAdvertLoaded() {
        super.isRewardAdvertLoaded();
        return mttRewardVideoAd != null && rewardState == AdvertState.LOADED;
    }

    @Override
    public void loadRewardAdvert(@NonNull Activity activity) {
        super.loadRewardAdvert(activity);
        if (!isInitSDK()) {
            return;
        }

        AdId adId = getRewardAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId)
                && (mttRewardVideoAd == null || !adId.adId.equals(currentRewardId))
                && rewardState != AdvertState.LOADING) {

            AdSlot adSlot = new AdSlot.Builder()
                    .setCodeId(adId.adId)
                    .setSupportDeepLink(true)
                    .setImageAcceptedSize(1080, 1920)
                    .build();

            mTTAdNative.loadRewardVideoAd(adSlot, rewardAdLister);

            currentRewardId = adId.adId;
            String message = "method: loadRewardAdvert, loading reward ad...";
            Log.d(TAG, message);
            rewardState = AdvertState.LOADING;
        }
    }

    @Override
    public void showRewardAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showRewardAdvert(activity, object, callback);

        if (mttRewardVideoAd != null && isCanShow(Yodo1Mas.AdType.Reward, callback)) {
            String message = "method: showRewardAdvert, show reward ad...";
            Log.d(TAG, message);
            mttRewardVideoAd.showRewardVideoAd(activity);
            rewardState = AdvertState.NONE;
        }
    }

    @Override
    public void dismissRewardAdvert() {
        super.dismissRewardAdvert();
        mttRewardVideoAd = null;
    }

    @Override
    public boolean isInterstitialAdvertLoaded() {
        super.isInterstitialAdvertLoaded();
        return mttFullVideoAd != null && interstitialState == AdvertState.LOADED;
    }

    @Override
    public void loadInterstitialAdvert(@NonNull Activity activity) {
        super.loadInterstitialAdvert(activity);
        if (!isInitSDK()) {
            return;
        }

        AdId adId = getInterstitialAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId)
                && (mttFullVideoAd == null || !adId.adId.equals(currentInterstitialId))) {
            // 判断Android当前的屏幕是横屏还是竖屏。横竖屏判断
            int orientation;
            if (activity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT) {
                orientation = TTAdConstant.VERTICAL;
            } else {
                orientation = TTAdConstant.HORIZONTAL;
            }
            //Orientation必填参数，期望视频的播放方向：TTAdConstant.HORIZONTAL 或 TTAdConstant.VERTICAL
            AdSlot adSlot = new AdSlot.Builder()
                    .setCodeId(adId.adId)
                    .setSupportDeepLink(true)
                    .setImageAcceptedSize(1080, 1920)
                    .setOrientation(orientation)
                    .build();
            //调用插屏广告异步请求接口
            mTTAdNative.loadFullScreenVideoAd(adSlot, interstitialListener);

            currentInterstitialId = adId.adId;
            String message = "method: loadInterstitialAdvert, loading interstitial ad...";
            Log.d(TAG, message);
            interstitialState = AdvertState.LOADING;
        }
    }

    @Override
    public void showInterstitialAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showInterstitialAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Interstitial, callback)) {
            String message = "method: showInterstitialAdvert, show interstitial ad...";
            Log.d(TAG, message);
            if (mttFullVideoAd != null) {
                mttFullVideoAd.showFullScreenVideoAd(activity);
            }
            interstitialState = AdvertState.NONE;
        }
    }

    @Override
    public void dismissInterstitialAdvert() {
        super.dismissInterstitialAdvert();
        mttFullVideoAd = null;
    }

    @Override
    public Yodo1MasBannerAdapterBase getYodo1MasBannerAdapter() {
        Yodo1MasPangleBannerAdapter bannerAdapter = new Yodo1MasPangleBannerAdapter();
        bannerAdapter.bannerAdIds.clear();
        bannerAdapter.bannerAdIds.addAll(bannerAdIds);
        bannerAdapter.relateAdapter = this;
        bannerAdapter.advertCode = getAdvertCode();
        bannerAdapter.sDKVersion = getSDKVersion();
        bannerAdapter.mediationVersion = getMediationVersion();
        return bannerAdapter;
    }

    @Override
    public void onActivityDestroy(@NonNull Activity activity) {
        super.onActivityDestroy(activity);
        if (nativeExpressAd != null) {
            nativeExpressAd.destroy();
            nativeExpressAd = null;
        }
    }

    private TTAdNative.RewardVideoAdListener rewardAdLister = new TTAdNative.RewardVideoAdListener() {

        @Override
        public void onError(int code, String message) {
            String msg = "method: onError, code: " + code + ", message: " + message;
            Log.d(TAG, msg);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Reward);

            rewardState = AdvertState.NONE;
            nextReward();
            loadRewardAdvertDelayed();
        }

        @Override
        public void onRewardVideoAdLoad(TTRewardVideoAd ttRewardVideoAd) {
            YLog.d(TAG + "RewardVideoAd onRewardVideoAdLoad");

            mttRewardVideoAd = ttRewardVideoAd;
            mttRewardVideoAd.setRewardAdInteractionListener(new TTRewardVideoAd.RewardAdInteractionListener() {

                @Override
                public void onAdShow() {
                    String message = "method: onAdDisplayed";
                    Log.d(TAG, message);
                    callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
                }

                @Override
                public void onAdVideoBarClick() {
                    String message = "method: onAdVideoBarClick ";
                    Log.d(TAG, message);
                }

                @Override
                public void onAdClose() {
                    String message = "method: onAdClose";
                    Log.d(TAG, message);
                    callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
                    loadRewardAdvert();
                }

                @Override
                public void onVideoComplete() {
                    String message = "method: onVideoComplete";
                    Log.d(TAG, message);
                }

                @Override
                public void onVideoError() {
                    String message = "method: onVideoError";
                    Log.d(TAG, message);
                    Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
                    callback(error, Yodo1Mas.AdType.Reward);

                    nextReward();
                    loadRewardAdvert();
                }

                @Override
                public void onRewardVerify(boolean rewardVerify, int rewardAmount, String rewardName, int errorCode, String errorMsg) {
                    String message = "RewardVideoAd onRewardVerify, rewardVerify: " + rewardVerify + ", rewardAmount:" + rewardAmount + ", rewardName:" + rewardName + ", errorCode:" + errorCode + ", errorMsg:" + errorMsg;
                    Log.d(TAG, message);
                    callback(Yodo1MasAdEvent.CODE_REWARD_EARNED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
                }

                @Override
                public void onSkippedVideo() {
                    YLog.d(TAG + "RewardVideoAd onSkippedVideo");
                }
            });
        }

        @Override
        public void onRewardVideoCached() {
            String message = "method: onLoad, reward: onRewardVideoCached";
            Log.d(TAG, message);
            rewardState = AdvertState.LOADED;
        }
    };


    private TTAdNative.FullScreenVideoAdListener interstitialListener = new TTAdNative.FullScreenVideoAdListener() {

        @Override
        public void onError(int code, String message) {
            String msg = "method: onError, code: " + code + ", message: " + message;
            Log.d(TAG, msg);
            interstitialState = AdvertState.NONE;

            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Interstitial);

            nextInterstitial();
            loadInterstitialAdvertDelayed();
        }

        @Override
        public void onFullScreenVideoAdLoad(TTFullScreenVideoAd ttFullScreenVideoAd) {
            YLog.d(TAG + "FullScreenVideoAd onFullScreenVideoAdLoad");

            mttFullVideoAd = ttFullScreenVideoAd;
            mttFullVideoAd.setFullScreenVideoAdInteractionListener(new TTFullScreenVideoAd.FullScreenVideoAdInteractionListener() {

                @Override
                public void onAdShow() {
                    String message = "method:FullScreenVideoAd onAdShow: ";
                    Log.d(TAG, message);
                    callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
                }

                @Override
                public void onAdVideoBarClick() {
                    String message = "method: FullScreenVideoAd onAdVideoBarClick: ";
                    Log.d(TAG, message);
                }

                @Override
                public void onAdClose() {
                    String message = "method: FullScreenVideoAd, onAdClose: ";
                    Log.d(TAG, message);
                    callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Interstitial, TAG + ":{" + message + "}");
                    loadInterstitialAdvert();
                }

                @Override
                public void onVideoComplete() {
                    String message = "method: FullScreenVideoAd onVideoComplete: ";
                    Log.d(TAG, message);
                }

                @Override
                public void onSkippedVideo() {
                    String message = "method: FullScreenVideoAd onSkippedVideo: ";
                    Log.d(TAG, message);
                }
            });
        }

        @Override
        public void onFullScreenVideoCached() {
            String message = "method: FullScreenVideoAd";
            Log.d(TAG, message);
            interstitialState = AdvertState.LOADED;
        }
    };

}
