package com.yodo1.mas.mediation.unityads;

import android.app.Activity;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.unity3d.ads.IUnityAdsInitializationListener;
import com.unity3d.ads.IUnityAdsListener;
import com.unity3d.ads.IUnityAdsLoadListener;
import com.unity3d.ads.UnityAds;
import com.unity3d.ads.metadata.MetaData;
import com.unity3d.services.banners.BannerErrorInfo;
import com.unity3d.services.banners.BannerView;
import com.unity3d.services.banners.UnityBannerSize;
import com.yodo1.mas.Yodo1Mas;
import com.yodo1.mas.analytics.Yodo1MasSensorHelper;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.helper.Yodo1MasBanner;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;

import org.json.JSONObject;

public class Yodo1MasUnityAdsAdapter extends Yodo1MasAdapterBase {

    private BannerView bannerAd;

    @Override
    public String getAdvertCode() {
        return "unity";
    }

    @Override
    public String getSDKVersion() {
        return BuildConfig.SDK_VERSION_NAME;
    }

    @Override
    public String getMediationVersion() {
        return BuildConfig.MAS_VERSION_NAME;
    }

    @Override
    public void initSDK(@NonNull Activity activity, @NonNull Config config, @Nullable InitCallback callback) {
        super.initSDK(activity, config, callback);
        if (!isInitSDK()) {
            init = true;
            UnityAds.removeListener(adsListener);
            UnityAds.addListener(adsListener);
            if (!UnityAds.isInitialized()) {
                if (!TextUtils.isEmpty(config.appId)) {
                    UnityAds.initialize(activity.getApplicationContext(), config.appId, Yodo1MasHelper.getInstance().isDebug(), new IUnityAdsInitializationListener() {
                        @Override
                        public void onInitializationComplete() {
                            String message = "method: onInitializationComplete, init successful";
                            Log.d(TAG, message);
                            updatePrivacy();
                            loadRewardAdvert();
                            loadInterstitialAdvert();
                            loadBannerAdvert();
                            if (callback != null) {
                                callback.onAdapterInitSuccessful(getAdvertCode());
                            }
                        }

                        @Override
                        public void onInitializationFailed(UnityAds.UnityAdsInitializationError adError, String adMessage) {
                            String message = "method: onInitializationFailed, error: " + adError + ", message:" + adMessage;
                            Log.d(TAG, message);
                            if (callback != null) {
                                callback.onAdapterInitFailed(getAdvertCode(), new Yodo1MasError(Yodo1MasError.CODE_ADVERT_UNINITIALIZED, message));
                            }
                        }
                    });
                } else {
                    if (callback != null) {
                        callback.onAdapterInitFailed(getAdvertCode(), new Yodo1MasError(Yodo1MasError.CODE_ADVERT_UNINITIALIZED, "config.appId is null"));
                    }
                }
            }
        } else {
            if (callback != null) {
                callback.onAdapterInitSuccessful(getAdvertCode());
            }
        }
    }

    @Override
    public boolean isInitSDK() {
        return super.isInitSDK() && UnityAds.isInitialized();
    }

    @Override
    public void updatePrivacy() {
        super.updatePrivacy();
        if (applicationContext != null) {
            MetaData data = new MetaData(applicationContext);
            data.set("gdpr.consent", Yodo1MasHelper.getInstance().isGDPRUserConsent());
            data.set("privacy.useroveragelimit", Yodo1MasHelper.getInstance().isCOPPAAgeRestricted());
            data.set("privacy.consent", !Yodo1MasHelper.getInstance().isCCPADoNotSell());
            data.commit();
        }
    }

    @Override
    public boolean isRewardAdvertLoaded() {
        super.isRewardAdvertLoaded();
        AdId adId = getRewardAdId();
        return adId != null && !TextUtils.isEmpty(adId.adId) && UnityAds.isReady(adId.adId);
    }

    @Override
    public void loadRewardAdvert(@NonNull Activity activity) {
        super.loadRewardAdvert(activity);
        if (!isInitSDK()) return;

        AdId adId = getRewardAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId)) {
            String message = "method: loadRewardAdvert, loading reward ad...";
            Log.d(TAG, message);
            UnityAds.load(adId.adId, loadListener);
        }
    }

    @Override
    public void showRewardAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showRewardAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Reward, callback)) {
            String message = "method: showRewardAdvert, show reward ad...";
            Log.d(TAG, message);
            UnityAds.show(activity, getRewardAdId().adId);
        }
    }

    @Override
    public boolean isInterstitialAdvertLoaded() {
        super.isInterstitialAdvertLoaded();
        AdId adId = getInterstitialAdId();
        return adId != null && !TextUtils.isEmpty(adId.adId) && UnityAds.isReady(adId.adId);
    }

    @Override
    public void loadInterstitialAdvert(@NonNull Activity activity) {
        if (!isInitSDK()) return;

        AdId adId = getInterstitialAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId)) {
            String message = "method: loadInterstitialAdvert, loading interstitial ad...";
            Log.d(TAG, message);
            UnityAds.load(adId.adId, loadListener);
        }
    }

    @Override
    public void showInterstitialAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showInterstitialAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Interstitial, callback)) {
            String message = "method: showInterstitialAdvert, show interstitial ad...";
            Log.d(TAG, message);
            UnityAds.show(activity, getInterstitialAdId().adId);
        }
    }

    @Override
    public boolean isBannerAdvertLoaded() {
        AdId adId = getBannerAdId();
        return adId != null && !TextUtils.isEmpty(adId.adId) && bannerAd != null && bannerState == AdvertState.LOADED;
    }

    @Override
    public void loadBannerAdvert(@NonNull Activity activity) {
        super.loadBannerAdvert(activity);
        AdId adId = getBannerAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId) && (bannerAd == null || !adId.adId.equals(bannerAd.getPlacementId()))) {
            bannerAd = new BannerView(activity, adId.adId, new UnityBannerSize(320, 50));
            bannerAd.setListener(bannerListener);
        }
        if (bannerAd != null && bannerState != AdvertState.LOADING) {
            String message = "method: loadBannerAdvert, loading banner ad...";
            Log.d(TAG, message);
            bannerAd.load();
            bannerState = AdvertState.LOADING;
        }
    }

    @Override
    public void showBannerAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showBannerAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Banner, callback)) {
            String message = "method: showBannerAdvert, show banner ad...";
            Log.d(TAG, message);
            Yodo1MasBanner.showBanner(activity, bannerAd, object);
            callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Banner, message);
        }
    }

    @Override
    public void dismissBannerAdvert(boolean destroy) {
        super.dismissBannerAdvert(destroy);
        if (bannerAd != null) {
            Yodo1MasBanner.removeBanner(bannerAd);
            if (destroy) {
                bannerAd.destroy();
                bannerAd = null;
                bannerState = AdvertState.NONE;
                loadBannerAdvert();
            }
        }
    }

    private final IUnityAdsLoadListener loadListener = new IUnityAdsLoadListener() {
        @Override
        public void onUnityAdsAdLoaded(String placementId) {
            String message = "method: onUnityAdsAdLoaded, placementId: " + placementId + "}";
            Log.d(TAG, message);
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                trackAdRequest(Yodo1Mas.AdType.Reward, Yodo1MasSensorHelper.AdResult.SUCCESS);
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                trackAdRequest(Yodo1Mas.AdType.Interstitial, Yodo1MasSensorHelper.AdResult.SUCCESS);
            }
        }

        @Override
        public void onUnityAdsFailedToLoad(String placementId) {
            String message = "method: onUnityAdsFailedToLoad, placementId: " + placementId + "}";
            Log.d(TAG, message);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                trackAdRequest(Yodo1Mas.AdType.Reward, Yodo1MasSensorHelper.AdResult.FAIL);
                callback(error, Yodo1Mas.AdType.Reward);

                nextReward();
                loadRewardAdvertDelayed();
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                trackAdRequest(Yodo1Mas.AdType.Interstitial, Yodo1MasSensorHelper.AdResult.FAIL);
                callback(error, Yodo1Mas.AdType.Interstitial);

                nextInterstitial();
                loadInterstitialAdvertDelayed();
            }
        }
    };

    private final IUnityAdsListener adsListener = new IUnityAdsListener() {
        @Override
        public void onUnityAdsReady(String placementId) {
            String message = "method: onUnityAdsStart, placementId: " + placementId + "}";
            Log.d(TAG, message);
        }

        @Override
        public void onUnityAdsStart(String placementId) {
            String message = "method: onUnityAdsReady, placementId: " + placementId + "}";
            Log.d(TAG, message);
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Interstitial, message);
            }
        }

        @Override
        public void onUnityAdsFinish(String placementId, UnityAds.FinishState result) {
            String message = "method: onUnityAdsFinish, placementId: " + placementId;
            Log.d(TAG, message);
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
                callback(Yodo1MasAdEvent.CODE_REWARD_EARNED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
                loadRewardAdvert();
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Interstitial, message);
                loadInterstitialAdvert();
            }
        }

        @Override
        public void onUnityAdsError(UnityAds.UnityAdsError adError, String adMessage) {
            String message = "method: onUnityAdsError, error: " + adError + ", message: " + adMessage;
            Log.d(TAG, message);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Reward);
            callback(error, Yodo1Mas.AdType.Interstitial);
        }
    };

    private final BannerView.IListener bannerListener = new BannerView.Listener() {
        @Override
        public void onBannerLoaded(BannerView bannerAdView) {
            super.onBannerLoaded(bannerAdView);
            String message = "method: onBannerLoaded";
            Log.d(TAG, message);
            bannerState = AdvertState.LOADED;
            trackAdRequest(Yodo1Mas.AdType.Banner, Yodo1MasSensorHelper.AdResult.SUCCESS);
        }

        @Override
        public void onBannerClick(BannerView bannerAdView) {
            super.onBannerClick(bannerAdView);
            String message = "method: onBannerClick";
            Log.d(TAG, message);
        }

        @Override
        public void onBannerFailedToLoad(BannerView bannerAdView, BannerErrorInfo errorInfo) {
            super.onBannerFailedToLoad(bannerAdView, errorInfo);
            String message = "method: onBannerFailedToLoad, error: " + errorInfo.errorMessage + ", code: " + errorInfo.errorCode;
            Log.d(TAG, message);
            trackAdRequest(Yodo1Mas.AdType.Banner, Yodo1MasSensorHelper.AdResult.FAIL);
            bannerState = AdvertState.NONE;

            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            callback(error, Yodo1Mas.AdType.Banner);

            nextBanner();
            loadBannerAdvertDelayed();
        }

        @Override
        public void onBannerLeftApplication(BannerView bannerView) {
            super.onBannerLeftApplication(bannerView);
            String message = "method: onBannerLeftApplication";
            Log.d(TAG, message);
        }
    };
}
