package com.yodo1.mas.mediation.unityads;

import android.app.Activity;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.unity3d.ads.IUnityAdsInitializationListener;
import com.unity3d.ads.IUnityAdsLoadListener;
import com.unity3d.ads.IUnityAdsShowListener;
import com.unity3d.ads.UnityAds;
import com.unity3d.ads.UnityAdsShowOptions;
import com.unity3d.ads.metadata.MetaData;
import com.yodo1.mas.Yodo1Mas;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;
import com.yodo1.mas.mediation.Yodo1MasBannerAdapterBase;

import org.json.JSONObject;

public class Yodo1MasUnityAdsAdapter extends Yodo1MasAdapterBase {

    private final IUnityAdsShowListener showListener = new IUnityAdsShowListener() {
        @Override
        public void onUnityAdsShowFailure(String placementId, UnityAds.UnityAdsShowError error, String message) {
            String msg = "method: onUnityAdsShowFailure, placementId: " + placementId + ", error: " + error.toString() + ", message: " + message + "}";
            Log.d(TAG, msg);
            Yodo1MasError masError = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + msg + "}");
            callbackDisplayError(masError, Yodo1Mas.AdType.Reward, error.toString(), message);
            callbackDisplayError(masError, Yodo1Mas.AdType.Interstitial, error.toString(), message);
        }

        @Override
        public void onUnityAdsShowStart(String placementId) {
            String message = "method: onUnityAdsShowStart, placementId: " + placementId + "}";
            Log.d(TAG, message);
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Interstitial, message);
            }
        }

        @Override
        public void onUnityAdsShowClick(String placementId) {
            String message = "method: onUnityAdsShowClick, placementId: " + placementId + "}";
            Log.d(TAG, message);
        }

        @Override
        public void onUnityAdsShowComplete(String placementId, UnityAds.UnityAdsShowCompletionState state) {
            String message = "method: onUnityAdsShowComplete, placementId: " + placementId + "}";
            Log.d(TAG, message);

            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                if (state == UnityAds.UnityAdsShowCompletionState.COMPLETED) {
                    callback(Yodo1MasAdEvent.CODE_REWARD_EARNED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
                }
                callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
                loadRewardAdvert();
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Interstitial, message);
                loadInterstitialAdvert();
            }
        }
    };
    private AdvertState rewardState = AdvertState.NONE;
    private AdvertState interstitialState = AdvertState.NONE;
    private final IUnityAdsLoadListener loadListener = new IUnityAdsLoadListener() {
        @Override
        public void onUnityAdsAdLoaded(String placementId) {
            String message = "method: onUnityAdsAdLoaded, placementId: " + placementId + "}";
            Log.d(TAG, message);
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                rewardState = AdvertState.LOADED;
                trackAdRequestSuccessed(Yodo1Mas.AdType.Reward);
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                interstitialState = AdvertState.LOADED;
                trackAdRequestSuccessed(Yodo1Mas.AdType.Interstitial);
            }
        }

        @Override
        public void onUnityAdsFailedToLoad(String placementId, UnityAds.UnityAdsLoadError error, String message) {
            String msg = "method: onUnityAdsFailedToLoad, placementId: " + placementId + " error: " + error.toString() + " message: " + message + "}";
            Log.d(TAG, msg);
            Yodo1MasError yodo1MasError = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                rewardState = AdvertState.NONE;
                trackAdRequestFailed(Yodo1Mas.AdType.Reward, error.toString(), message);
                callback(yodo1MasError, Yodo1Mas.AdType.Reward);

                nextReward();
                loadRewardAdvertDelayed();
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                interstitialState = AdvertState.NONE;

                trackAdRequestFailed(Yodo1Mas.AdType.Interstitial, error.toString(), message);
                callback(yodo1MasError, Yodo1Mas.AdType.Interstitial);

                nextInterstitial();
                loadInterstitialAdvertDelayed();
            }
        }
    };

    /**
     * Set UnityAds's privacy
     * About UnityAds's privacy, please see this link https://docs.unity3d.com/Packages/com.unity.ads@3.6/manual/LegalDataPrivacy.html?q=useroveragelimit
     *
     * @param activity Activity
     */
    public static void setUnityAdsPrivacy(@NonNull Activity activity) {

        MetaData data = new MetaData(activity);
        data.set("gdpr.consent", Yodo1MasHelper.getInstance().isGDPRUserConsent());
        /**
         * // If the user is over the specified age limit:
         * MetaData ageGateMetaData = new MetaData(this);
         * ageGateMetaData.set("privacy.useroveragelimit", true);
         * ageGateMetaData.commit();
         *
         * // If the user is under the specified age limit:
         * MetaData ageGateMetaData = new MetaData(this);
         * ageGateMetaData.set("privacy.useroveragelimit", false);
         * ageGateMetaData.commit();
         */
        data.set("privacy.useroveragelimit", !Yodo1MasHelper.getInstance().isCOPPAAgeRestricted());
        /**
         * // If the user opts in to targeted advertising:
         * MetaData privacyMetaData = new MetaData(this);
         * privacyMetaData.set("privacy.consent", true);
         * privacyMetaData.commit();
         *
         * // If the user opts out of targeted advertising:
         * MetaData privacyMetaData = new MetaData(this);
         * privacyMetaData.set("privacy.consent", false);
         * privacyMetaData.commit();
         */
        data.set("privacy.consent", !Yodo1MasHelper.getInstance().isCCPADoNotSell());

        /**
         * Google Play Families compliance, you can find out details from the below link
         * https://docs.unity.com/ads/GoogleFamiliesCompliance.html?utm_campaign=Other_global_Transactional_2022-03-dg-operate-unity-ads-GoogleFamiliesPolicyUpdate-campaign&utm_content=2022-03-dg-operate-unity-ads-GoogleFamiliesPolicyUpdate&utm_medium=email&utm_source=Eloqua#AndroidCode
         */
        // This is a mixed audience game.
        data.set("privacy.mode", "mixed");
        // true If the user opts out of personalized ads
        // false If the user opts in to personalized ads
        data.set("user.nonbehavioral", Yodo1MasHelper.getInstance().isCOPPAAgeRestricted());

        data.commit();
    }

    @Override
    public String getAdvertCode() {
        return "unity";
    }

    @Override
    public String getSDKVersion() {
        return BuildConfig.SDK_VERSION_NAME;
    }

    @Override
    public String getMediationVersion() {
        return BuildConfig.MAS_VERSION_NAME;
    }

    @Override
    public void initSDK(@NonNull Activity activity, @NonNull Config config, @Nullable InitCallback callback) {
        super.initSDK(activity, config, callback);
        if (TextUtils.isEmpty(config.appId)) {
            if (callback != null) {
                callback.onAdapterInitFailed(getAdvertCode(), new Yodo1MasError(Yodo1MasError.CODE_ADVERT_UNINITIALIZED, "config.appId is null"));
            }
            return;
        }

        if (!isInitSDK()) {
            init = true;
            if (!UnityAds.isInitialized()) {
                updatePrivacy();
                UnityAds.initialize(activity.getApplicationContext(), config.appId, Yodo1MasHelper.getInstance().isDebug(), new IUnityAdsInitializationListener() {
                    @Override
                    public void onInitializationComplete() {
                        String message = "method: onInitializationComplete, init successful";
                        Log.d(TAG, message);
                        loadRewardAdvert();
                        loadInterstitialAdvert();
                        if (callback != null) {
                            callback.onAdapterInitSuccessful(getAdvertCode());
                        }
                    }

                    @Override
                    public void onInitializationFailed(UnityAds.UnityAdsInitializationError adError, String adMessage) {
                        String message = "method: onInitializationFailed, error: " + adError + ", message:" + adMessage;
                        Log.d(TAG, message);
                        if (callback != null) {
                            callback.onAdapterInitFailed(getAdvertCode(), new Yodo1MasError(Yodo1MasError.CODE_ADVERT_UNINITIALIZED, message));
                        }
                    }
                });
            }
        } else {
            if (callback != null) {
                callback.onAdapterInitSuccessful(getAdvertCode());
            }
        }
    }

    @Override
    public boolean isInitSDK() {
        return super.isInitSDK() && UnityAds.isInitialized();
    }

    @Override
    public void updatePrivacy() {
        super.updatePrivacy();
        setUnityAdsPrivacy(currentActivity);
    }

    @Override
    public boolean isRewardAdvertLoaded() {
        super.isRewardAdvertLoaded();
        AdId adId = getRewardAdId();
        return adId != null && !TextUtils.isEmpty(adId.adId) && rewardState == AdvertState.LOADED;
    }

    @Override
    public void loadRewardAdvert(@NonNull Activity activity) {
        super.loadRewardAdvert(activity);
        if (!isInitSDK()) {
            return;
        }

        AdId adId = getRewardAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId) && rewardState != AdvertState.LOADING) {
            String message = "method: loadRewardAdvert, loading reward ad...";
            Log.d(TAG, message);
            UnityAds.load(adId.adId, loadListener);
        }
    }

    @Override
    public void showRewardAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showRewardAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Reward, callback)) {
            String message = "method: showRewardAdvert, show reward ad...";
            Log.d(TAG, message);
            UnityAds.show(activity, getRewardAdId().adId, new UnityAdsShowOptions(), showListener);
            rewardState = AdvertState.NONE;
        }
    }

    @Override
    public boolean isInterstitialAdvertLoaded() {
        super.isInterstitialAdvertLoaded();
        AdId adId = getInterstitialAdId();
        return adId != null && !TextUtils.isEmpty(adId.adId) && interstitialState == AdvertState.LOADED;
    }

    @Override
    public void loadInterstitialAdvert(@NonNull Activity activity) {
        if (!isInitSDK()) {
            return;
        }

        AdId adId = getInterstitialAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId) && interstitialState != AdvertState.LOADING) {
            String message = "method: loadInterstitialAdvert, loading interstitial ad...";
            Log.d(TAG, message);
            UnityAds.load(adId.adId, loadListener);
        }
    }

    @Override
    public void showInterstitialAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showInterstitialAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Interstitial, callback)) {
            String message = "method: showInterstitialAdvert, show interstitial ad...";
            Log.d(TAG, message);
            UnityAds.show(activity, getInterstitialAdId().adId, new UnityAdsShowOptions(), showListener);
            interstitialState = AdvertState.NONE;
        }
    }

    @Override
    public Yodo1MasBannerAdapterBase getYodo1MasBannerAdapter() {
        Yodo1MasUnityAdsBannerAdapter bannerAdapter = new Yodo1MasUnityAdsBannerAdapter();
        bannerAdapter.bannerAdIds.clear();
        bannerAdapter.bannerAdIds.addAll(bannerAdIds);
        bannerAdapter.relateAdapter = this;
        bannerAdapter.advertCode = getAdvertCode();
        bannerAdapter.sdkVersion = getSDKVersion();
        bannerAdapter.mediationVersion = getMediationVersion();
        return bannerAdapter;
    }
}
