package com.yodo1.mas.mediation.vungle;

import android.app.Activity;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.vungle.warren.AdConfig;
import com.vungle.warren.Banners;
import com.vungle.warren.LoadAdCallback;
import com.vungle.warren.PlayAdCallback;
import com.vungle.warren.Vungle;
import com.vungle.warren.VungleBanner;
import com.vungle.warren.error.VungleException;
import com.yodo1.mas.Yodo1Mas;
import com.yodo1.mas.analytics.Yodo1MasSensorHelper;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.helper.Yodo1MasBanner;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;

import org.json.JSONObject;

public class Yodo1MasVungleAdapter extends Yodo1MasAdapterBase {

    private VungleBanner bannerAd;
    private String currBannerUnitId;

    @Override
    public String getAdvertCode() {
        return "vungle";
    }

    @Override
    public String getSDKVersion() {
        return BuildConfig.SDK_VERSION_NAME;
    }

    @Override
    public String getMediationVersion() {
        return BuildConfig.MAS_VERSION_NAME;
    }

    @Override
    public void initSDK(@NonNull Activity activity, @NonNull Config config, @Nullable InitCallback callback) {
        super.initSDK(activity, config, callback);
        if (!isInitSDK()) {
            init = true;
            if (!Vungle.isInitialized()) {
                if (!TextUtils.isEmpty(config.appId)) {
                    Vungle.init(config.appId, activity.getApplicationContext(), new com.vungle.warren.InitCallback() {
                        @Override
                        public void onSuccess() {
                            String message = "method: onSuccess, init successful";
                            Log.d(TAG, message);
                            updatePrivacy();
                            loadRewardAdvert();
                            loadInterstitialAdvert();
                            loadBannerAdvert();
                            if (callback != null) {
                                callback.onAdapterInitSuccessful(getAdvertCode());
                            }
                        }

                        @Override
                        public void onError(VungleException exception) {
                            String message = "method: onError, exception: " + exception.toString();
                            Log.d(TAG, message);
                            if (callback != null) {
                                callback.onAdapterInitFailed(getAdvertCode(), new Yodo1MasError(Yodo1MasError.CODE_ADVERT_UNINITIALIZED, message));
                            }
                        }

                        @Override
                        public void onAutoCacheAdAvailable(String placementId) {
                            String message = "method: onAutoCacheAdAvailable, placementId: " + placementId;
                            Log.d(TAG, message);
                        }
                    });
                } else {
                    if (callback != null) {
                        callback.onAdapterInitFailed(getAdvertCode(), new Yodo1MasError(Yodo1MasError.CODE_ADVERT_UNINITIALIZED, "config.appId is null"));
                    }
                }
            }
        } else {
            if (callback != null) {
                callback.onAdapterInitSuccessful(getAdvertCode());
            }
        }
    }

    @Override
    public boolean isInitSDK() {
        return super.isInitSDK() && Vungle.isInitialized();
    }

    @Override
    public void updatePrivacy() {
        super.updatePrivacy();
        Vungle.Consent gdprStatus = Yodo1MasHelper.getInstance().isGDPRUserConsent() ? Vungle.Consent.OPTED_IN : Vungle.Consent.OPTED_OUT;
        Vungle.updateConsentStatus(gdprStatus, "");

        Vungle.Consent ccpaStatus = Yodo1MasHelper.getInstance().isCCPADoNotSell() ? Vungle.Consent.OPTED_OUT : Vungle.Consent.OPTED_IN;
        Vungle.updateCCPAStatus(ccpaStatus);
    }

    @Override
    public boolean isRewardAdvertLoaded() {
        super.isRewardAdvertLoaded();
        AdId adId = getRewardAdId();
        return adId != null && !TextUtils.isEmpty(adId.adId) && Vungle.canPlayAd(adId.adId);
    }

    @Override
    public void loadRewardAdvert(@NonNull Activity activity) {
        super.loadRewardAdvert(activity);
        if (!isInitSDK()) return;

        AdId adId = getRewardAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId)) {
            String message = "method: loadRewardAdvert, loading reward ad...";
            Log.d(TAG, message);
            Vungle.loadAd(adId.adId, loadListener);
        }
    }

    @Override
    public void showRewardAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showRewardAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Reward, callback)) {
            String message = "method: showRewardAdvert, show reward ad...";
            Log.d(TAG, message);
            Vungle.playAd(getRewardAdId().adId, null, adsListener);
        }
    }

    @Override
    public boolean isInterstitialAdvertLoaded() {
        super.isInterstitialAdvertLoaded();
        AdId adId = getInterstitialAdId();
        return adId != null && !TextUtils.isEmpty(adId.adId) && Vungle.canPlayAd(adId.adId);
    }

    @Override
    public void loadInterstitialAdvert(@NonNull Activity activity) {
        super.loadInterstitialAdvert(activity);
        if (!isInitSDK()) return;

        AdId adId = getInterstitialAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId)) {
            String message = "method: loadInterstitialAdvert, loading interstitial ad...";
            Log.d(TAG, message);
            Vungle.loadAd(adId.adId, loadListener);
        }
    }

    @Override
    public void showInterstitialAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showInterstitialAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Interstitial, callback)) {
            String message = "method: showInterstitialAdvert, show interstitial ad...";
            Log.d(TAG, message);
            Vungle.playAd(getInterstitialAdId().adId, null, adsListener);
        }
    }


    @Override
    public boolean isBannerAdvertLoaded() {
        super.isBannerAdvertLoaded();
        AdId adId = getBannerAdId();
        return adId != null && !TextUtils.isEmpty(adId.adId) && Banners.canPlayAd(adId.adId, AdConfig.AdSize.BANNER);
    }

    @Override
    public void loadBannerAdvert(@NonNull Activity activity) {
        super.loadBannerAdvert(activity);

        AdId adId = getBannerAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId) && (TextUtils.isEmpty(currBannerUnitId) || !adId.adId.equals(currBannerUnitId))) {
            String message = "method: loadBannerAdvert, loading banner ad...";
            Log.d(TAG, message);

            Banners.loadBanner(getBannerAdId().adId, AdConfig.AdSize.BANNER, loadListener);
            currBannerUnitId = adId.adId;
        }
    }

    @Override
    public void showBannerAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showBannerAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Banner, callback)) {
            String message = "method: showBannerAdvert, show banner ad...";
            Log.d(TAG, message);
            bannerAd = Banners.getBanner(getBannerAdId().adId, AdConfig.AdSize.BANNER, adsListener);
            if (bannerAd != null) {
                Yodo1MasBanner.showBanner(activity, bannerAd, object);
            } else {
                Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, message);
                callback(error, Yodo1Mas.AdType.Banner);
                nextBanner();
                loadBannerAdvertDelayed();
            }
        }
    }

    @Override
    public void dismissBannerAdvert(boolean destroy) {
        super.dismissBannerAdvert(destroy);
        if (bannerAd != null) {
            Yodo1MasBanner.removeBanner(bannerAd);
            if (destroy) {
                bannerAd.destroyAd();
                bannerAd = null;
                currBannerUnitId = null;
                bannerState = AdvertState.NONE;
                loadBannerAdvert();
            }
        }
    }

    private final LoadAdCallback loadListener = new LoadAdCallback() {
        @Override
        public void onAdLoad(String placementId) {
            String message = "method: onAdLoad, placementId: " + placementId + "}";
            Log.d(TAG, message);

            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                trackAdRequest(Yodo1Mas.AdType.Reward, Yodo1MasSensorHelper.AdResult.SUCCESS);
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                trackAdRequest(Yodo1Mas.AdType.Interstitial, Yodo1MasSensorHelper.AdResult.SUCCESS);
            } else if (getBannerAdId() != null && placementId.equals(getBannerAdId().adId)) {
                trackAdRequest(Yodo1Mas.AdType.Banner, Yodo1MasSensorHelper.AdResult.SUCCESS);
            }
        }

        @Override
        public void onError(String placementId, VungleException exception) {
            String message = "method: onError, placementId: " + placementId + "exception: " + exception.toString() + "}";
            Log.d(TAG, message);

            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                trackAdRequest(Yodo1Mas.AdType.Reward, Yodo1MasSensorHelper.AdResult.FAIL);
                callback(error, Yodo1Mas.AdType.Reward);

                nextReward();
                loadRewardAdvertDelayed();
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                trackAdRequest(Yodo1Mas.AdType.Interstitial, Yodo1MasSensorHelper.AdResult.FAIL);
                callback(error, Yodo1Mas.AdType.Interstitial);

                nextInterstitial();
                loadInterstitialAdvertDelayed();
            } else if (getBannerAdId() != null && placementId.equals(getBannerAdId().adId)) {
                trackAdRequest(Yodo1Mas.AdType.Banner, Yodo1MasSensorHelper.AdResult.FAIL);
                callback(error, Yodo1Mas.AdType.Banner);

                nextBanner();
                loadBannerAdvertDelayed();
            }
        }
    };

    private final PlayAdCallback adsListener = new PlayAdCallback() {
        @Override
        public void onAdStart(String placementId) {
            String message = "method: onAdStart, placementId: " + placementId + "}";
            Log.d(TAG, message);
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Interstitial, message);
            } else if (getBannerAdId() != null && placementId.equals(getBannerAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Banner, message);
            }
        }

        @Override
        public void onAdEnd(String placementId, boolean completed, boolean isCTAClicked) {

        }

        @Override
        public void onAdEnd(String placementId) {
            String message = "method: onEnd, placementId: " + placementId + "}";
            Log.d(TAG, message);
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
                loadRewardAdvert();
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Interstitial, message);
                loadInterstitialAdvert();
            } else if (getBannerAdId() != null && placementId.equals(getBannerAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Banner, message);
                loadBannerAdvert();
            }
        }

        @Override
        public void onAdClick(String placementId) {
            String message = "method: onAdClick, placementId: " + placementId + "}";
            Log.d(TAG, message);
        }

        @Override
        public void onAdRewarded(String placementId) {
            String message = "method: onAdRewarded, placementId: " + placementId + "}";
            Log.d(TAG, message);
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_REWARD_EARNED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
            }
        }

        @Override
        public void onAdLeftApplication(String placementId) {

        }

        @Override
        public void onError(String placementId, VungleException exception) {
            String message = "method: onError, placementId: " + placementId + "exception: " + exception.toString() + "}";
            Log.d(TAG, message);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                callback(error, Yodo1Mas.AdType.Reward);

                nextReward();
                loadRewardAdvertDelayed();
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                callback(error, Yodo1Mas.AdType.Interstitial);

                nextInterstitial();
                loadInterstitialAdvertDelayed();
            } else if (getBannerAdId() != null && placementId.equals(getBannerAdId().adId)) {
                callback(error, Yodo1Mas.AdType.Banner);

                nextBanner();
                loadBannerAdvertDelayed();
            }
        }

        @Override
        public void onAdViewed(String placementId) {

        }
    };
}
