package com.yodo1.mas.mediation.vungle;

import android.app.Activity;
import android.text.TextUtils;
import android.util.Log;

import com.vungle.warren.LoadAdCallback;
import com.vungle.warren.PlayAdCallback;
import com.vungle.warren.Vungle;
import com.vungle.warren.error.VungleException;
import com.yodo1.mas.Yodo1Mas;
import com.yodo1.mas.analytics.Yodo1MasSensorHelper;
import com.yodo1.mas.error.Yodo1MasError;
import com.yodo1.mas.event.Yodo1MasAdEvent;
import com.yodo1.mas.helper.Yodo1MasHelper;
import com.yodo1.mas.mediation.Yodo1MasAdapterBase;
import com.yodo1.mas.mediation.Yodo1MasBannerAdapterBase;

import org.json.JSONObject;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

public class Yodo1MasVungleAdapter extends Yodo1MasAdapterBase {

    @Override
    public String getAdvertCode() {
        return "vungle";
    }

    @Override
    public String getSDKVersion() {
        return BuildConfig.SDK_VERSION_NAME;
    }

    @Override
    public String getMediationVersion() {
        return BuildConfig.MAS_VERSION_NAME;
    }

    @Override
    public void initSDK(@NonNull Activity activity, @NonNull Config config, @Nullable InitCallback callback) {
        super.initSDK(activity, config, callback);
        if (!isInitSDK()) {
            init = true;
            if (!Vungle.isInitialized()) {
                if (!TextUtils.isEmpty(config.appId)) {
                    setVungleCOPPAPrivacy();
                    Vungle.init(config.appId, activity.getApplicationContext(), new com.vungle.warren.InitCallback() {
                        @Override
                        public void onSuccess() {
                            String message = "method: onSuccess, init successful";
                            Log.d(TAG, message);
                            updatePrivacy();
                            loadRewardAdvert();
                            loadInterstitialAdvert();
                            if (callback != null) {
                                callback.onAdapterInitSuccessful(getAdvertCode());
                            }
                        }

                        @Override
                        public void onError(VungleException exception) {
                            String message = "method: onError, exception: " + exception.toString();
                            Log.d(TAG, message);
                            if (callback != null) {
                                callback.onAdapterInitFailed(getAdvertCode(), new Yodo1MasError(Yodo1MasError.CODE_ADVERT_UNINITIALIZED, message));
                            }
                        }

                        @Override
                        public void onAutoCacheAdAvailable(String placementId) {
                            String message = "method: onAutoCacheAdAvailable, placementId: " + placementId;
                            Log.d(TAG, message);
                        }
                    });
                } else {
                    if (callback != null) {
                        callback.onAdapterInitFailed(getAdvertCode(), new Yodo1MasError(Yodo1MasError.CODE_ADVERT_UNINITIALIZED, "config.appId is null"));
                    }
                }
            }
        } else {
            if (callback != null) {
                callback.onAdapterInitSuccessful(getAdvertCode());
            }
        }
    }

    @Override
    public boolean isInitSDK() {
        return super.isInitSDK() && Vungle.isInitialized();
    }

    @Override
    public void updatePrivacy() {
        super.updatePrivacy();
        setVunglePrivacy();
    }

    /**
     * need call before initialization the Vungle SDK
     */
    private void setVungleCOPPAPrivacy() {
        Vungle.updateUserCoppaStatus(Yodo1MasHelper.getInstance().isCOPPAAgeRestricted());
    }

    public static void setVunglePrivacy() {
        Vungle.Consent gdprStatus = Yodo1MasHelper.getInstance().isGDPRUserConsent() ? Vungle.Consent.OPTED_IN : Vungle.Consent.OPTED_OUT;
        Vungle.updateConsentStatus(gdprStatus, "");

        Vungle.Consent ccpaStatus = Yodo1MasHelper.getInstance().isCCPADoNotSell() ? Vungle.Consent.OPTED_OUT : Vungle.Consent.OPTED_IN;
        Vungle.updateCCPAStatus(ccpaStatus);
    }

    @Override
    public boolean isRewardAdvertLoaded() {
        super.isRewardAdvertLoaded();
        AdId adId = getRewardAdId();
        return adId != null && !TextUtils.isEmpty(adId.adId) && Vungle.canPlayAd(adId.adId);
    }

    @Override
    public void loadRewardAdvert(@NonNull Activity activity) {
        super.loadRewardAdvert(activity);
        if (!isInitSDK()) {
            return;
        }

        AdId adId = getRewardAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId)) {
            String message = "method: loadRewardAdvert, loading reward ad...";
            Log.d(TAG, message);
            Vungle.loadAd(adId.adId, loadListener);
        }
    }

    @Override
    public void showRewardAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showRewardAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Reward, callback)) {
            String message = "method: showRewardAdvert, show reward ad...";
            Log.d(TAG, message);
            Vungle.playAd(getRewardAdId().adId, null, adsListener);
        }
    }

    @Override
    public boolean isInterstitialAdvertLoaded() {
        super.isInterstitialAdvertLoaded();
        AdId adId = getInterstitialAdId();
        return adId != null && !TextUtils.isEmpty(adId.adId) && Vungle.canPlayAd(adId.adId);
    }

    @Override
    public void loadInterstitialAdvert(@NonNull Activity activity) {
        super.loadInterstitialAdvert(activity);
        if (!isInitSDK()) {
            return;
        }

        AdId adId = getInterstitialAdId();
        if (adId != null && !TextUtils.isEmpty(adId.adId)) {
            String message = "method: loadInterstitialAdvert, loading interstitial ad...";
            Log.d(TAG, message);
            Vungle.loadAd(adId.adId, loadListener);
        }
    }

    @Override
    public void showInterstitialAdvert(@NonNull Activity activity, @Nullable JSONObject object, @Nullable AdvertCallback callback) {
        super.showInterstitialAdvert(activity, object, callback);
        if (isCanShow(Yodo1Mas.AdType.Interstitial, callback)) {
            String message = "method: showInterstitialAdvert, show interstitial ad...";
            Log.d(TAG, message);
            Vungle.playAd(getInterstitialAdId().adId, null, adsListener);
        }
    }

    @Override
    public Yodo1MasBannerAdapterBase getYodo1MasBannerAdapter() {
        Yodo1MasVungleBannerAdapter bannerAdapter = new Yodo1MasVungleBannerAdapter();
        bannerAdapter.bannerAdIds.clear();
        bannerAdapter.bannerAdIds.addAll(bannerAdIds);
        bannerAdapter.relateAdapter = this;
        bannerAdapter.advertCode = getAdvertCode();
        bannerAdapter.sdkVersion = getSDKVersion();
        bannerAdapter.mediationVersion = getMediationVersion();
        return bannerAdapter;
    }

    private final LoadAdCallback loadListener = new LoadAdCallback() {
        @Override
        public void onAdLoad(String placementId) {
            String message = "method: onAdLoad, placementId: " + placementId + "}";
            Log.d(TAG, message);

            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                trackAdRequestSuccessed(Yodo1Mas.AdType.Reward);
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                trackAdRequestSuccessed(Yodo1Mas.AdType.Interstitial);
            }
        }

        @Override
        public void onError(String placementId, VungleException exception) {
            String message = "method: onError, placementId: " + placementId + "exception: " + exception.toString() + "}";
            Log.d(TAG, message);

            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_LOAD_FAIL, TAG + ":{" + message + "}");
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                trackAdRequestFailed(Yodo1Mas.AdType.Reward, exception.getExceptionCode(), exception.getLocalizedMessage());
                callback(error, Yodo1Mas.AdType.Reward);

                nextReward();
                loadRewardAdvertDelayed();
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                trackAdRequestFailed(Yodo1Mas.AdType.Interstitial, exception.getExceptionCode(), exception.getLocalizedMessage());
                callback(error, Yodo1Mas.AdType.Interstitial);

                nextInterstitial();
                loadInterstitialAdvertDelayed();
            }
        }
    };

    private final PlayAdCallback adsListener = new PlayAdCallback() {
        @Override
        public void creativeId(String creativeId) {
            String message = "method: creativeId, creativeId: " + creativeId + "}";
            Log.d(TAG, message);
        }

        @Override
        public void onAdStart(String placementId) {
            String message = "method: onAdStart, placementId: " + placementId + "}";
            Log.d(TAG, message);
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_OPENED, Yodo1Mas.AdType.Interstitial, message);
            }
        }

        @Override
        public void onAdEnd(String placementId, boolean completed, boolean isCTAClicked) {

        }

        @Override
        public void onAdEnd(String placementId) {
            String message = "method: onEnd, placementId: " + placementId + "}";
            Log.d(TAG, message);
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
                loadRewardAdvert();
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_CLOSED, Yodo1Mas.AdType.Interstitial, message);
                loadInterstitialAdvert();
            }
        }

        @Override
        public void onAdClick(String placementId) {
            String message = "method: onAdClick, placementId: " + placementId + "}";
            Log.d(TAG, message);
        }

        @Override
        public void onAdRewarded(String placementId) {
            String message = "method: onAdRewarded, placementId: " + placementId + "}";
            Log.d(TAG, message);
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                callback(Yodo1MasAdEvent.CODE_REWARD_EARNED, Yodo1Mas.AdType.Reward, TAG + ":{" + message + "}");
            }
        }

        @Override
        public void onAdLeftApplication(String placementId) {

        }

        @Override
        public void onError(String placementId, VungleException exception) {
            String message = "method: onError, placementId: " + placementId + "exception: " + exception.toString() + "}";
            Log.d(TAG, message);
            Yodo1MasError error = new Yodo1MasError(Yodo1MasError.CODE_ADVERT_SHOW_FAIL, TAG + ":{" + message + "}");
            if (getRewardAdId() != null && placementId.equals(getRewardAdId().adId)) {
                callbackDisplayError(error, Yodo1Mas.AdType.Reward, exception.getExceptionCode(), exception.getLocalizedMessage());

                nextReward();
                loadRewardAdvertDelayed();
            } else if (getInterstitialAdId() != null && placementId.equals(getInterstitialAdId().adId)) {
                callbackDisplayError(error, Yodo1Mas.AdType.Interstitial, exception.getExceptionCode(), exception.getLocalizedMessage());

                nextInterstitial();
                loadInterstitialAdvertDelayed();
            }
        }

        @Override
        public void onAdViewed(String placementId) {

        }
    };
}
