// Copyright (c) 2018, Yubico AB
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this
//    list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright notice,
//    this list of conditions and the following disclaimer in the documentation
//    and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

package com.yubico.webauthn.data;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.yubico.internal.util.json.JsonStringSerializable;
import com.yubico.internal.util.json.JsonStringSerializer;
import java.util.Optional;
import java.util.stream.Stream;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.NonNull;


/**
 * Relying Parties may use this to specify their preference regarding attestation conveyance during credential
 * generation.
 *
 * @see <a href="https://www.w3.org/TR/2019/PR-webauthn-20190117/#attestation-conveyance">§5.4.6. Attestation Conveyance
 * Preference
 * Enumeration (enum AttestationConveyancePreference)
 * </a>
 */
@JsonSerialize(using = JsonStringSerializer.class)
@AllArgsConstructor(access = AccessLevel.PRIVATE)
public enum AttestationConveyancePreference implements JsonStringSerializable {

    /**
     * Indicates that the Relying Party is not interested in authenticator attestation.
     * <p>
     * For example, in order to potentially avoid having to obtain user consent to relay identifying information to the
     * Relying Party, or to save a roundtrip to an <a href="https://www.w3.org/TR/2019/PR-webauthn-20190117/#attestation-ca">Attestation
     * CA</a>.
     * </p>
     * <p>
     * This is the default value.
     * </p>
     */
    NONE("none"),

    /**
     * Indicates that the Relying Party prefers an attestation conveyance yielding verifiable attestation statements,
     * but allows the client to decide how to obtain such attestation statements. The client MAY replace the
     * authenticator-generated attestation statements with attestation statements generated by an <a
     * href="https://www.w3.org/TR/2019/PR-webauthn-20190117/#anonymization-ca">Anonymization CA</a>, in order to
     * protect the user’s privacy, or to assist Relying Parties with attestation verification in a heterogeneous
     * ecosystem.
     * <p>
     * Note: There is no guarantee that the Relying Party will obtain a verifiable attestation statement in this case.
     * For example, in the case that the authenticator employs self attestation.
     * </p>
     */
    INDIRECT("indirect"),

    /**
     * Indicates that the Relying Party wants to receive the attestation statement as generated by the authenticator.
     */
    DIRECT("direct");

    @NonNull
    private final String id;

    private static Optional<AttestationConveyancePreference> fromString(@NonNull String id) {
        return Stream.of(values()).filter(v -> v.id.equals(id)).findAny();
    }

    @JsonCreator
    private static AttestationConveyancePreference fromJsonString(@NonNull String id) {
        return fromString(id).orElseThrow(() -> new IllegalArgumentException(String.format(
            "Unknown %s value: %s", AttestationConveyancePreference.class.getSimpleName(), id
        )));
    }

    @Override
    public String toJsonString() {
        return id;
    }

}
