// Copyright (c) 2018, Yubico AB
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this
//    list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright notice,
//    this list of conditions and the following disclaimer in the documentation
//    and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

package com.yubico.webauthn.data;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;
import java.util.Optional;
import java.util.stream.Stream;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NonNull;

/**
 * Relying Parties may use this to specify their preference regarding attestation conveyance during
 * credential generation.
 *
 * @see <a href="https://www.w3.org/TR/2021/REC-webauthn-2-20210408/#attestation-conveyance">§5.4.6.
 *     Attestation Conveyance Preference Enumeration (enum AttestationConveyancePreference) </a>
 */
@AllArgsConstructor(access = AccessLevel.PRIVATE)
public enum AttestationConveyancePreference {

  /**
   * Indicates that the Relying Party is not interested in authenticator attestation.
   *
   * <p>For example, in order to potentially avoid having to obtain user consent to relay
   * identifying information to the Relying Party, or to save a roundtrip to an <a
   * href="https://www.w3.org/TR/2021/REC-webauthn-2-20210408/#attestation-ca">Attestation CA</a>.
   *
   * <p>This is the default value.
   */
  NONE("none"),

  /**
   * Indicates that the Relying Party prefers an attestation conveyance yielding verifiable
   * attestation statements, but allows the client to decide how to obtain such attestation
   * statements. The client MAY replace the authenticator-generated attestation statements with
   * attestation statements generated by an <a
   * href="https://www.w3.org/TR/2021/REC-webauthn-2-20210408/#anonymization-ca">Anonymization
   * CA</a>, in order to protect the user’s privacy, or to assist Relying Parties with attestation
   * verification in a heterogeneous ecosystem.
   *
   * <p>Note: There is no guarantee that the Relying Party will obtain a verifiable attestation
   * statement in this case. For example, in the case that the authenticator employs self
   * attestation.
   */
  INDIRECT("indirect"),

  /**
   * Indicates that the Relying Party wants to receive the attestation statement as generated by the
   * authenticator.
   */
  DIRECT("direct"),

  /**
   * This value indicates that the Relying Party wants to receive an attestation statement that may
   * include uniquely identifying information. This is intended for controlled deployments within an
   * enterprise where the organization wishes to tie registrations to specific authenticators. User
   * agents MUST NOT provide such an attestation unless the user agent or authenticator
   * configuration permits it for the requested RP ID.
   *
   * <p>If permitted, the user agent SHOULD signal to the authenticator (at invocation time) that
   * enterprise attestation is requested, and convey the resulting AAGUID and attestation statement,
   * unaltered, to the Relying Party.
   */
  ENTERPRISE("enterprise");

  @JsonValue @Getter @NonNull private final String value;

  private static Optional<AttestationConveyancePreference> fromString(@NonNull String value) {
    return Stream.of(values()).filter(v -> v.value.equals(value)).findAny();
  }

  @JsonCreator
  private static AttestationConveyancePreference fromJsonString(@NonNull String value) {
    return fromString(value)
        .orElseThrow(
            () ->
                new IllegalArgumentException(
                    String.format(
                        "Unknown %s value: %s",
                        AttestationConveyancePreference.class.getSimpleName(), value)));
  }
}
