/**
 * Copyright (C) 2014-2017 Xavier Witdouck
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.zavtech.morpheus.viz.js;

import java.util.function.Consumer;

/**
 * A class that provides a basic API to programmatically generate simple Javascript code
 *
 * @author Xavier Witdouck
 *
 * <p><strong>This is open source software released under the <a href="http://www.apache.org/licenses/LICENSE-2.0">Apache 2.0 License</a></strong></p>
 */
public class JsCode {

    private String indent = "";
    private StringBuilder code = new StringBuilder();


    /**
     * Constructor
     */
    public JsCode() {
        super();
    }

    /**
     * Returns a Javascript string generated by the consumer logic
     * @param consumer  the consumer to write Javascript programmatically
     * @return          the Javascript string
     */
    public static String create(Consumer<JsCode> consumer) {
        final JsCode js = new JsCode();
        consumer.accept(js);
        return js.toString();
    }


    /**
     * Deletes the entire code buffer
     * @return  this writer
     */
    public JsCode clear() {
        this.code.delete(0, code.length());
        return this;
    }


    /**
     * Writes formatted code string to the output buffer
     * @param code  the formatted code String (see String.format() for details)
     * @param args  the arguments for the formatted code string
     * @return      this writer
     */
    public JsCode write(String code, Object... args) {
        this.code.append(String.format(code, args));
        return this;
    }


    /**
     * Starts a new line and indents the new line based on current indentation
     * @return      this code reference
     */
    public JsCode newLine() {
        return newLine(1);
    }

    /**
     * Starts a new line and indents the new line based on current indentation
     * @return      this code reference
     */
    public JsCode newLine(int count) {
        for (int i=0; i<count; ++i) {
            this.code.append("\n");
        }
        this.code.append(indent);
        return this;
    }


    /**
     * Increases the indentation for the current line by a certain number of spaces
     * @param count     the number of spaces to indent by
     * @return          this writer
     */
    public JsCode indent(int count) {
        final StringBuilder indentation = new StringBuilder(indent);
        for (int i=0; i<count; ++i) {
            indentation.append(" ");
            code.append(" ");
        }
        this.indent = indentation.toString();
        return this;
    }


    /**
     * Removes spaces from the current indentation
     * @param count the number of spaces to remove
     * @return      this code refernce
     */
    public JsCode unident(int count) {
        this.indent = indent.substring(0, indent.length()-count);
        return this;
    }


    /**
     * Writes a new object to the code buffer based on the consumer
     * @param consumer  the consumer with the object write instructions
     * @return          this writer
     */
    public JsCode newObject(Consumer<JsObject> consumer) {
        return newObject(false, consumer);
    }


    public JsCode newFunction(String name, Consumer<JsCode> consumer) {
        return newFunction(name, null, consumer);
    }


    public JsCode newFunction(String name, String args, Consumer<JsCode> consumer) {
        this.newLine();
        this.write("function %s(", name);
        this.write(args != null ? args : "");
        this.write(") {");
        this.newLine();
        this.indent(4);
        consumer.accept(this);
        this.unident(4);
        this.newLine();
        this.write("}");
        return this;
    }


    /**
     * Writes a new object, optionally on one line, based on the consumer
     * @param inline        if true, write the object definition in one line
     * @param consumer      the consumer with the object write instructions
     * @return              this writer
     */
    public JsCode newObject(boolean inline, Consumer<JsObject> consumer) {
        this.write("{");
        this.indent(4);
        this.newLine();
        consumer.accept(new JsObject(this, inline));
        this.unident(4);
        this.newLine();
        this.write("}");
        return this;
    }


    /**
     * Writes a new array to the code buffer based on the consumer
     * @param consumer  the consumer with array element instructions
     * @return          this writer
     */
    public JsCode newArray(Consumer<JsArray> consumer) {
        return newArray(false, consumer);
    }


    /**
     * Writes a new array, optionally in one line, based ont he consuner specified
     * @param inline        if true, write the array definition in one line
     * @param consumer      the consumer with the array write instructions
     * @return              this writer
     */
    public JsCode newArray(boolean inline, Consumer<JsArray> consumer) {
        this.code.append("[");
        if (!inline) {
            //this.indent(4);
        }
        consumer.accept(new JsArray(this, inline));
        if (!inline) {
            //this.unident(4);
            this.newLine();
            this.write("]");
        } else {
            this.write("]");
        }
        return this;
    }


    @Override()
    public String toString() {
        return code.toString();
    }


    public static void main(String[] args) {
        final JsCode code = new JsCode();
        code.write("x = ");
        code.newObject(object -> {
            object.newAttribute("zav", "witdouck");
            object.newAttribute("x", 22);
            object.newAttribute("kilo", "weigth");
            object.newAttribute("enabled", false);
            object.newObject("test", test -> {
                test.newAttribute("whatever", "again");
                test.newAttribute("second", "level");
                test.newArray("inline", true, a -> {
                    for (int i=0; i<10; ++i) {
                        a.append(i);
                    }
                });
                test.newArray("third", false, x -> {
                    x.appendObject(p -> {
                        p.newAttribute("AAA", 22323);
                        p.newAttribute("BBB", 95843);
                        p.newAttribute("CCC", false);
                        p.newAttribute("XXX", "YYYY");
                    });
                    x.appendObject(p -> {
                        p.newAttribute("AAA", 22323);
                        p.newAttribute("BBB", 95843);
                        p.newAttribute("CCC", false);
                        p.newAttribute("XXX", "YYYY");
                    });
                    x.appendObject(p -> {
                        p.newAttribute("AAA", 22323);
                        p.newAttribute("BBB", 95843);
                        p.newAttribute("CCC", false);
                        p.newAttribute("XXX", "YYYY");
                    });
                });
            });
        });

        System.out.println(code.toString());

    }


}
