/*
 * Decompiled with CFR 0.152.
 */
package com.zaxxer.hikari;

import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariPoolMBean;
import com.zaxxer.hikari.javassist.HikariInstrumentationAgent;
import com.zaxxer.hikari.proxy.IHikariConnectionProxy;
import com.zaxxer.hikari.proxy.JavassistProxyFactoryFactory;
import com.zaxxer.hikari.util.ClassLoaderUtils;
import com.zaxxer.hikari.util.PropertyBeanSetter;
import java.lang.management.ManagementFactory;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class HikariPool
implements HikariPoolMBean {
    private static final Logger LOGGER = LoggerFactory.getLogger(HikariPool.class);
    private final HikariConfig configuration;
    private final LinkedTransferQueue<IHikariConnectionProxy> idleConnections;
    private final AtomicInteger totalConnections;
    private final AtomicInteger idleConnectionCount;
    private final DataSource dataSource;
    private final long leakDetectionThreshold;
    private final boolean jdbc4ConnectionTest;
    private final boolean delegationProxies;
    private final Timer houseKeepingTimer;

    HikariPool(HikariConfig configuration) {
        configuration.validate();
        this.configuration = configuration;
        this.totalConnections = new AtomicInteger();
        this.idleConnectionCount = new AtomicInteger();
        this.idleConnections = new LinkedTransferQueue();
        this.jdbc4ConnectionTest = configuration.isJdbc4ConnectionTest();
        this.leakDetectionThreshold = configuration.getLeakDetectionThreshold();
        try {
            Class<?> clazz = ClassLoaderUtils.loadClass(configuration.getDataSourceClassName());
            this.dataSource = (DataSource)clazz.newInstance();
            PropertyBeanSetter.setTargetFromProperties(this.dataSource, configuration.getDataSourceProperties());
            HikariInstrumentationAgent instrumentationAgent = new HikariInstrumentationAgent(this.dataSource);
            boolean bl = this.delegationProxies = !instrumentationAgent.loadTransformerAgent();
            if (this.delegationProxies) {
                LOGGER.info("Falling back to Javassist delegate-based proxies.");
            }
        }
        catch (Exception e) {
            throw new RuntimeException("Could not create datasource class: " + configuration.getDataSourceClassName(), e);
        }
        this.registerMBean();
        this.houseKeepingTimer = new Timer("Hikari Housekeeping Timer", true);
        long idleTimeout = configuration.getIdleTimeout();
        if (idleTimeout > 0L || configuration.getMaxLifetime() > 0L) {
            this.houseKeepingTimer.scheduleAtFixedRate((TimerTask)new HouseKeeper(), TimeUnit.SECONDS.toMillis(30L), TimeUnit.SECONDS.toMillis(30L));
        }
        this.fillPool();
    }

    Connection getConnection() throws SQLException {
        try {
            long timeout = this.configuration.getConnectionTimeout();
            long start = System.currentTimeMillis();
            do {
                IHikariConnectionProxy connectionProxy;
                if (this.idleConnectionCount.get() == 0) {
                    this.addConnections();
                }
                if ((connectionProxy = this.idleConnections.poll(timeout, TimeUnit.MILLISECONDS)) == null) {
                    LOGGER.error("Timeout of {}ms encountered waiting for connection", (Object)this.configuration.getConnectionTimeout());
                    throw new SQLException("Timeout of encountered waiting for connection");
                }
                this.idleConnectionCount.decrementAndGet();
                long maxLifetime = this.configuration.getMaxLifetime();
                if (maxLifetime > 0L && start - connectionProxy.getCreationTime() > maxLifetime) {
                    this.closeConnection(connectionProxy);
                    timeout -= System.currentTimeMillis() - start;
                    continue;
                }
                connectionProxy.unclose();
                Connection connection = (Connection)((Object)connectionProxy);
                if (!this.isConnectionAlive(connection, timeout)) {
                    this.closeConnection(connectionProxy);
                    timeout -= System.currentTimeMillis() - start;
                    continue;
                }
                if (this.leakDetectionThreshold > 0L) {
                    connectionProxy.captureStack(this.leakDetectionThreshold, this.houseKeepingTimer);
                }
                return connection;
            } while (timeout > 0L);
            throw new SQLException("Timeout of encountered waiting for connection");
        }
        catch (InterruptedException e) {
            return null;
        }
    }

    public void releaseConnection(IHikariConnectionProxy connectionProxy) {
        if (!connectionProxy.isBrokenConnection()) {
            connectionProxy.markLastAccess();
            this.idleConnectionCount.incrementAndGet();
            this.idleConnections.put(connectionProxy);
        } else {
            this.closeConnection(connectionProxy);
        }
    }

    public int getActiveConnections() {
        return Math.min(this.configuration.getMaximumPoolSize(), this.totalConnections.get() - this.idleConnectionCount.get());
    }

    public int getIdleConnections() {
        return this.idleConnectionCount.get();
    }

    public int getTotalConnections() {
        return this.totalConnections.get();
    }

    public int getThreadsAwaitingConnection() {
        return this.idleConnections.getWaitingConsumerCount();
    }

    public void closeIdleConnections() {
        IHikariConnectionProxy connectionProxy;
        int idleCount = this.idleConnectionCount.get();
        for (int i = 0; i < idleCount && (connectionProxy = this.idleConnections.poll()) != null; ++i) {
            this.idleConnectionCount.decrementAndGet();
            this.closeConnection(connectionProxy);
        }
    }

    private void fillPool() {
        int maxIters = this.configuration.getMinimumPoolSize() / this.configuration.getAcquireIncrement() + 1;
        while (this.totalConnections.get() < this.configuration.getMinimumPoolSize() && maxIters-- > 0) {
            this.addConnections();
        }
    }

    private synchronized void addConnections() {
        int max = this.configuration.getMaximumPoolSize();
        int increment = this.configuration.getAcquireIncrement();
        for (int i = 0; this.totalConnections.get() < max && i < increment; ++i) {
            this.addConnection();
        }
    }

    private void addConnection() {
        int retries = 0;
        block4: while (true) {
            try {
                while (true) {
                    Connection connection = this.dataSource.getConnection();
                    IHikariConnectionProxy proxyConnection = this.delegationProxies ? (IHikariConnectionProxy)((Object)JavassistProxyFactoryFactory.getProxyFactory().getProxyConnection(connection)) : (IHikariConnectionProxy)((Object)connection);
                    proxyConnection.setParentPool(this);
                    boolean alive = this.isConnectionAlive((Connection)((Object)proxyConnection), this.configuration.getConnectionTimeout());
                    if (alive) {
                        connection.setAutoCommit(this.configuration.isAutoCommit());
                        this.idleConnectionCount.incrementAndGet();
                        this.totalConnections.incrementAndGet();
                        this.idleConnections.add(proxyConnection);
                        break block4;
                    }
                    Thread.sleep(this.configuration.getAcquireRetryDelay());
                }
            }
            catch (Exception e) {
                if (retries++ > this.configuration.getAcquireRetries()) {
                    LOGGER.error("Maximum connection creation retries exceeded", (Throwable)e);
                    break;
                }
                try {
                    Thread.sleep(this.configuration.getAcquireRetryDelay());
                }
                catch (InterruptedException e1) {
                    break;
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean isConnectionAlive(Connection connection, long timeoutMs) {
        if (timeoutMs < 500L) {
            timeoutMs = 500L;
        }
        try {
            if (this.jdbc4ConnectionTest) {
                return connection.isValid((int)timeoutMs * 1000);
            }
            Statement statement = connection.createStatement();
            try {
                statement.executeQuery(this.configuration.getConnectionTestQuery());
            }
            finally {
                statement.close();
            }
            return true;
        }
        catch (SQLException e) {
            LOGGER.error("Exception during keep alive check.  Connection must be dead.");
            return false;
        }
    }

    private void closeConnection(IHikariConnectionProxy connectionProxy) {
        try {
            this.totalConnections.decrementAndGet();
            connectionProxy.__close();
        }
        catch (SQLException e) {
            return;
        }
    }

    private void registerMBean() {
        try {
            MBeanServer mBeanServer = ManagementFactory.getPlatformMBeanServer();
            ObjectName poolConfigName = new ObjectName("com.zaxxer.hikari:type=PoolConfig (" + this.configuration.getPoolName() + ")");
            ObjectName poolName = new ObjectName("com.zaxxer.hikari:type=Pool (" + this.configuration.getPoolName() + ")");
            if (!mBeanServer.isRegistered(poolConfigName)) {
                mBeanServer.registerMBean(this.configuration, poolConfigName);
                mBeanServer.registerMBean(this, poolName);
            } else {
                LOGGER.error("You cannot use the same HikariConfig for separate pool instances.");
            }
        }
        catch (Exception e) {
            LOGGER.warn("Unable to register management beans.", (Throwable)e);
        }
    }

    private class HouseKeeper
    extends TimerTask {
        private HouseKeeper() {
        }

        public void run() {
            IHikariConnectionProxy connectionProxy;
            HikariPool.this.houseKeepingTimer.purge();
            long now = System.currentTimeMillis();
            long idleTimeout = HikariPool.this.configuration.getIdleTimeout();
            long maxLifetime = HikariPool.this.configuration.getMaxLifetime();
            int idleCount = HikariPool.this.idleConnectionCount.get();
            for (int i = 0; i < idleCount && (connectionProxy = (IHikariConnectionProxy)HikariPool.this.idleConnections.poll()) != null; ++i) {
                HikariPool.this.idleConnectionCount.decrementAndGet();
                if (idleTimeout > 0L && now > connectionProxy.getLastAccess() + idleTimeout || maxLifetime > 0L && now > connectionProxy.getCreationTime() + maxLifetime) {
                    HikariPool.this.closeConnection(connectionProxy);
                    continue;
                }
                HikariPool.this.idleConnectionCount.incrementAndGet();
                HikariPool.this.idleConnections.add(connectionProxy);
            }
            HikariPool.this.addConnections();
        }
    }
}

