/*
 * Decompiled with CFR 0.152.
 */
package com.zaxxer.hikari;

import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariMBeanElf;
import com.zaxxer.hikari.HikariPoolMBean;
import com.zaxxer.hikari.IConnectionCustomizer;
import com.zaxxer.hikari.proxy.IHikariConnectionProxy;
import com.zaxxer.hikari.proxy.ProxyFactory;
import com.zaxxer.hikari.util.ConcurrentBag;
import com.zaxxer.hikari.util.PropertyBeanSetter;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class HikariPool
implements HikariPoolMBean,
ConcurrentBag.IBagStateListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(HikariPool.class);
    final DataSource dataSource;
    private final IConnectionCustomizer connectionCustomizer;
    private final HikariConfig configuration;
    private final ConcurrentBag<IHikariConnectionProxy> idleConnectionBag;
    private final Timer houseKeepingTimer;
    private final long leakDetectionThreshold;
    private final AtomicInteger totalConnections;
    private final boolean isAutoCommit;
    private final boolean jdbc4ConnectionTest;
    private final boolean isRegisteredMbeans;
    private final String catalog;
    private int transactionIsolation;
    private volatile boolean shutdown;
    private boolean debug;

    HikariPool(HikariConfig configuration) {
        configuration.validate();
        this.configuration = configuration;
        this.totalConnections = new AtomicInteger();
        this.idleConnectionBag = new ConcurrentBag();
        this.idleConnectionBag.addBagStateListener(this);
        this.debug = LOGGER.isDebugEnabled();
        this.catalog = configuration.getCatalog();
        this.connectionCustomizer = configuration.getConnectionCustomizer();
        this.isAutoCommit = configuration.isAutoCommit();
        this.isRegisteredMbeans = configuration.isRegisterMbeans();
        this.jdbc4ConnectionTest = configuration.isJdbc4ConnectionTest();
        this.leakDetectionThreshold = configuration.getLeakDetectionThreshold();
        this.transactionIsolation = configuration.getTransactionIsolation();
        if (configuration.getDataSource() == null) {
            String dsClassName = configuration.getDataSourceClassName();
            try {
                Class<?> clazz = this.getClass().getClassLoader().loadClass(dsClassName);
                this.dataSource = (DataSource)clazz.newInstance();
                PropertyBeanSetter.setTargetFromProperties(this.dataSource, configuration.getDataSourceProperties());
            }
            catch (Exception e) {
                throw new RuntimeException("Could not create datasource instance: " + dsClassName, e);
            }
        } else {
            this.dataSource = configuration.getDataSource();
        }
        if (this.isRegisteredMbeans) {
            HikariMBeanElf.registerMBeans(configuration, this);
        }
        this.houseKeepingTimer = new Timer("Hikari Housekeeping Timer", true);
        this.fillPool();
        long idleTimeout = configuration.getIdleTimeout();
        if (idleTimeout > 0L || configuration.getMaxLifetime() > 0L) {
            long delayPeriod = Long.getLong("com.zaxxer.hikari.housekeeping.period", TimeUnit.SECONDS.toMillis(30L));
            this.houseKeepingTimer.scheduleAtFixedRate((TimerTask)new HouseKeeper(), delayPeriod, delayPeriod);
        }
    }

    Connection getConnection() throws SQLException {
        if (this.shutdown) {
            throw new SQLException("Pool has been shutdown");
        }
        try {
            IHikariConnectionProxy connectionProxy;
            long timeout = this.configuration.getConnectionTimeout();
            long start = System.currentTimeMillis();
            while ((connectionProxy = this.idleConnectionBag.borrow(timeout, TimeUnit.MILLISECONDS)) != null) {
                connectionProxy.unclose();
                if (System.currentTimeMillis() - connectionProxy.getLastAccess() <= 1000L || this.isConnectionAlive(connectionProxy, timeout)) {
                    if (this.leakDetectionThreshold > 0L) {
                        connectionProxy.captureStack(this.leakDetectionThreshold, this.houseKeepingTimer);
                    }
                    return connectionProxy;
                }
                this.closeConnection(connectionProxy);
                if ((timeout -= System.currentTimeMillis() - start) > 0L) continue;
            }
            this.logPoolState(new String[0]);
            String msg = String.format("Timeout of %dms encountered waiting for connection.", this.configuration.getConnectionTimeout());
            LOGGER.error(msg);
            this.logPoolState("Timeout failure ");
            throw new SQLException(msg);
        }
        catch (InterruptedException e) {
            return null;
        }
    }

    public void releaseConnection(IHikariConnectionProxy connectionProxy) {
        if (!connectionProxy.isBrokenConnection() && !this.shutdown) {
            this.idleConnectionBag.requite(connectionProxy);
        } else {
            LOGGER.debug("Connection returned to pool is broken, or the pool is shutting down.  Closing connection.");
            this.closeConnection(connectionProxy);
        }
    }

    public String toString() {
        return this.configuration.getPoolName();
    }

    void shutdown() {
        LOGGER.info("HikariCP pool " + this.configuration.getPoolName() + " is being shutdown.");
        this.logPoolState("State at shutdown ");
        this.shutdown = true;
        this.houseKeepingTimer.cancel();
        this.closeIdleConnections();
        if (this.isRegisteredMbeans) {
            HikariMBeanElf.unregisterMBeans(this.configuration, this);
        }
    }

    @Override
    public void bagIsEmpty() {
        this.addConnections(AddConnectionStrategy.ONLY_IF_EMPTY);
    }

    @Override
    public int getActiveConnections() {
        return Math.min(this.configuration.getMaximumPoolSize(), this.totalConnections.get() - this.getIdleConnections());
    }

    @Override
    public int getIdleConnections() {
        return this.idleConnectionBag.values(0).size();
    }

    @Override
    public int getTotalConnections() {
        return this.totalConnections.get();
    }

    @Override
    public int getThreadsAwaitingConnection() {
        return this.idleConnectionBag.getPendingQueue();
    }

    @Override
    public void closeIdleConnections() {
        List<IHikariConnectionProxy> list = this.idleConnectionBag.values(0);
        for (IHikariConnectionProxy connectionProxy : list) {
            if (!this.idleConnectionBag.reserve(connectionProxy)) continue;
            this.closeConnection(connectionProxy);
        }
    }

    private void fillPool() {
        int maxIters = this.configuration.getMinimumPoolSize() * this.configuration.getAcquireRetries();
        while (maxIters-- > 0 && this.totalConnections.get() < this.configuration.getMinimumPoolSize()) {
            int beforeCount = this.totalConnections.get();
            this.addConnection();
            if (!this.configuration.isInitializationFailFast() || beforeCount != this.totalConnections.get()) continue;
            throw new RuntimeException("Fail-fast during pool initialization");
        }
        this.logPoolState("Initial fill ");
    }

    private void addConnections(AddConnectionStrategy strategy) {
        int maxIterations = this.configuration.getAcquireIncrement();
        switch (strategy) {
            case ONLY_IF_EMPTY: {
                while (maxIterations-- > 0) {
                    this.addConnection();
                }
                break;
            }
            case MAINTAIN_MINIMUM: {
                int min = this.configuration.getMinimumPoolSize();
                while (maxIterations-- > 0 && this.totalConnections.get() < min) {
                    this.addConnection();
                }
                break;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addConnection() {
        int retries = 0;
        while (!this.shutdown) {
            try {
                if (this.totalConnections.incrementAndGet() > this.configuration.getMaximumPoolSize()) {
                    this.totalConnections.decrementAndGet();
                    break;
                }
                Connection connection = this.dataSource.getConnection();
                if (this.transactionIsolation < 0) {
                    this.transactionIsolation = connection.getTransactionIsolation();
                }
                if (this.connectionCustomizer != null) {
                    this.connectionCustomizer.customize(connection);
                }
                IHikariConnectionProxy proxyConnection = ProxyFactory.getProxyConnection(this, connection, this.transactionIsolation, this.isAutoCommit, this.catalog);
                String initSql = this.configuration.getConnectionInitSql();
                if (initSql != null && initSql.length() > 0) {
                    connection.setAutoCommit(true);
                    Statement statement = connection.createStatement();
                    try {
                        statement.execute(initSql);
                    }
                    finally {
                        statement.close();
                    }
                }
                proxyConnection.resetConnectionState();
                this.idleConnectionBag.add(proxyConnection);
            }
            catch (Exception e) {
                if (retries++ > this.configuration.getAcquireRetries()) {
                    if (this.debug) {
                        LOGGER.error("Maximum connection creation retries exceeded: {}", (Object)e.getMessage(), (Object)e);
                    } else {
                        LOGGER.error("Maximum connection creation retries exceeded: {}", (Object)e.getMessage());
                    }
                    this.totalConnections.decrementAndGet();
                    break;
                }
                try {
                    Thread.sleep(this.configuration.getAcquireRetryDelay());
                    continue;
                }
                catch (InterruptedException e1) {
                    this.totalConnections.decrementAndGet();
                }
            }
            break;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean isConnectionAlive(IHikariConnectionProxy connection, long timeoutMs) {
        try {
            try {
                if (timeoutMs < 1000L) {
                    timeoutMs = 1000L;
                }
                if (this.jdbc4ConnectionTest) {
                    boolean bl = connection.isValid((int)TimeUnit.MILLISECONDS.toSeconds(timeoutMs));
                    return bl;
                }
                Statement statement = connection.createStatement();
                try {
                    statement.setQueryTimeout((int)TimeUnit.MILLISECONDS.toSeconds(timeoutMs));
                    statement.executeQuery(this.configuration.getConnectionTestQuery());
                    return true;
                }
                finally {
                    statement.close();
                }
            }
            finally {
                if (!this.isAutoCommit) {
                    connection.commit();
                }
            }
        }
        catch (SQLException e) {
            LOGGER.warn("Exception during keep alive check, that means the connection must be dead.", (Throwable)e);
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void closeConnection(IHikariConnectionProxy connectionProxy) {
        try {
            this.totalConnections.decrementAndGet();
            connectionProxy.realClose();
        }
        catch (SQLException e) {
            return;
        }
        finally {
            this.idleConnectionBag.remove(connectionProxy);
        }
    }

    private void logPoolState(String ... prefix) {
        int total = this.totalConnections.get();
        int idle = this.getIdleConnections();
        LOGGER.debug("{}Pool stats (total={}, inUse={}, avail={}, waiting={})", new Object[]{prefix.length > 0 ? prefix[0] : "", total, total - idle, idle, this.getThreadsAwaitingConnection()});
    }

    private static enum AddConnectionStrategy {
        ONLY_IF_EMPTY,
        MAINTAIN_MINIMUM;

    }

    private class HouseKeeper
    extends TimerTask {
        private HouseKeeper() {
        }

        @Override
        public void run() {
            HikariPool.this.debug = LOGGER.isDebugEnabled();
            HikariPool.this.houseKeepingTimer.purge();
            HikariPool.this.logPoolState(new String[]{"Before pool cleanup "});
            long now = System.currentTimeMillis();
            long idleTimeout = HikariPool.this.configuration.getIdleTimeout();
            long maxLifetime = HikariPool.this.configuration.getMaxLifetime();
            for (IHikariConnectionProxy connectionProxy : HikariPool.this.idleConnectionBag.values(0)) {
                if (!HikariPool.this.idleConnectionBag.reserve(connectionProxy)) continue;
                if (idleTimeout > 0L && now > connectionProxy.getLastAccess() + idleTimeout || maxLifetime > 0L && now > connectionProxy.getCreationTime() + maxLifetime) {
                    HikariPool.this.closeConnection(connectionProxy);
                    continue;
                }
                HikariPool.this.idleConnectionBag.unreserve(connectionProxy);
            }
            HikariPool.this.addConnections(AddConnectionStrategy.MAINTAIN_MINIMUM);
            HikariPool.this.logPoolState(new String[]{"After pool cleanup "});
        }
    }
}

