/*
 * Copyright (C) 2013 Brett Wooldridge
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.zaxxer.hikari.util;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.zaxxer.hikari.HikariConfig;

/**
 * A class that reflectively sets bean properties on a target object.
 *
 * @author Brett Wooldridge
 */
public final class PropertyBeanSetter
{
    private static final Logger LOGGER = LoggerFactory.getLogger(PropertyBeanSetter.class);

    public static void setTargetFromProperties(Object target, Properties properties)
    {
        if (target == null || properties == null)
        {
            return;
        }

        for (Object propKey : properties.keySet())
        {
            String propName = propKey.toString();
            Object propValue = properties.get(propKey);

            if (target instanceof HikariConfig && propName.startsWith("dataSource."))
            {
                HikariConfig config = (HikariConfig) target;
                config.addDataSourceProperty(propName.substring("dataSource.".length()), propValue);
            }
            else
            {
                setProperty(target, propName, propValue);
            }
        }
    }

    /**
     * Get the bean-style property names for the specified object.
     *
     * @param targetClass the target object
     * @return a set of property names
     */
    public static Set<String> getPropertyNames(Class<?> targetClass)
    {
        HashSet<String> set = new HashSet<String>();
        try
        {
            BeanInfo info = Introspector.getBeanInfo(targetClass);
            for (PropertyDescriptor descr : info.getPropertyDescriptors())
            {
                set.add(descr.getName());
            }

            return set;
        }
        catch (IntrospectionException e)
        {
            throw new RuntimeException(e);
        }
    }

    private static void setProperty(Object target, String propName, Object propValue)
    {
        String capitalized = "set" + propName.substring(0, 1).toUpperCase() + propName.substring(1);
        PropertyDescriptor propertyDescriptor;
        try
        {
            propertyDescriptor = new PropertyDescriptor(propName, target.getClass(), null, capitalized);
        }
        catch (IntrospectionException e)
        {
            capitalized = "set" + propName.toUpperCase();
            try
            {
                propertyDescriptor = new PropertyDescriptor(propName, target.getClass(), null, capitalized);
            }
            catch (IntrospectionException e1)
            {
                LOGGER.error("Property {} is does not exist on target class {}", propName, target.getClass());
                throw new RuntimeException(e);
            }
        }

        try
        {
            Method writeMethod = propertyDescriptor.getWriteMethod();
            Class<?> paramClass = writeMethod.getParameterTypes()[0];
            if (paramClass == int.class)
            {
                writeMethod.invoke(target, Integer.parseInt(propValue.toString()));
            }
            else if (paramClass == long.class)
            {
                writeMethod.invoke(target, Long.parseLong(propValue.toString()));
            }
            else if (paramClass == boolean.class)
            {
                writeMethod.invoke(target, Boolean.parseBoolean(propValue.toString()));
            }
            else if (paramClass == String.class)
            {
                writeMethod.invoke(target, propValue.toString());
            }
            else
            {
                writeMethod.invoke(target, propValue);
            }
        }
        catch (Exception e)
        {
            LOGGER.error("Exception setting property {} on target class {}", propName, target.getClass(), e);
            throw new RuntimeException(e);
        }
    }
}
