/*
 * Copyright (C) 2013 Brett Wooldridge
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.zaxxer.hikari.proxy;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Wrapper;

/**
 * This is the proxy class for java.sql.Statement.
 *
 * @author Brett Wooldridge
 */
public abstract class StatementProxy implements Statement
{
   protected final ConnectionProxy connection;
   protected final Statement delegate;

   private boolean isClosed;

   protected StatementProxy(ConnectionProxy connection, Statement statement)
   {
      this.connection = connection;
      this.delegate = statement;
   }

   protected final SQLException checkException(SQLException e)
   {
      return connection.checkException(e);
   }

   /** {@inheritDoc} */
   @Override
   public final String toString()
   {
      final String delegateToString = delegate.toString();
      return new StringBuilder(64 + delegateToString.length())
         .append(this.getClass().getSimpleName()).append('@').append(System.identityHashCode(this))
         .append(" wrapping ")
         .append(delegateToString).toString();
   }

   // **********************************************************************
   //                 Overridden java.sql.Statement Methods
   // **********************************************************************

   /** {@inheritDoc} */
   @Override
   public final void close() throws SQLException
   {
      if (isClosed) {
         return;
      }

      isClosed = true;
      connection.untrackStatement(delegate);

      try {
         delegate.close();
      }
      catch (SQLException e) {
         throw connection.checkException(e);
      }
   }

   /** {@inheritDoc} */
   @Override
   public Connection getConnection() throws SQLException
   {
      return connection;
   }

   /** {@inheritDoc} */
   @Override
   public boolean execute(String sql) throws SQLException
   {
      connection.markCommitStateDirty();
      return delegate.execute(sql);
   }

   /** {@inheritDoc} */
   @Override
   public boolean execute(String sql, int autoGeneratedKeys) throws SQLException
   {
      connection.markCommitStateDirty();
      return delegate.execute(sql, autoGeneratedKeys);
   }

   /** {@inheritDoc} */
   @Override
   public ResultSet executeQuery(String sql) throws SQLException
   {
      connection.markCommitStateDirty();
      ResultSet resultSet = delegate.executeQuery(sql);
      return ProxyFactory.getProxyResultSet(connection, resultSet); 
   }

   /** {@inheritDoc} */
   @Override
   public int executeUpdate(String sql) throws SQLException
   {
      connection.markCommitStateDirty();
      return delegate.executeUpdate(sql);
   }

   /** {@inheritDoc} */
   @Override
   public int[] executeBatch() throws SQLException
   {
      connection.markCommitStateDirty();
      return delegate.executeBatch();
   }

   /** {@inheritDoc} */
   @Override
   public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException
   {
      connection.markCommitStateDirty();
      return delegate.executeUpdate(sql, autoGeneratedKeys);
   }

   /** {@inheritDoc} */
   @Override
   public int executeUpdate(String sql, int[] columnIndexes) throws SQLException
   {
      connection.markCommitStateDirty();
      return delegate.executeUpdate(sql, columnIndexes);
   }

   /** {@inheritDoc} */
   @Override
   public int executeUpdate(String sql, String[] columnNames) throws SQLException
   {
      connection.markCommitStateDirty();
      return delegate.executeUpdate(sql, columnNames);
   }

   /** {@inheritDoc} */
   @Override
   public boolean execute(String sql, int[] columnIndexes) throws SQLException
   {
      connection.markCommitStateDirty();
      return delegate.execute(sql, columnIndexes);
   }

   /** {@inheritDoc} */
   @Override
   public boolean execute(String sql, String[] columnNames) throws SQLException
   {
      connection.markCommitStateDirty();
      return delegate.execute(sql, columnNames);
   }

   /** {@inheritDoc} */
   @Override
   public long[] executeLargeBatch() throws SQLException
   {
      connection.markCommitStateDirty();
      return delegate.executeLargeBatch();
   }

   /** {@inheritDoc} */
   @Override
   public long executeLargeUpdate(String sql) throws SQLException
   {
      connection.markCommitStateDirty();
      return delegate.executeLargeUpdate(sql);
   }

   /** {@inheritDoc} */
   @Override
   public long executeLargeUpdate(String sql, int autoGeneratedKeys) throws SQLException
   {
      connection.markCommitStateDirty();
      return delegate.executeLargeUpdate(sql, autoGeneratedKeys);
   }

   /** {@inheritDoc} */
   @Override
   public long executeLargeUpdate(String sql, int[] columnIndexes) throws SQLException
   {
      connection.markCommitStateDirty();
      return delegate.executeLargeUpdate(sql, columnIndexes);
   }

   /** {@inheritDoc} */
   @Override
   public long executeLargeUpdate(String sql, String[] columnNames) throws SQLException
   {
      connection.markCommitStateDirty();
      return delegate.executeLargeUpdate(sql, columnNames);
   }

   /** {@inheritDoc} */
   @Override
   public ResultSet getResultSet() throws SQLException {
      final ResultSet resultSet = delegate.getResultSet();
      if (resultSet != null) {
         return ProxyFactory.getProxyResultSet(connection, resultSet);
      }
      return null;
   }

   /** {@inheritDoc} */
   @Override
   @SuppressWarnings("unchecked")
   public final <T> T unwrap(Class<T> iface) throws SQLException
   {
      if (iface.isInstance(delegate)) {
         return (T) delegate;
      }
      else if (delegate instanceof Wrapper) {
          return (T) delegate.unwrap(iface);
      }

      throw new SQLException("Wrapped statement is not an instance of " + iface);
   }
}
