/*
 * Decompiled with CFR 0.152.
 */
package com.zendesk.maxwell.schema;

import com.zendesk.maxwell.CaseSensitivity;
import com.zendesk.maxwell.schema.Database;
import com.zendesk.maxwell.schema.Schema;
import com.zendesk.maxwell.schema.Table;
import com.zendesk.maxwell.schema.columndef.ColumnDef;
import com.zendesk.maxwell.schema.columndef.JsonColumnDef;
import com.zendesk.maxwell.schema.columndef.StringColumnDef;
import com.zendesk.maxwell.schema.ddl.InvalidSchemaError;
import com.zendesk.maxwell.util.Sql;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SchemaCapturer
implements AutoCloseable {
    private final Connection connection;
    static final Logger LOGGER = LoggerFactory.getLogger(SchemaCapturer.class);
    public static final HashSet<String> IGNORED_DATABASES = new HashSet<String>(Arrays.asList("performance_schema", "information_schema"));
    private final Set<String> includeDatabases;
    private final Set<String> includeTables;
    private final CaseSensitivity sensitivity;
    private final boolean isMySQLAtLeast56;
    private final PreparedStatement tablePreparedStatement;
    private final PreparedStatement columnPreparedStatement;
    private final PreparedStatement pkPreparedStatement;
    static final String MARIA_VERSION_REGEX = "[\\d\\.]+-(\\d+)\\.(\\d+)";

    public SchemaCapturer(Connection c, CaseSensitivity sensitivity) throws SQLException {
        this(c, sensitivity, Collections.emptySet(), Collections.emptySet());
    }

    SchemaCapturer(Connection c, CaseSensitivity sensitivity, Set<String> includeDatabases, Set<String> includeTables) throws SQLException {
        this.includeDatabases = includeDatabases;
        this.includeTables = includeTables;
        this.connection = c;
        this.sensitivity = sensitivity;
        this.isMySQLAtLeast56 = this.isMySQLAtLeast56();
        String dateTimePrecision = "";
        if (this.isMySQLAtLeast56) {
            dateTimePrecision = "DATETIME_PRECISION, ";
        }
        Object tblSql = "SELECT TABLES.TABLE_NAME, CCSA.CHARACTER_SET_NAME FROM INFORMATION_SCHEMA.TABLES LEFT JOIN information_schema.COLLATION_CHARACTER_SET_APPLICABILITY AS CCSA ON TABLES.TABLE_COLLATION = CCSA.COLLATION_NAME WHERE TABLES.TABLE_SCHEMA = ? ";
        if (!includeTables.isEmpty()) {
            tblSql = (String)tblSql + " AND TABLES.TABLE_NAME IN " + Sql.inListSQL(includeTables.size());
        }
        this.tablePreparedStatement = this.connection.prepareStatement((String)tblSql);
        String columnSql = "SELECT TABLE_NAME,COLUMN_NAME, DATA_TYPE, CHARACTER_SET_NAME, ORDINAL_POSITION, COLUMN_TYPE, " + dateTimePrecision + "COLUMN_KEY FROM `information_schema`.`COLUMNS` WHERE TABLE_SCHEMA = ? ORDER BY TABLE_NAME, ORDINAL_POSITION";
        this.columnPreparedStatement = this.connection.prepareStatement(columnSql);
        String pkSQl = "SELECT TABLE_NAME, COLUMN_NAME, ORDINAL_POSITION FROM information_schema.KEY_COLUMN_USAGE WHERE CONSTRAINT_NAME = 'PRIMARY' AND TABLE_SCHEMA = ? ORDER BY TABLE_NAME, ORDINAL_POSITION";
        this.pkPreparedStatement = this.connection.prepareStatement(pkSQl);
    }

    public SchemaCapturer(Connection c, CaseSensitivity sensitivity, String dbName) throws SQLException {
        this(c, sensitivity, Collections.singleton(dbName), Collections.emptySet());
    }

    public SchemaCapturer(Connection c, CaseSensitivity sensitivity, String dbName, String tblName) throws SQLException {
        this(c, sensitivity, Collections.singleton(dbName), Collections.singleton(tblName));
    }

    public Schema capture() throws SQLException {
        LOGGER.debug("Capturing schemas...");
        ArrayList<Database> databases = new ArrayList<Database>();
        Object dbCaptureQuery = "SELECT SCHEMA_NAME, DEFAULT_CHARACTER_SET_NAME FROM INFORMATION_SCHEMA.SCHEMATA";
        if (this.includeDatabases.size() > 0) {
            dbCaptureQuery = (String)dbCaptureQuery + " WHERE SCHEMA_NAME IN " + Sql.inListSQL(this.includeDatabases.size());
        }
        dbCaptureQuery = (String)dbCaptureQuery + " ORDER BY SCHEMA_NAME";
        try (PreparedStatement statement = this.connection.prepareStatement((String)dbCaptureQuery);){
            Sql.prepareInList(statement, 1, this.includeDatabases);
            try (ResultSet rs = statement.executeQuery();){
                while (rs.next()) {
                    String dbName = rs.getString("SCHEMA_NAME");
                    String charset = rs.getString("DEFAULT_CHARACTER_SET_NAME");
                    if (IGNORED_DATABASES.contains(dbName)) continue;
                    Database db = new Database(dbName, charset);
                    databases.add(db);
                }
            }
        }
        int size = databases.size();
        LOGGER.debug("Starting schema capture of {} databases...", (Object)size);
        int counter = 1;
        for (Database db : databases) {
            LOGGER.debug("{}/{} Capturing {}...", new Object[]{counter, size, db.getName()});
            this.captureDatabase(db);
            ++counter;
        }
        LOGGER.debug("{} database schemas captured!", (Object)size);
        Schema s = new Schema(databases, this.captureDefaultCharset(), this.sensitivity);
        try {
            if (this.isMariaDB() && this.mariaSupportsJSON()) {
                this.detectMariaDBJSON(s);
            }
        }
        catch (InvalidSchemaError e) {
            e.printStackTrace();
        }
        return s;
    }

    private String captureDefaultCharset() throws SQLException {
        LOGGER.debug("Capturing Default Charset");
        try (Statement stmt = this.connection.createStatement();){
            String string;
            block12: {
                ResultSet rs = stmt.executeQuery("select @@character_set_server");
                try {
                    rs.next();
                    string = rs.getString("@@character_set_server");
                    if (rs == null) break block12;
                }
                catch (Throwable throwable) {
                    if (rs != null) {
                        try {
                            rs.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                rs.close();
            }
            return string;
        }
    }

    private void captureDatabase(Database db) throws SQLException {
        this.tablePreparedStatement.setString(1, db.getName());
        Sql.prepareInList(this.tablePreparedStatement, 2, this.includeTables);
        HashMap<String, Table> tables = new HashMap<String, Table>();
        try (ResultSet rs = this.tablePreparedStatement.executeQuery();){
            while (rs.next()) {
                String tableName = rs.getString("TABLE_NAME");
                String characterSetName = rs.getString("CHARACTER_SET_NAME");
                Table t = db.buildTable(tableName, characterSetName);
                tables.put(tableName, t);
            }
        }
        this.captureTables(db, tables);
    }

    private boolean isMySQLAtLeast56() throws SQLException {
        if (this.isMariaDB()) {
            return true;
        }
        DatabaseMetaData meta = this.connection.getMetaData();
        int major = meta.getDatabaseMajorVersion();
        int minor = meta.getDatabaseMinorVersion();
        return major == 5 && minor >= 6 || major > 5;
    }

    private boolean isMariaDB() throws SQLException {
        DatabaseMetaData meta = this.connection.getMetaData();
        return meta.getDatabaseProductVersion().toLowerCase().contains("maria");
    }

    private boolean mariaSupportsJSON() throws SQLException {
        DatabaseMetaData meta = this.connection.getMetaData();
        String versionString = meta.getDatabaseProductVersion();
        Pattern pattern = Pattern.compile(MARIA_VERSION_REGEX);
        Matcher m = pattern.matcher(versionString);
        if (m.find()) {
            int major = Integer.parseInt(m.group(1));
            int minor = Integer.parseInt(m.group(2));
            return major >= 10 && minor > 1;
        }
        return false;
    }

    private void captureTables(Database db, HashMap<String, Table> tables) throws SQLException {
        this.columnPreparedStatement.setString(1, db.getName());
        try (ResultSet r = this.columnPreparedStatement.executeQuery();){
            HashMap<String, Integer> pkIndexCounters = new HashMap<String, Integer>();
            for (String tableName : tables.keySet()) {
                pkIndexCounters.put(tableName, 0);
            }
            while (r.next()) {
                String tableName;
                String[] enumValues = null;
                tableName = r.getString("TABLE_NAME");
                if (!tables.containsKey(tableName)) continue;
                Table t = tables.get(tableName);
                String colName = r.getString("COLUMN_NAME");
                String colType = r.getString("DATA_TYPE");
                String colEnc = r.getString("CHARACTER_SET_NAME");
                short colPos = (short)(r.getInt("ORDINAL_POSITION") - 1);
                boolean colSigned = !r.getString("COLUMN_TYPE").matches(".* unsigned$");
                Long columnLength = null;
                if (this.isMySQLAtLeast56) {
                    columnLength = r.getLong("DATETIME_PRECISION");
                }
                if (r.getString("COLUMN_KEY").equals("PRI")) {
                    t.pkIndex = (Integer)pkIndexCounters.get(tableName);
                }
                if (colType.equals("enum") || colType.equals("set")) {
                    String expandedType = r.getString("COLUMN_TYPE");
                    enumValues = SchemaCapturer.extractEnumValues(expandedType);
                }
                t.addColumn(ColumnDef.build(colName, colEnc, colType, colPos, colSigned, enumValues, columnLength));
                pkIndexCounters.put(tableName, (Integer)pkIndexCounters.get(tableName) + 1);
            }
        }
        this.captureTablesPK(db, tables);
    }

    private void captureTablesPK(Database db, HashMap<String, Table> tables) throws SQLException {
        this.pkPreparedStatement.setString(1, db.getName());
        HashMap tablePKMap = new HashMap();
        try (ResultSet rs = this.pkPreparedStatement.executeQuery();){
            for (String tableName : tables.keySet()) {
                tablePKMap.put(tableName, new ArrayList());
            }
            while (rs.next()) {
                String tableName;
                int ordinalPosition = rs.getInt("ORDINAL_POSITION");
                tableName = rs.getString("TABLE_NAME");
                String columnName = rs.getString("COLUMN_NAME");
                ArrayList pkList = (ArrayList)tablePKMap.get(tableName);
                if (pkList == null) continue;
                pkList.add(ordinalPosition - 1, columnName);
            }
        }
        for (Map.Entry<String, Table> entry : tables.entrySet()) {
            String key = entry.getKey();
            Table table = entry.getValue();
            table.setPKList((List)tablePKMap.get(key));
        }
    }

    static String[] extractEnumValues(String expandedType) {
        Matcher matcher = Pattern.compile("(enum|set)\\((.*)\\)").matcher(expandedType);
        matcher.matches();
        Object enumValues = matcher.group(2);
        if (!((String)enumValues).endsWith(",")) {
            enumValues = (String)enumValues + ",";
        }
        String regex = "('.*?'),";
        Pattern pattern = Pattern.compile(regex);
        Matcher enumMatcher = pattern.matcher((CharSequence)enumValues);
        ArrayList<String> result = new ArrayList<String>();
        while (enumMatcher.find()) {
            String value = enumMatcher.group(0);
            if (value.startsWith("'")) {
                value = value.substring(1);
            }
            if (value.endsWith("',")) {
                value = value.substring(0, value.length() - 2);
            }
            result.add(value);
        }
        return result.toArray(new String[0]);
    }

    @Override
    public void close() throws SQLException {
        try (PreparedStatement p1 = this.tablePreparedStatement;
             PreparedStatement p2 = this.columnPreparedStatement;){
            PreparedStatement p3 = this.pkPreparedStatement;
            if (p3 != null) {
                p3.close();
            }
        }
    }

    private void detectMariaDBJSON(Schema schema) throws SQLException, InvalidSchemaError {
        String checkConstraintSQL = "SELECT CONSTRAINT_SCHEMA, TABLE_NAME, CONSTRAINT_NAME, CHECK_CLAUSE from INFORMATION_SCHEMA.CHECK_CONSTRAINTS where CHECK_CLAUSE LIKE 'json_valid(%)'";
        String regex = "json_valid\\(`(.*)`\\)";
        Pattern pattern = Pattern.compile(regex);
        try (PreparedStatement statement = this.connection.prepareStatement(checkConstraintSQL);
             ResultSet rs = statement.executeQuery();){
            while (rs.next()) {
                ColumnDef cd;
                short i;
                Table t;
                String checkClause = rs.getString("CHECK_CLAUSE");
                Matcher m = pattern.matcher(checkClause);
                if (!m.find()) continue;
                String column = m.group(1);
                Database d = schema.findDatabase(rs.getString("CONSTRAINT_SCHEMA"));
                if (d == null || (t = d.findTable(rs.getString("TABLE_NAME"))) == null || (i = t.findColumnIndex(column)) < 0 || !((cd = t.findColumn(i)) instanceof StringColumnDef)) continue;
                t.replaceColumn(i, JsonColumnDef.create(cd.getName(), "json", i));
            }
        }
    }
}

