/*
 * Decompiled with CFR 0.152.
 */
package de.bwaldvogel.mongo.backend;

import de.bwaldvogel.mongo.MongoBackend;
import de.bwaldvogel.mongo.MongoCollection;
import de.bwaldvogel.mongo.MongoDatabase;
import de.bwaldvogel.mongo.backend.Assert;
import de.bwaldvogel.mongo.backend.Index;
import de.bwaldvogel.mongo.backend.IndexKey;
import de.bwaldvogel.mongo.backend.LimitedList;
import de.bwaldvogel.mongo.backend.Utils;
import de.bwaldvogel.mongo.backend.aggregation.Aggregation;
import de.bwaldvogel.mongo.backend.aggregation.stage.AddFieldsStage;
import de.bwaldvogel.mongo.backend.aggregation.stage.GroupStage;
import de.bwaldvogel.mongo.backend.aggregation.stage.LimitStage;
import de.bwaldvogel.mongo.backend.aggregation.stage.LookupStage;
import de.bwaldvogel.mongo.backend.aggregation.stage.MatchStage;
import de.bwaldvogel.mongo.backend.aggregation.stage.OrderByStage;
import de.bwaldvogel.mongo.backend.aggregation.stage.ProjectStage;
import de.bwaldvogel.mongo.backend.aggregation.stage.ReplaceRootStage;
import de.bwaldvogel.mongo.backend.aggregation.stage.SkipStage;
import de.bwaldvogel.mongo.backend.aggregation.stage.UnwindStage;
import de.bwaldvogel.mongo.bson.Document;
import de.bwaldvogel.mongo.exception.MongoServerError;
import de.bwaldvogel.mongo.exception.MongoServerException;
import de.bwaldvogel.mongo.exception.MongoSilentServerException;
import de.bwaldvogel.mongo.exception.NoSuchCommandException;
import de.bwaldvogel.mongo.wire.message.MongoDelete;
import de.bwaldvogel.mongo.wire.message.MongoInsert;
import de.bwaldvogel.mongo.wire.message.MongoQuery;
import de.bwaldvogel.mongo.wire.message.MongoUpdate;
import io.netty.channel.Channel;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractMongoDatabase<P>
implements MongoDatabase {
    private static final String NAMESPACES_COLLECTION_NAME = "system.namespaces";
    private static final String INDEXES_COLLECTION_NAME = "system.indexes";
    private static final Logger log = LoggerFactory.getLogger(AbstractMongoDatabase.class);
    protected final String databaseName;
    private final MongoBackend backend;
    private final Map<String, MongoCollection<P>> collections = new ConcurrentHashMap<String, MongoCollection<P>>();
    private final AtomicReference<MongoCollection<P>> indexes = new AtomicReference();
    private final Map<Channel, List<Document>> lastResults = new ConcurrentHashMap<Channel, List<Document>>();
    private MongoCollection<P> namespaces;

    protected AbstractMongoDatabase(String databaseName, MongoBackend backend) {
        this.databaseName = databaseName;
        this.backend = backend;
    }

    protected void initializeNamespacesAndIndexes() {
        this.namespaces = this.openOrCreateCollection(NAMESPACES_COLLECTION_NAME, "name");
        this.collections.put(this.namespaces.getCollectionName(), this.namespaces);
        if (this.namespaces.count() > 0) {
            for (Document namespace : this.namespaces.queryAll()) {
                String name = namespace.get("name").toString();
                log.debug("opening {}", (Object)name);
                String collectionName = this.extractCollectionNameFromNamespace(name);
                MongoCollection<P> collection = this.openOrCreateCollection(collectionName, "_id");
                this.collections.put(collectionName, collection);
                log.debug("opened collection '{}'", (Object)collectionName);
            }
            MongoCollection<P> indexCollection = this.openOrCreateCollection(INDEXES_COLLECTION_NAME, null);
            this.collections.put(indexCollection.getCollectionName(), indexCollection);
            this.indexes.set(indexCollection);
            for (Document indexDescription : indexCollection.queryAll()) {
                this.openOrCreateIndex(indexDescription);
            }
        }
    }

    @Override
    public final String getDatabaseName() {
        return this.databaseName;
    }

    public String toString() {
        return this.getClass().getSimpleName() + "(" + this.getDatabaseName() + ")";
    }

    private Document commandDropDatabase() {
        this.backend.dropDatabase(this.getDatabaseName());
        Document response = new Document("dropped", this.getDatabaseName());
        Utils.markOkay(response);
        return response;
    }

    @Override
    public Document handleCommand(Channel channel, String command, Document query) {
        if (command.equalsIgnoreCase("getlasterror")) {
            return this.commandGetLastError(channel, command, query);
        }
        if (command.equalsIgnoreCase("getpreverror")) {
            return this.commandGetPrevError(channel);
        }
        if (command.equalsIgnoreCase("reseterror")) {
            return this.commandResetError(channel);
        }
        this.clearLastStatus(channel);
        if (command.equalsIgnoreCase("find")) {
            return this.commandFind(command, query);
        }
        if (command.equalsIgnoreCase("insert")) {
            return this.commandInsert(channel, command, query);
        }
        if (command.equalsIgnoreCase("update")) {
            return this.commandUpdate(channel, command, query);
        }
        if (command.equalsIgnoreCase("delete")) {
            return this.commandDelete(channel, command, query);
        }
        if (command.equalsIgnoreCase("create")) {
            return this.commandCreate(command, query);
        }
        if (command.equalsIgnoreCase("createIndexes")) {
            return this.commandCreateIndexes(query);
        }
        if (command.equalsIgnoreCase("count")) {
            return this.commandCount(command, query);
        }
        if (command.equalsIgnoreCase("aggregate")) {
            return this.commandAggregate(command, query);
        }
        if (command.equalsIgnoreCase("distinct")) {
            MongoCollection<P> collection = this.resolveCollection(command, query, true);
            return collection.handleDistinct(query);
        }
        if (command.equalsIgnoreCase("drop")) {
            return this.commandDrop(query);
        }
        if (command.equalsIgnoreCase("dropDatabase")) {
            return this.commandDropDatabase();
        }
        if (command.equalsIgnoreCase("dbstats")) {
            return this.commandDatabaseStats();
        }
        if (command.equalsIgnoreCase("collstats")) {
            MongoCollection<P> collection = this.resolveCollection(command, query, true);
            return collection.getStats();
        }
        if (command.equalsIgnoreCase("validate")) {
            MongoCollection<P> collection = this.resolveCollection(command, query, false);
            if (collection == null) {
                throw new MongoServerError(26, "NamespaceNotFound", "ns not found");
            }
            return collection.validate();
        }
        if (command.equalsIgnoreCase("findAndModify")) {
            String collectionName = query.get(command).toString();
            MongoCollection<P> collection = this.resolveOrCreateCollection(collectionName);
            return collection.findAndModify(query);
        }
        if (command.equalsIgnoreCase("listCollections")) {
            return this.listCollections();
        }
        if (command.equalsIgnoreCase("listIndexes")) {
            String collectionName = query.get(command).toString();
            return this.listIndexes(collectionName);
        }
        log.error("unknown query: {}", (Object)query);
        throw new NoSuchCommandException(command);
    }

    private Document listCollections() {
        ArrayList<Document> firstBatch = new ArrayList<Document>();
        for (Document collection : this.namespaces.queryAll()) {
            Document collectionDescription = new Document();
            Document collectionOptions = new Document();
            String namespace = (String)collection.get("name");
            if (namespace.endsWith(INDEXES_COLLECTION_NAME)) continue;
            String collectionName = this.extractCollectionNameFromNamespace(namespace);
            collectionDescription.put("name", (Object)collectionName);
            collectionDescription.put("options", (Object)collectionOptions);
            collectionDescription.put("info", (Object)new Document("readOnly", false));
            collectionDescription.put("type", (Object)"collection");
            collectionDescription.put("idIndex", (Object)new Document("key", new Document("_id", 1)).append("name", "_id_").append("ns", namespace).append("v", 2));
            firstBatch.add(collectionDescription);
        }
        return Utils.cursorResponse(this.getDatabaseName() + ".$cmd.listCollections", firstBatch);
    }

    private Document listIndexes(String collectionName) {
        Iterable indexes = Optional.ofNullable(this.resolveCollection(INDEXES_COLLECTION_NAME, false)).map(collection -> collection.handleQuery(new Document("ns", this.getDatabaseName() + "." + collectionName))).orElse(Collections.emptyList());
        return Utils.cursorResponse(this.getDatabaseName() + ".$cmd.listIndexes", indexes);
    }

    private synchronized MongoCollection<P> resolveOrCreateCollection(String collectionName) {
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection != null) {
            return collection;
        }
        return this.createCollection(collectionName);
    }

    private Document commandFind(String command, Document query) {
        ArrayList<Document> documents = new ArrayList<Document>();
        String collectionName = (String)query.get(command);
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection != null) {
            int numberToSkip = ((Number)query.getOrDefault("skip", 0)).intValue();
            int numberToReturn = ((Number)query.getOrDefault("limit", 0)).intValue();
            Document projection = (Document)query.get("projection");
            Document querySelector = new Document();
            querySelector.put("$query", (Object)query.getOrDefault("filter", new Document()));
            querySelector.put("$orderby", query.get("sort"));
            for (Document document : collection.handleQuery(querySelector, numberToSkip, numberToReturn, projection)) {
                documents.add(document);
            }
        }
        return Utils.cursorResponse(this.getDatabaseName() + "." + collectionName, documents);
    }

    private Document commandInsert(Channel channel, String command, Document query) {
        String collectionName = query.get(command).toString();
        boolean isOrdered = Utils.isTrue(query.get("ordered"));
        log.trace("ordered: {}", (Object)isOrdered);
        List documents = (List)query.get("documents");
        ArrayList<Document> writeErrors = new ArrayList<Document>();
        int n = 0;
        for (Document document : documents) {
            try {
                this.insertDocuments(channel, collectionName, Collections.singletonList(document));
                ++n;
            }
            catch (MongoServerError e) {
                Document error = new Document();
                error.put("index", (Object)n);
                error.put("errmsg", (Object)e.getMessageWithoutErrorCode());
                error.put("code", (Object)e.getCode());
                error.putIfNotNull("codeName", e.getCodeName());
                writeErrors.add(error);
            }
        }
        Document result = new Document();
        result.put("n", (Object)n);
        if (!writeErrors.isEmpty()) {
            result.put("writeErrors", (Object)writeErrors);
        }
        Utils.markOkay(result);
        return result;
    }

    private Document commandUpdate(Channel channel, String command, Document query) {
        this.clearLastStatus(channel);
        String collectionName = query.get(command).toString();
        boolean isOrdered = Utils.isTrue(query.get("ordered"));
        log.trace("ordered: {}", (Object)isOrdered);
        List updates = (List)query.get("updates");
        int nMatched = 0;
        int nModified = 0;
        ArrayList<Document> upserts = new ArrayList<Document>();
        ArrayList<Document> writeErrors = new ArrayList<Document>();
        Document response = new Document();
        for (int i = 0; i < updates.size(); ++i) {
            Document result;
            Document updateObj = (Document)updates.get(i);
            Document selector = (Document)updateObj.get("q");
            Document update = (Document)updateObj.get("u");
            boolean multi = Utils.isTrue(updateObj.get("multi"));
            boolean upsert = Utils.isTrue(updateObj.get("upsert"));
            try {
                result = this.updateDocuments(collectionName, selector, update, multi, upsert);
            }
            catch (MongoServerException e) {
                writeErrors.add(this.toWriteError(i, e));
                continue;
            }
            if (result.containsKey("upserted")) {
                Object id = result.get("upserted");
                Document upserted = new Document("index", upserts.size());
                upserted.put("_id", id);
                upserts.add(upserted);
            }
            nMatched += ((Integer)result.get("n")).intValue();
            nModified += ((Integer)result.get("nModified")).intValue();
        }
        response.put("n", (Object)(nMatched + upserts.size()));
        response.put("nModified", (Object)nModified);
        if (!upserts.isEmpty()) {
            response.put("upserted", (Object)upserts);
        }
        if (!writeErrors.isEmpty()) {
            response.put("writeErrors", (Object)writeErrors);
        }
        Utils.markOkay(response);
        this.putLastResult(channel, response);
        return response;
    }

    private Document commandDelete(Channel channel, String command, Document query) {
        String collectionName = query.get(command).toString();
        boolean isOrdered = Utils.isTrue(query.get("ordered"));
        log.trace("ordered: {}", (Object)isOrdered);
        List deletes = (List)query.get("deletes");
        int n = 0;
        for (Document delete : deletes) {
            Document selector = (Document)delete.get("q");
            int limit = ((Number)delete.get("limit")).intValue();
            Document result = this.deleteDocuments(channel, collectionName, selector, limit);
            Integer resultNumber = (Integer)result.get("n");
            n += resultNumber.intValue();
        }
        Document response = new Document("n", n);
        Utils.markOkay(response);
        return response;
    }

    private Document commandCreate(String command, Document query) {
        String collectionName = query.get(command).toString();
        boolean isCapped = Utils.isTrue(query.get("capped"));
        if (isCapped) {
            throw new MongoServerException("Creating capped collections is not yet implemented");
        }
        Object autoIndexId = query.get("autoIndexId");
        if (autoIndexId != null && !Utils.isTrue(autoIndexId)) {
            throw new MongoServerException("Disabling autoIndexId is not yet implemented");
        }
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection != null) {
            throw new MongoServerError(48, "NamespaceExists", "a collection '" + this.getDatabaseName() + "." + collectionName + "' already exists");
        }
        this.createCollection(collectionName);
        Document response = new Document();
        Utils.markOkay(response);
        return response;
    }

    private Document commandCreateIndexes(Document query) {
        int indexesBefore = this.countIndexes();
        Collection indexDescriptions = (Collection)query.get("indexes");
        for (Document indexDescription : indexDescriptions) {
            this.addIndex(indexDescription);
        }
        int indexesAfter = this.countIndexes();
        Document response = new Document();
        response.put("numIndexesBefore", (Object)indexesBefore);
        response.put("numIndexesAfter", (Object)indexesAfter);
        Utils.markOkay(response);
        return response;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int countIndexes() {
        MongoCollection<P> indexesCollection;
        AtomicReference<MongoCollection<P>> atomicReference = this.indexes;
        synchronized (atomicReference) {
            indexesCollection = this.indexes.get();
        }
        if (indexesCollection == null) {
            return 0;
        }
        return indexesCollection.count();
    }

    private Collection<MongoCollection<P>> collections() {
        return this.collections.values().stream().filter(collection -> !AbstractMongoDatabase.isSystemCollection(collection.getCollectionName())).collect(Collectors.toCollection(LinkedHashSet::new));
    }

    private Document commandDatabaseStats() {
        Document response = new Document("db", this.getDatabaseName());
        response.put("collections", (Object)this.collections().size());
        long storageSize = this.getStorageSize();
        long fileSize = this.getFileSize();
        long indexSize = 0L;
        int objects = 0;
        double dataSize = 0.0;
        double averageObjectSize = 0.0;
        for (MongoCollection<P> collection : this.collections()) {
            Document stats = collection.getStats();
            objects += ((Number)stats.get("count")).intValue();
            dataSize += ((Number)stats.get("size")).doubleValue();
            Document indexSizes = (Document)stats.get("indexSize");
            for (String indexName : indexSizes.keySet()) {
                indexSize += ((Number)indexSizes.get(indexName)).longValue();
            }
        }
        if (objects > 0) {
            averageObjectSize = dataSize / (double)objects;
        }
        response.put("objects", (Object)objects);
        response.put("avgObjSize", (Object)averageObjectSize);
        if (dataSize == 0.0) {
            response.put("dataSize", (Object)0);
        } else {
            response.put("dataSize", (Object)dataSize);
        }
        response.put("storageSize", (Object)storageSize);
        response.put("numExtents", (Object)0);
        response.put("indexes", (Object)this.countIndexes());
        response.put("indexSize", (Object)indexSize);
        response.put("fileSize", (Object)fileSize);
        response.put("nsSizeMB", (Object)0);
        Utils.markOkay(response);
        return response;
    }

    protected abstract long getFileSize();

    protected abstract long getStorageSize();

    private Document commandDrop(Document query) {
        String collectionName = query.get("drop").toString();
        MongoCollection<P> collection = this.collections.remove(collectionName);
        if (collection == null) {
            throw new MongoSilentServerException("ns not found");
        }
        Document response = new Document();
        this.namespaces.removeDocument(new Document("name", collection.getFullName()));
        response.put("nIndexesWas", (Object)collection.getNumIndexes());
        response.put("ns", (Object)collection.getFullName());
        Utils.markOkay(response);
        return response;
    }

    private Document commandGetLastError(Channel channel, String command, Document query) {
        Document result;
        List<Document> results;
        Iterator<String> it = query.keySet().iterator();
        String cmd = it.next();
        Assert.equals(cmd, command);
        if (it.hasNext()) {
            String subCommand;
            switch (subCommand = it.next()) {
                case "w": {
                    break;
                }
                case "fsync": {
                    break;
                }
                default: {
                    throw new MongoServerException("unknown subcommand: " + subCommand);
                }
            }
        }
        if ((results = this.lastResults.get(channel)) != null && !results.isEmpty()) {
            result = results.get(results.size() - 1);
            if (result == null) {
                result = new Document();
            }
        } else {
            result = new Document();
            result.put("err", (Object)null);
            result.put("n", (Object)0);
        }
        Utils.markOkay(result);
        return result;
    }

    private Document commandGetPrevError(Channel channel) {
        List<Document> results = this.lastResults.get(channel);
        if (results != null) {
            for (int i = 1; i < results.size(); ++i) {
                Document result = results.get(results.size() - i);
                if (result == null) continue;
                boolean isRelevant = false;
                if (result.get("err") != null) {
                    isRelevant = true;
                } else if (((Number)result.get("n")).intValue() > 0) {
                    isRelevant = true;
                }
                if (!isRelevant) continue;
                result.put("nPrev", (Object)i);
                Utils.markOkay(result);
                return result;
            }
        }
        Document result = new Document();
        result.put("nPrev", (Object)-1);
        result.put("n", (Object)0);
        result.put("err", (Object)null);
        Utils.markOkay(result);
        return result;
    }

    private Document commandResetError(Channel channel) {
        List<Document> results = this.lastResults.get(channel);
        if (results != null) {
            results.clear();
        }
        Document result = new Document();
        Utils.markOkay(result);
        return result;
    }

    private Document commandCount(String command, Document query) {
        MongoCollection<P> collection = this.resolveCollection(command, query, false);
        Document response = new Document();
        if (collection == null) {
            response.put("n", (Object)0);
        } else {
            Document queryObject = (Document)query.get("query");
            int limit = this.getOptionalNumber(query, "limit", -1);
            int skip = this.getOptionalNumber(query, "skip", 0);
            response.put("n", (Object)collection.count(queryObject, skip, limit));
        }
        Utils.markOkay(response);
        return response;
    }

    private Document commandAggregate(String command, Document query) {
        String collectionName = query.get(command).toString();
        Document cursor = (Document)query.get("cursor");
        if (cursor == null) {
            throw new MongoServerError(9, "FailedToParse", "The 'cursor' option is required, except for aggregate with the explain argument");
        }
        if (!cursor.isEmpty()) {
            throw new MongoServerException("Non-empty cursor is not yet implemented");
        }
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        Aggregation aggregation = new Aggregation(collection);
        List pipeline = (List)query.get("pipeline");
        block26: for (Document stage : pipeline) {
            String stageOperation;
            if (stage.size() != 1) {
                throw new MongoServerError(40323, "A pipeline stage specification object must contain exactly one field.");
            }
            switch (stageOperation = stage.keySet().iterator().next()) {
                case "$match": {
                    Document matchQuery = (Document)stage.get(stageOperation);
                    aggregation.addStage(new MatchStage(matchQuery));
                    continue block26;
                }
                case "$skip": {
                    Number numSkip = (Number)stage.get(stageOperation);
                    aggregation.addStage(new SkipStage(numSkip.longValue()));
                    continue block26;
                }
                case "$limit": {
                    Number numLimit = (Number)stage.get(stageOperation);
                    aggregation.addStage(new LimitStage(numLimit.longValue()));
                    continue block26;
                }
                case "$sort": {
                    Document orderBy = (Document)stage.get(stageOperation);
                    aggregation.addStage(new OrderByStage(orderBy));
                    continue block26;
                }
                case "$project": {
                    aggregation.addStage(new ProjectStage((Document)stage.get(stageOperation)));
                    continue block26;
                }
                case "$count": {
                    String count = (String)stage.get(stageOperation);
                    aggregation.addStage(new GroupStage(new Document("_id", null).append(count, new Document("$sum", 1))));
                    aggregation.addStage(new ProjectStage(new Document("_id", 0)));
                    continue block26;
                }
                case "$group": {
                    Document groupDetails = (Document)stage.get(stageOperation);
                    aggregation.addStage(new GroupStage(groupDetails));
                    continue block26;
                }
                case "$addFields": {
                    Document addFieldsDetails = (Document)stage.get(stageOperation);
                    aggregation.addStage(new AddFieldsStage(addFieldsDetails));
                    continue block26;
                }
                case "$unwind": {
                    String unwindField = (String)stage.get(stageOperation);
                    aggregation.addStage(new UnwindStage(unwindField));
                    continue block26;
                }
                case "$lookup": {
                    Document lookup = (Document)stage.get(stageOperation);
                    aggregation.addStage(new LookupStage(lookup, this));
                    continue block26;
                }
                case "$replaceRoot": {
                    Document replaceRoot = (Document)stage.get(stageOperation);
                    aggregation.addStage(new ReplaceRootStage(replaceRoot));
                    continue block26;
                }
            }
            throw new MongoServerError(40324, "Unrecognized pipeline stage name: '" + stageOperation + "'");
        }
        return Utils.cursorResponse(this.getDatabaseName() + "." + collectionName, aggregation.getResult());
    }

    private int getOptionalNumber(Document query, String fieldName, int defaultValue) {
        Number limitNumber = (Number)query.get(fieldName);
        return limitNumber != null ? limitNumber.intValue() : defaultValue;
    }

    @Override
    public Iterable<Document> handleQuery(MongoQuery query) {
        this.clearLastStatus(query.getChannel());
        String collectionName = query.getCollectionName();
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection == null) {
            return Collections.emptyList();
        }
        int numSkip = query.getNumberToSkip();
        int numReturn = query.getNumberToReturn();
        Document fieldSelector = query.getReturnFieldSelector();
        return collection.handleQuery(query.getQuery(), numSkip, numReturn, fieldSelector);
    }

    @Override
    public void handleClose(Channel channel) {
        this.lastResults.remove(channel);
    }

    private synchronized void clearLastStatus(Channel channel) {
        List results = this.lastResults.computeIfAbsent(channel, k -> new LimitedList(10));
        results.add(null);
    }

    @Override
    public void handleInsert(MongoInsert insert) {
        Channel channel = insert.getChannel();
        String collectionName = insert.getCollectionName();
        List<Document> documents = insert.getDocuments();
        if (collectionName.equals(INDEXES_COLLECTION_NAME)) {
            for (Document indexDescription : documents) {
                this.addIndex(indexDescription);
            }
        } else {
            try {
                this.insertDocuments(channel, collectionName, documents);
            }
            catch (MongoServerException e) {
                log.error("failed to insert {}", (Object)insert, (Object)e);
            }
        }
    }

    private MongoCollection<P> resolveCollection(String command, Document query, boolean throwIfNotFound) {
        String collectionName = query.get(command).toString();
        return this.resolveCollection(collectionName, throwIfNotFound);
    }

    public synchronized MongoCollection<P> resolveCollection(String collectionName, boolean throwIfNotFound) {
        this.checkCollectionName(collectionName);
        MongoCollection<P> collection = this.collections.get(collectionName);
        if (collection == null && throwIfNotFound) {
            throw new MongoServerException("Collection [" + this.getDatabaseName() + "." + collectionName + "] not found.");
        }
        return collection;
    }

    private void checkCollectionName(String collectionName) {
        if (collectionName.length() > 128) {
            throw new MongoServerError(10080, "ns name too long, max size is 128");
        }
        if (collectionName.isEmpty()) {
            throw new MongoServerError(16256, "Invalid ns [" + collectionName + "]");
        }
    }

    @Override
    public boolean isEmpty() {
        return this.collections.isEmpty();
    }

    private void addNamespace(MongoCollection<P> collection) {
        this.collections.put(collection.getCollectionName(), collection);
        if (!AbstractMongoDatabase.isSystemCollection(collection.getCollectionName())) {
            this.namespaces.addDocument(new Document("name", collection.getFullName()));
        }
    }

    @Override
    public void handleDelete(MongoDelete delete) {
        Channel channel = delete.getChannel();
        String collectionName = delete.getCollectionName();
        Document selector = delete.getSelector();
        int limit = delete.isSingleRemove() ? 1 : Integer.MAX_VALUE;
        try {
            this.deleteDocuments(channel, collectionName, selector, limit);
        }
        catch (MongoServerException e) {
            log.error("failed to delete {}", (Object)delete, (Object)e);
        }
    }

    @Override
    public void handleUpdate(MongoUpdate updateCommand) {
        Channel channel = updateCommand.getChannel();
        String collectionName = updateCommand.getCollectionName();
        Document selector = updateCommand.getSelector();
        Document update = updateCommand.getUpdate();
        boolean multi = updateCommand.isMulti();
        boolean upsert = updateCommand.isUpsert();
        this.clearLastStatus(channel);
        try {
            Document result = this.updateDocuments(collectionName, selector, update, multi, upsert);
            this.putLastResult(channel, result);
        }
        catch (MongoServerException e) {
            this.putLastError(channel, e);
            log.error("failed to update {}", (Object)updateCommand, (Object)e);
        }
    }

    private void addIndex(Document indexDescription) {
        if (!indexDescription.containsKey("v")) {
            indexDescription.put("v", (Object)2);
        }
        this.openOrCreateIndex(indexDescription);
        this.getOrCreateIndexesCollection().addDocument(indexDescription);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private MongoCollection<P> getOrCreateIndexesCollection() {
        AtomicReference<MongoCollection<P>> atomicReference = this.indexes;
        synchronized (atomicReference) {
            if (this.indexes.get() == null) {
                MongoCollection<P> indexCollection = this.openOrCreateCollection(INDEXES_COLLECTION_NAME, null);
                this.addNamespace(indexCollection);
                this.indexes.set(indexCollection);
            }
            return this.indexes.get();
        }
    }

    private String extractCollectionNameFromNamespace(String namespace) {
        Assert.startsWith(namespace, this.databaseName);
        return namespace.substring(this.databaseName.length() + 1);
    }

    private void openOrCreateIndex(Document indexDescription) {
        String ns = indexDescription.get("ns").toString();
        String collectionName = this.extractCollectionNameFromNamespace(ns);
        MongoCollection<P> collection = this.resolveOrCreateCollection(collectionName);
        Document key = (Document)indexDescription.get("key");
        if (key.keySet().equals(Collections.singleton("_id"))) {
            boolean ascending = AbstractMongoDatabase.isAscending(key.get("_id"));
            collection.addIndex(this.openOrCreateIdIndex(collectionName, ascending));
            log.info("adding unique _id index for collection {}", (Object)collectionName);
        } else if (Utils.isTrue(indexDescription.get("unique"))) {
            ArrayList<IndexKey> keys = new ArrayList<IndexKey>();
            for (Map.Entry<String, Object> entry : key.entrySet()) {
                String field = entry.getKey();
                boolean ascending = AbstractMongoDatabase.isAscending(entry.getValue());
                keys.add(new IndexKey(field, ascending));
            }
            boolean sparse = Utils.isTrue(indexDescription.get("sparse"));
            log.info("adding {} unique index {} for collection {}", new Object[]{sparse ? "sparse" : "non-sparse", keys, collectionName});
            collection.addIndex(this.openOrCreateUniqueIndex(collectionName, keys, sparse));
        } else {
            log.warn("adding non-unique non-id index with key {} is not yet implemented", (Object)key);
        }
    }

    private static boolean isAscending(Object keyValue) {
        return Objects.equals(Utils.normalizeValue(keyValue), 1.0);
    }

    private Index<P> openOrCreateIdIndex(String collectionName, boolean ascending) {
        return this.openOrCreateUniqueIndex(collectionName, Collections.singletonList(new IndexKey("_id", ascending)), false);
    }

    protected abstract Index<P> openOrCreateUniqueIndex(String var1, List<IndexKey> var2, boolean var3);

    private void insertDocuments(Channel channel, String collectionName, List<Document> documents) {
        this.clearLastStatus(channel);
        try {
            if (AbstractMongoDatabase.isSystemCollection(collectionName)) {
                throw new MongoServerError(16459, "attempt to insert in system namespace");
            }
            MongoCollection<P> collection = this.resolveOrCreateCollection(collectionName);
            collection.insertDocuments(documents);
            Document result = new Document("n", 0);
            result.put("err", (Object)null);
            this.putLastResult(channel, result);
        }
        catch (MongoServerError e) {
            this.putLastError(channel, e);
            throw e;
        }
    }

    private Document deleteDocuments(Channel channel, String collectionName, Document selector, int limit) {
        this.clearLastStatus(channel);
        try {
            if (AbstractMongoDatabase.isSystemCollection(collectionName)) {
                throw new MongoServerError(73, "InvalidNamespace", "cannot write to '" + this.getDatabaseName() + "." + collectionName + "'");
            }
            MongoCollection<P> collection = this.resolveCollection(collectionName, false);
            int n = collection == null ? 0 : collection.deleteDocuments(selector, limit);
            Document result = new Document("n", n);
            this.putLastResult(channel, result);
            return result;
        }
        catch (MongoServerError e) {
            this.putLastError(channel, e);
            throw e;
        }
    }

    private Document updateDocuments(String collectionName, Document selector, Document update, boolean multi, boolean upsert) {
        if (AbstractMongoDatabase.isSystemCollection(collectionName)) {
            throw new MongoServerError(10156, "cannot update system collection");
        }
        MongoCollection<P> collection = this.resolveOrCreateCollection(collectionName);
        return collection.updateDocuments(selector, update, multi, upsert);
    }

    private void putLastError(Channel channel, MongoServerException ex) {
        Document error = this.toError(channel, ex);
        this.putLastResult(channel, error);
    }

    private Document toWriteError(int index, MongoServerException e) {
        Document error = new Document();
        error.put("index", (Object)index);
        error.put("errmsg", (Object)e.getMessageWithoutErrorCode());
        if (e instanceof MongoServerError) {
            MongoServerError err = (MongoServerError)e;
            error.put("code", (Object)err.getCode());
            error.putIfNotNull("codeName", err.getCodeName());
        }
        return error;
    }

    private Document toError(Channel channel, MongoServerException ex) {
        Document error = new Document();
        error.put("err", (Object)ex.getMessageWithoutErrorCode());
        if (ex instanceof MongoServerError) {
            MongoServerError err = (MongoServerError)ex;
            error.put("code", (Object)err.getCode());
            error.putIfNotNull("codeName", err.getCodeName());
        }
        error.put("connectionId", (Object)channel.id().asShortText());
        return error;
    }

    private synchronized void putLastResult(Channel channel, Document result) {
        List<Document> results = this.lastResults.get(channel);
        Document last = results.get(results.size() - 1);
        Assert.isNull(last, () -> "last result already set: " + last);
        results.set(results.size() - 1, result);
    }

    private MongoCollection<P> createCollection(String collectionName) {
        this.checkCollectionName(collectionName);
        if (collectionName.contains("$")) {
            throw new MongoServerError(10093, "cannot insert into reserved $ collection");
        }
        MongoCollection<P> collection = this.openOrCreateCollection(collectionName, "_id");
        this.addNamespace(collection);
        Document indexDescription = new Document();
        indexDescription.put("name", (Object)"_id_");
        indexDescription.put("ns", (Object)collection.getFullName());
        indexDescription.put("key", (Object)new Document("_id", 1));
        this.addIndex(indexDescription);
        log.info("created collection {}", (Object)collection.getFullName());
        return collection;
    }

    protected abstract MongoCollection<P> openOrCreateCollection(String var1, String var2);

    @Override
    public void drop() {
        log.debug("dropping {}", (Object)this);
        for (String collectionName : this.collections.keySet()) {
            if (AbstractMongoDatabase.isSystemCollection(collectionName)) continue;
            this.dropCollection(collectionName);
        }
        this.dropCollectionIfExists(INDEXES_COLLECTION_NAME);
        this.dropCollectionIfExists(NAMESPACES_COLLECTION_NAME);
    }

    private void dropCollectionIfExists(String collectionName) {
        if (this.collections.containsKey(collectionName)) {
            this.dropCollection(collectionName);
        }
    }

    @Override
    public void dropCollection(String collectionName) {
        this.unregisterCollection(collectionName);
    }

    public MongoCollection<P> unregisterCollection(String collectionName) {
        MongoCollection<P> removedCollection = this.collections.remove(collectionName);
        this.namespaces.deleteDocuments(new Document("name", removedCollection.getFullName()), 1);
        return removedCollection;
    }

    @Override
    public void moveCollection(MongoDatabase oldDatabase, MongoCollection<?> collection, String newCollectionName) {
        oldDatabase.unregisterCollection(collection.getCollectionName());
        collection.renameTo(this.getDatabaseName(), newCollectionName);
        MongoCollection<?> newCollection = collection;
        this.collections.put(newCollectionName, newCollection);
        ArrayList<Document> newDocuments = new ArrayList<Document>();
        newDocuments.add(new Document("name", collection.getFullName()));
        this.namespaces.insertDocuments(newDocuments);
    }

    private static boolean isSystemCollection(String collectionName) {
        return collectionName.startsWith("system.");
    }
}

