/*
 * Decompiled with CFR 0.152.
 */
package de.bwaldvogel.mongo.backend;

import de.bwaldvogel.mongo.MongoCollection;
import de.bwaldvogel.mongo.MongoDatabase;
import de.bwaldvogel.mongo.backend.ArrayFilters;
import de.bwaldvogel.mongo.backend.Assert;
import de.bwaldvogel.mongo.backend.CollectionOptions;
import de.bwaldvogel.mongo.backend.CollectionUtils;
import de.bwaldvogel.mongo.backend.Cursor;
import de.bwaldvogel.mongo.backend.CursorRegistry;
import de.bwaldvogel.mongo.backend.EmptyIndex;
import de.bwaldvogel.mongo.backend.Index;
import de.bwaldvogel.mongo.backend.IndexKey;
import de.bwaldvogel.mongo.backend.LimitedList;
import de.bwaldvogel.mongo.backend.QueryParameters;
import de.bwaldvogel.mongo.backend.QueryResult;
import de.bwaldvogel.mongo.backend.Utils;
import de.bwaldvogel.mongo.backend.aggregation.Aggregation;
import de.bwaldvogel.mongo.bson.Document;
import de.bwaldvogel.mongo.exception.IndexNotFoundException;
import de.bwaldvogel.mongo.exception.MongoServerError;
import de.bwaldvogel.mongo.exception.MongoServerException;
import de.bwaldvogel.mongo.exception.MongoSilentServerException;
import de.bwaldvogel.mongo.exception.NamespaceExistsException;
import de.bwaldvogel.mongo.exception.NoSuchCommandException;
import de.bwaldvogel.mongo.oplog.NoopOplog;
import de.bwaldvogel.mongo.oplog.Oplog;
import de.bwaldvogel.mongo.util.FutureUtils;
import de.bwaldvogel.mongo.wire.message.MongoDelete;
import de.bwaldvogel.mongo.wire.message.MongoInsert;
import de.bwaldvogel.mongo.wire.message.MongoQuery;
import de.bwaldvogel.mongo.wire.message.MongoUpdate;
import io.netty.channel.Channel;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractMongoDatabase<P>
implements MongoDatabase {
    private static final String NAMESPACES_COLLECTION_NAME = "system.namespaces";
    private static final String INDEXES_COLLECTION_NAME = "system.indexes";
    private static final Logger log = LoggerFactory.getLogger(AbstractMongoDatabase.class);
    protected final String databaseName;
    private final Map<String, MongoCollection<P>> collections = new ConcurrentHashMap<String, MongoCollection<P>>();
    protected final AtomicReference<MongoCollection<P>> indexes = new AtomicReference();
    private final Map<Channel, List<Document>> lastResults = new ConcurrentHashMap<Channel, List<Document>>();
    private MongoCollection<P> namespaces;
    protected final CursorRegistry cursorRegistry;

    protected AbstractMongoDatabase(String databaseName, CursorRegistry cursorRegistry) {
        this.databaseName = databaseName;
        this.cursorRegistry = cursorRegistry;
    }

    protected void initializeNamespacesAndIndexes() {
        this.namespaces = this.openOrCreateCollection(NAMESPACES_COLLECTION_NAME, CollectionOptions.withIdField("name"));
        this.collections.put(this.namespaces.getCollectionName(), this.namespaces);
        if (!this.namespaces.isEmpty()) {
            for (String name : this.listCollectionNamespaces()) {
                log.debug("opening {}", (Object)name);
                String collectionName = this.extractCollectionNameFromNamespace(name);
                MongoCollection<P> collection = this.openOrCreateCollection(collectionName, CollectionOptions.withDefaults());
                this.collections.put(collectionName, collection);
                log.debug("opened collection '{}'", (Object)collectionName);
            }
            MongoCollection<P> indexCollection = this.openOrCreateCollection(INDEXES_COLLECTION_NAME, CollectionOptions.withoutIdField());
            this.collections.put(indexCollection.getCollectionName(), indexCollection);
            this.indexes.set(indexCollection);
            for (Document indexDescription : indexCollection.queryAll()) {
                this.openOrCreateIndex(indexDescription);
            }
        }
    }

    @Override
    public final String getDatabaseName() {
        return this.databaseName;
    }

    public String toString() {
        return this.getClass().getSimpleName() + "(" + this.getDatabaseName() + ")";
    }

    private Document commandError(Channel channel, String command, Document query) {
        if (command.equalsIgnoreCase("getlasterror")) {
            return this.commandGetLastError(channel, command, query);
        }
        if (command.equalsIgnoreCase("reseterror")) {
            return this.commandResetError(channel);
        }
        return null;
    }

    private Document handleCommandSync(Channel channel, String command, Document query, Oplog oplog) {
        if (command.equalsIgnoreCase("find")) {
            return this.commandFind(command, query);
        }
        if (command.equalsIgnoreCase("insert")) {
            return this.commandInsert(channel, command, query, oplog);
        }
        if (command.equalsIgnoreCase("update")) {
            return this.commandUpdate(channel, command, query, oplog);
        }
        if (command.equalsIgnoreCase("delete")) {
            return this.commandDelete(channel, command, query, oplog);
        }
        if (command.equalsIgnoreCase("create")) {
            return this.commandCreate(command, query);
        }
        if (command.equalsIgnoreCase("createIndexes")) {
            String collectionName = (String)query.get(command);
            return this.commandCreateIndexes(query, collectionName);
        }
        if (command.equalsIgnoreCase("count")) {
            return this.commandCount(command, query);
        }
        if (command.equalsIgnoreCase("aggregate")) {
            return this.commandAggregate(command, query, oplog);
        }
        if (command.equalsIgnoreCase("distinct")) {
            MongoCollection<P> collection = this.resolveCollection(command, query);
            if (collection == null) {
                Document response = new Document("values", Collections.emptyList());
                Utils.markOkay(response);
                return response;
            }
            return collection.handleDistinct(query);
        }
        if (command.equalsIgnoreCase("drop")) {
            return this.commandDrop(query, oplog);
        }
        if (command.equalsIgnoreCase("dropIndexes")) {
            return this.commandDropIndexes(query);
        }
        if (command.equalsIgnoreCase("dbstats")) {
            return this.commandDatabaseStats();
        }
        if (command.equalsIgnoreCase("collstats")) {
            MongoCollection<P> collection = this.resolveCollection(command, query);
            if (collection == null) {
                Document emptyStats = new Document().append("count", 0).append("size", 0);
                Utils.markOkay(emptyStats);
                return emptyStats;
            }
            return collection.getStats();
        }
        if (command.equalsIgnoreCase("validate")) {
            MongoCollection<P> collection = this.resolveCollection(command, query);
            if (collection == null) {
                throw new MongoServerError(26, "NamespaceNotFound", "ns not found");
            }
            return collection.validate();
        }
        if (command.equalsIgnoreCase("findAndModify")) {
            String collectionName = query.get(command).toString();
            MongoCollection<P> collection = this.resolveOrCreateCollection(collectionName);
            return collection.findAndModify(query);
        }
        if (command.equalsIgnoreCase("listCollections")) {
            return this.listCollections();
        }
        if (command.equalsIgnoreCase("listIndexes")) {
            String collectionName = query.get(command).toString();
            return this.listIndexes(collectionName);
        }
        log.error("unknown query: {}", (Object)query);
        throw new NoSuchCommandException(command);
    }

    @Override
    public Document handleCommand(Channel channel, String command, Document query, Oplog oplog) {
        Document commandErrorDocument = this.commandError(channel, command, query);
        if (commandErrorDocument != null) {
            return commandErrorDocument;
        }
        this.clearLastStatus(channel);
        return this.handleCommandSync(channel, command, query, oplog);
    }

    @Override
    public CompletionStage<Document> handleCommandAsync(Channel channel, String command, Document query, Oplog oplog) {
        Document commandErrorDocument = this.commandError(channel, command, query);
        if (commandErrorDocument != null) {
            return FutureUtils.wrap(() -> commandErrorDocument);
        }
        this.clearLastStatus(channel);
        if ("find".equalsIgnoreCase(command)) {
            return this.commandFindAsync(command, query);
        }
        return FutureUtils.wrap(() -> this.handleCommandSync(channel, command, query, oplog));
    }

    private Document listCollections() {
        ArrayList<Document> firstBatch = new ArrayList<Document>();
        for (String namespace : this.listCollectionNamespaces()) {
            if (namespace.endsWith(INDEXES_COLLECTION_NAME)) continue;
            Document collectionDescription = new Document();
            Document collectionOptions = new Document();
            String collectionName = this.extractCollectionNameFromNamespace(namespace);
            collectionDescription.put("name", (Object)collectionName);
            collectionDescription.put("options", (Object)collectionOptions);
            collectionDescription.put("info", (Object)new Document("readOnly", false));
            collectionDescription.put("type", (Object)"collection");
            collectionDescription.put("idIndex", (Object)AbstractMongoDatabase.getPrimaryKeyIndexDescription(namespace));
            firstBatch.add(collectionDescription);
        }
        return Utils.firstBatchCursorResponse(this.getDatabaseName() + ".$cmd.listCollections", firstBatch);
    }

    private static Document getPrimaryKeyIndexDescription(String namespace) {
        return new Document("key", new Document("_id", 1)).append("name", "_id_").append("ns", namespace).append("v", 2);
    }

    private Iterable<String> listCollectionNamespaces() {
        return this.namespaces.queryAllAsStream().map(document -> document.get("name").toString())::iterator;
    }

    private Document listIndexes(String collectionName) {
        Iterable indexes = Optional.ofNullable(this.resolveCollection(INDEXES_COLLECTION_NAME, false)).map(collection -> collection.handleQuery(new Document("ns", this.getFullCollectionNamespace(collectionName)))).orElse(Collections.emptyList());
        return Utils.firstBatchCursorResponse(this.getDatabaseName() + ".$cmd.listIndexes", indexes);
    }

    protected MongoCollection<P> resolveOrCreateCollection(String collectionName) {
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection != null) {
            return collection;
        }
        return this.createCollection(collectionName, CollectionOptions.withDefaults());
    }

    private Document commandFind(String command, Document query) {
        String collectionName = (String)query.get(command);
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection == null) {
            return Utils.firstBatchCursorResponse(this.getFullCollectionNamespace(collectionName), Collections.emptyList());
        }
        QueryParameters queryParameters = AbstractMongoDatabase.toQueryParameters(query);
        QueryResult queryResult = collection.handleQuery(queryParameters);
        return this.toCursorResponse(collection, queryResult);
    }

    private CompletionStage<Document> commandFindAsync(String command, Document query) {
        String collectionName = (String)query.get(command);
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection == null) {
            return FutureUtils.wrap(() -> Utils.firstBatchCursorResponse(this.getFullCollectionNamespace(collectionName), Collections.emptyList()));
        }
        QueryParameters queryParameters = AbstractMongoDatabase.toQueryParameters(query);
        return collection.handleQueryAsync(queryParameters).thenApply(queryResult -> this.toCursorResponse(collection, (QueryResult)queryResult));
    }

    private static QueryParameters toQueryParameters(Document query) {
        int numberToSkip = ((Number)query.getOrDefault("skip", 0)).intValue();
        int numberToReturn = ((Number)query.getOrDefault("limit", 0)).intValue();
        int batchSize = ((Number)query.getOrDefault("batchSize", 0)).intValue();
        Document querySelector = new Document();
        querySelector.put("$query", (Object)query.getOrDefault("filter", new Document()));
        querySelector.put("$orderby", query.get("sort"));
        Document projection = (Document)query.get("projection");
        return new QueryParameters(querySelector, numberToSkip, numberToReturn, batchSize, projection);
    }

    private QueryParameters toQueryParameters(MongoQuery query, int numberToSkip, int batchSize) {
        return new QueryParameters(query.getQuery(), numberToSkip, 0, batchSize, query.getReturnFieldSelector());
    }

    private Document toCursorResponse(MongoCollection<P> collection, QueryResult queryResult) {
        ArrayList<Document> documents = new ArrayList<Document>();
        for (Document document : queryResult) {
            documents.add(document);
        }
        return Utils.firstBatchCursorResponse(collection.getFullName(), documents, queryResult.getCursorId());
    }

    private Document commandInsert(Channel channel, String command, Document query, Oplog oplog) {
        String collectionName = query.get(command).toString();
        boolean isOrdered = Utils.isTrue(query.get("ordered"));
        log.trace("ordered: {}", (Object)isOrdered);
        List documents = (List)query.get("documents");
        List<Document> writeErrors = this.insertDocuments(channel, collectionName, documents, oplog, isOrdered);
        Document result = new Document();
        result.put("n", (Object)documents.size());
        if (!writeErrors.isEmpty()) {
            result.put("writeErrors", (Object)writeErrors);
        }
        Utils.markOkay(result);
        return result;
    }

    private Document commandUpdate(Channel channel, String command, Document query, Oplog oplog) {
        this.clearLastStatus(channel);
        String collectionName = query.get(command).toString();
        boolean isOrdered = Utils.isTrue(query.get("ordered"));
        log.trace("ordered: {}", (Object)isOrdered);
        List updates = (List)query.get("updates");
        int nMatched = 0;
        int nModified = 0;
        ArrayList<Document> upserts = new ArrayList<Document>();
        ArrayList<Document> writeErrors = new ArrayList<Document>();
        Document response = new Document();
        for (int i = 0; i < updates.size(); ++i) {
            Document result;
            Document updateObj = (Document)updates.get(i);
            Document selector = (Document)updateObj.get("q");
            Document update = (Document)updateObj.get("u");
            ArrayFilters arrayFilters = ArrayFilters.parse(updateObj, update);
            boolean multi = Utils.isTrue(updateObj.get("multi"));
            boolean upsert = Utils.isTrue(updateObj.get("upsert"));
            try {
                result = this.updateDocuments(collectionName, selector, update, arrayFilters, multi, upsert, oplog);
            }
            catch (MongoServerException e) {
                writeErrors.add(this.toWriteError(i, e));
                continue;
            }
            if (result.containsKey("upserted")) {
                Object id = result.get("upserted");
                Document upserted = new Document("index", i);
                upserted.put("_id", id);
                upserts.add(upserted);
            }
            nMatched += ((Integer)result.get("n")).intValue();
            nModified += ((Integer)result.get("nModified")).intValue();
        }
        response.put("n", (Object)(nMatched + upserts.size()));
        response.put("nModified", (Object)nModified);
        if (!upserts.isEmpty()) {
            response.put("upserted", (Object)upserts);
        }
        if (!writeErrors.isEmpty()) {
            response.put("writeErrors", (Object)writeErrors);
        }
        Utils.markOkay(response);
        this.putLastResult(channel, response);
        return response;
    }

    private Document commandDelete(Channel channel, String command, Document query, Oplog oplog) {
        String collectionName = query.get(command).toString();
        boolean isOrdered = Utils.isTrue(query.get("ordered"));
        log.trace("ordered: {}", (Object)isOrdered);
        List deletes = (List)query.get("deletes");
        int n = 0;
        for (Document delete : deletes) {
            Document selector = (Document)delete.get("q");
            int limit = ((Number)delete.get("limit")).intValue();
            Document result = this.deleteDocuments(channel, collectionName, selector, limit, oplog);
            Integer resultNumber = (Integer)result.get("n");
            n += resultNumber.intValue();
        }
        Document response = new Document("n", n);
        Utils.markOkay(response);
        return response;
    }

    private Document commandCreate(String command, Document query) {
        String collectionName = query.get(command).toString();
        CollectionOptions collectionOptions = CollectionOptions.fromQuery(query);
        collectionOptions.validate();
        this.createCollectionOrThrowIfExists(collectionName, collectionOptions);
        Document response = new Document();
        Utils.markOkay(response);
        return response;
    }

    public MongoCollection<P> createCollectionOrThrowIfExists(String collectionName, CollectionOptions options) {
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection != null) {
            throw new NamespaceExistsException("a collection '" + collection.getFullName() + "' already exists");
        }
        return this.createCollection(collectionName, options);
    }

    private Document commandCreateIndexes(Document query, String collectionName) {
        int indexesBefore = this.countIndexes();
        Collection indexDescriptions = (Collection)query.get("indexes");
        for (Document indexDescription : indexDescriptions) {
            indexDescription.putIfAbsent("ns", this.getFullCollectionNamespace(collectionName));
            this.addIndex(indexDescription);
        }
        int indexesAfter = this.countIndexes();
        Document response = new Document();
        response.put("numIndexesBefore", (Object)indexesBefore);
        response.put("numIndexesAfter", (Object)indexesAfter);
        Utils.markOkay(response);
        return response;
    }

    private Document commandDropIndexes(Document query) {
        String collectionName = (String)query.get("dropIndexes");
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection != null) {
            this.dropIndexes(collection, query);
        }
        Document response = new Document();
        Utils.markOkay(response);
        return response;
    }

    private void dropIndexes(MongoCollection<P> collection, Document query) {
        Object index = query.get("index");
        Assert.notNull(index, () -> "Index name must not be null");
        MongoCollection<P> indexCollection = this.indexes.get();
        if (Objects.equals(index, "*")) {
            for (Document indexDocument : indexCollection.queryAll()) {
                Document indexKeys = (Document)indexDocument.get("key");
                if (this.isPrimaryKeyIndex(indexKeys)) continue;
                this.dropIndex(collection, indexDocument);
            }
        } else if (index instanceof String) {
            this.dropIndex(collection, new Document("name", index));
        } else {
            Document indexKeys = (Document)index;
            Document indexQuery = new Document("key", indexKeys).append("ns", collection.getFullName());
            Document indexToDrop = CollectionUtils.getSingleElement(indexCollection.handleQuery(indexQuery), () -> new IndexNotFoundException(indexKeys));
            int numDeleted = this.dropIndex(collection, indexToDrop);
            Assert.equals(numDeleted, 1L, () -> "Expected one deleted document");
        }
    }

    private int dropIndex(MongoCollection<P> collection, Document indexDescription) {
        String indexName = (String)indexDescription.get("name");
        this.dropIndex(collection, indexName);
        return this.indexes.get().deleteDocuments(indexDescription, -1);
    }

    protected void dropIndex(MongoCollection<P> collection, String indexName) {
        collection.dropIndex(indexName);
    }

    protected int countIndexes() {
        MongoCollection<P> indexesCollection = this.indexes.get();
        if (indexesCollection == null) {
            return 0;
        }
        return indexesCollection.count();
    }

    private Collection<MongoCollection<P>> collections() {
        return this.collections.values().stream().filter(collection -> !AbstractMongoDatabase.isSystemCollection(collection.getCollectionName())).collect(Collectors.toCollection(LinkedHashSet::new));
    }

    private Document commandDatabaseStats() {
        Document response = new Document("db", this.getDatabaseName());
        response.put("collections", (Object)this.collections().size());
        long storageSize = this.getStorageSize();
        long fileSize = this.getFileSize();
        long indexSize = 0L;
        int objects = 0;
        double dataSize = 0.0;
        double averageObjectSize = 0.0;
        for (MongoCollection<P> collection : this.collections()) {
            Document stats = collection.getStats();
            objects += ((Number)stats.get("count")).intValue();
            dataSize += ((Number)stats.get("size")).doubleValue();
            Document indexSizes = (Document)stats.get("indexSize");
            for (String indexName : indexSizes.keySet()) {
                indexSize += ((Number)indexSizes.get(indexName)).longValue();
            }
        }
        if (objects > 0) {
            averageObjectSize = dataSize / (double)objects;
        }
        response.put("objects", (Object)objects);
        response.put("avgObjSize", (Object)averageObjectSize);
        if (dataSize == 0.0) {
            response.put("dataSize", (Object)0);
        } else {
            response.put("dataSize", (Object)dataSize);
        }
        response.put("storageSize", (Object)storageSize);
        response.put("numExtents", (Object)0);
        response.put("indexes", (Object)this.countIndexes());
        response.put("indexSize", (Object)indexSize);
        response.put("fileSize", (Object)fileSize);
        response.put("nsSizeMB", (Object)0);
        Utils.markOkay(response);
        return response;
    }

    protected abstract long getFileSize();

    protected abstract long getStorageSize();

    private Document commandDrop(Document query, Oplog oplog) {
        String collectionName = query.get("drop").toString();
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection == null) {
            throw new MongoSilentServerException("ns not found");
        }
        int numIndexes = collection.getNumIndexes();
        this.dropCollection(collectionName, oplog);
        Document response = new Document();
        response.put("nIndexesWas", (Object)numIndexes);
        response.put("ns", (Object)collection.getFullName());
        Utils.markOkay(response);
        return response;
    }

    private Document commandGetLastError(Channel channel, String command, Document query) {
        List writeErrors;
        Document result;
        query.forEach((subCommand, value) -> {
            if (subCommand.equals(command)) {
                return;
            }
            switch (subCommand) {
                case "w": {
                    break;
                }
                case "fsync": {
                    break;
                }
                case "$db": {
                    Assert.equals(value, this.getDatabaseName());
                    break;
                }
                default: {
                    throw new MongoServerException("unknown subcommand: " + subCommand);
                }
            }
        });
        List<Document> results = this.lastResults.get(channel);
        if (results != null && !results.isEmpty()) {
            result = results.get(results.size() - 1);
            if (result == null) {
                result = new Document();
            }
        } else {
            result = new Document();
            result.put("err", (Object)null);
            result.put("n", (Object)0);
        }
        if (result.containsKey("writeErrors") && (writeErrors = (List)result.get("writeErrors")).size() == 1) {
            result.putAll((Map)CollectionUtils.getSingleElement(writeErrors));
            result.remove("writeErrors");
        }
        Utils.markOkay(result);
        return result;
    }

    private Document commandResetError(Channel channel) {
        List<Document> results = this.lastResults.get(channel);
        if (results != null) {
            results.clear();
        }
        Document result = new Document();
        Utils.markOkay(result);
        return result;
    }

    private Document commandCount(String command, Document query) {
        MongoCollection<P> collection = this.resolveCollection(command, query);
        Document response = new Document();
        if (collection == null) {
            response.put("n", (Object)0);
        } else {
            Document queryObject = (Document)query.get("query");
            int limit = this.getOptionalNumber(query, "limit", -1);
            int skip = this.getOptionalNumber(query, "skip", 0);
            response.put("n", (Object)collection.count(queryObject, skip, limit));
        }
        Utils.markOkay(response);
        return response;
    }

    private Document commandAggregate(String command, Document query, Oplog oplog) {
        Document changeStream;
        String collectionName = query.get(command).toString();
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        List<Document> pipelineObject = Aggregation.parse(query.get("pipeline"));
        List<Document> pipeline = Aggregation.parse(pipelineObject);
        if (!pipeline.isEmpty() && (changeStream = (Document)pipeline.get(0).get("$changeStream")) != null) {
            Aggregation aggregation = Aggregation.fromPipeline(pipeline.subList(1, pipeline.size()), (MongoDatabase)this, collection, oplog);
            aggregation.validate(query);
            return this.commandChangeStreamPipeline(query, oplog, collectionName, changeStream, aggregation);
        }
        Aggregation aggregation = Aggregation.fromPipeline(pipeline, (MongoDatabase)this, collection, oplog);
        aggregation.validate(query);
        return Utils.firstBatchCursorResponse(this.getFullCollectionNamespace(collectionName), aggregation.computeResult());
    }

    private Document commandChangeStreamPipeline(Document query, Oplog oplog, String collectionName, Document changeStreamDocument, Aggregation aggregation) {
        Document cursorDocument = (Document)query.get("cursor");
        int batchSize = cursorDocument.getOrDefault("batchSize", 0);
        String namespace = this.getFullCollectionNamespace(collectionName);
        Cursor cursor = oplog.createCursor(changeStreamDocument, namespace, aggregation);
        return Utils.firstBatchCursorResponse(namespace, cursor.takeDocuments(batchSize), cursor);
    }

    private int getOptionalNumber(Document query, String fieldName, int defaultValue) {
        Number limitNumber = (Number)query.get(fieldName);
        return limitNumber != null ? limitNumber.intValue() : defaultValue;
    }

    @Override
    public QueryResult handleQuery(MongoQuery query) {
        this.clearLastStatus(query.getChannel());
        String collectionName = query.getCollectionName();
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection == null) {
            return new QueryResult();
        }
        int numberToSkip = query.getNumberToSkip();
        int batchSize = query.getNumberToReturn();
        if (batchSize < -1) {
            batchSize = -batchSize;
        }
        QueryParameters queryParameters = this.toQueryParameters(query, numberToSkip, batchSize);
        return collection.handleQuery(queryParameters);
    }

    @Override
    public CompletionStage<QueryResult> handleQueryAsync(MongoQuery query) {
        this.clearLastStatus(query.getChannel());
        String collectionName = query.getCollectionName();
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection == null) {
            return FutureUtils.wrap(QueryResult::new);
        }
        int numberToSkip = query.getNumberToSkip();
        int batchSize = query.getNumberToReturn();
        if (batchSize < -1) {
            batchSize = -batchSize;
        }
        QueryParameters queryData = this.toQueryParameters(query, numberToSkip, batchSize);
        return collection.handleQueryAsync(queryData);
    }

    @Override
    public void handleClose(Channel channel) {
        this.lastResults.remove(channel);
    }

    protected void clearLastStatus(Channel channel) {
        List results = this.lastResults.computeIfAbsent(channel, k -> new LimitedList(10));
        results.add(null);
    }

    @Override
    public void handleInsert(MongoInsert insert, Oplog oplog) {
        Channel channel = insert.getChannel();
        String collectionName = insert.getCollectionName();
        List<Document> documents = insert.getDocuments();
        if (collectionName.equals(INDEXES_COLLECTION_NAME)) {
            for (Document indexDescription : documents) {
                this.addIndex(indexDescription);
            }
        } else {
            try {
                this.insertDocuments(channel, collectionName, documents, oplog, true);
            }
            catch (MongoServerException e) {
                log.error("failed to insert {}", (Object)insert, (Object)e);
            }
        }
    }

    private MongoCollection<P> resolveCollection(String command, Document query) {
        String collectionName = query.get(command).toString();
        return this.resolveCollection(collectionName, false);
    }

    public MongoCollection<P> resolveCollection(String collectionName, boolean throwIfNotFound) {
        this.checkCollectionName(collectionName);
        MongoCollection<P> collection = this.collections.get(collectionName);
        if (collection == null && throwIfNotFound) {
            throw new MongoServerException("Collection [" + this.getFullCollectionNamespace(collectionName) + "] not found.");
        }
        return collection;
    }

    private void checkCollectionName(String collectionName) {
        if (collectionName.length() > 128) {
            throw new MongoServerError(10080, "ns name too long, max size is 128");
        }
        if (collectionName.isEmpty()) {
            throw new MongoServerError(16256, "Invalid ns [" + collectionName + "]");
        }
    }

    @Override
    public boolean isEmpty() {
        return this.collections.isEmpty();
    }

    private void addNamespace(MongoCollection<P> collection) {
        this.collections.put(collection.getCollectionName(), collection);
        if (!AbstractMongoDatabase.isSystemCollection(collection.getCollectionName())) {
            this.namespaces.addDocument(new Document("name", collection.getFullName()));
        }
    }

    @Override
    public void handleDelete(MongoDelete delete, Oplog oplog) {
        Channel channel = delete.getChannel();
        String collectionName = delete.getCollectionName();
        Document selector = delete.getSelector();
        int limit = delete.isSingleRemove() ? 1 : Integer.MAX_VALUE;
        try {
            this.deleteDocuments(channel, collectionName, selector, limit, oplog);
        }
        catch (MongoServerException e) {
            log.error("failed to delete {}", (Object)delete, (Object)e);
        }
    }

    @Override
    public void handleUpdate(MongoUpdate updateCommand, Oplog oplog) {
        Channel channel = updateCommand.getChannel();
        String collectionName = updateCommand.getCollectionName();
        Document selector = updateCommand.getSelector();
        Document update = updateCommand.getUpdate();
        boolean multi = updateCommand.isMulti();
        boolean upsert = updateCommand.isUpsert();
        ArrayFilters arrayFilters = ArrayFilters.empty();
        this.clearLastStatus(channel);
        try {
            Document result = this.updateDocuments(collectionName, selector, update, arrayFilters, multi, upsert, oplog);
            this.putLastResult(channel, result);
        }
        catch (MongoServerException e) {
            this.putLastError(channel, e);
            log.error("failed to update {}", (Object)updateCommand, (Object)e);
        }
    }

    protected void addIndex(Document indexDescription) {
        if (!indexDescription.containsKey("v")) {
            indexDescription.put("v", (Object)2);
        }
        this.openOrCreateIndex(indexDescription);
    }

    protected MongoCollection<P> getOrCreateIndexesCollection() {
        if (this.indexes.get() == null) {
            MongoCollection<P> indexCollection = this.openOrCreateCollection(INDEXES_COLLECTION_NAME, CollectionOptions.withoutIdField());
            this.addNamespace(indexCollection);
            this.indexes.set(indexCollection);
        }
        return this.indexes.get();
    }

    private String extractCollectionNameFromNamespace(String namespace) {
        Assert.startsWith(namespace, this.databaseName);
        return namespace.substring(this.databaseName.length() + 1);
    }

    private void openOrCreateIndex(Document indexDescription) {
        String ns = indexDescription.get("ns").toString();
        String collectionName = this.extractCollectionNameFromNamespace(ns);
        MongoCollection<P> collection = this.resolveOrCreateCollection(collectionName);
        Index<P> index = this.openOrCreateIndex(collectionName, indexDescription);
        MongoCollection<P> indexesCollection = this.getOrCreateIndexesCollection();
        if (index != null) {
            collection.addIndex(index);
            indexesCollection.addDocumentIfMissing(indexDescription);
        }
    }

    private Index<P> openOrCreateIndex(String collectionName, Document indexDescription) {
        String indexName = (String)indexDescription.get("name");
        Document key = (Document)indexDescription.get("key");
        if (this.isPrimaryKeyIndex(key)) {
            if (!indexName.equals("_id_")) {
                log.warn("Ignoring primary key index with name '{}'", (Object)indexName);
                return null;
            }
            boolean ascending = AbstractMongoDatabase.isAscending(key.get("_id"));
            Index<P> index = this.openOrCreateIdIndex(collectionName, indexName, ascending);
            log.info("adding unique _id index for collection {}", (Object)collectionName);
            return index;
        }
        ArrayList<IndexKey> keys = new ArrayList<IndexKey>();
        for (Map.Entry<String, Object> entry : key.entrySet()) {
            String field = entry.getKey();
            boolean ascending = AbstractMongoDatabase.isAscending(entry.getValue());
            keys.add(new IndexKey(field, ascending));
        }
        boolean sparse = Utils.isTrue(indexDescription.get("sparse"));
        if (Utils.isTrue(indexDescription.get("unique"))) {
            log.info("adding {} unique index {} for collection {}", new Object[]{sparse ? "sparse" : "non-sparse", keys, collectionName});
            return this.openOrCreateUniqueIndex(collectionName, indexName, keys, sparse);
        }
        return this.openOrCreateSecondaryIndex(collectionName, indexName, keys, sparse);
    }

    protected boolean isPrimaryKeyIndex(Document key) {
        return key.keySet().equals(Collections.singleton("_id"));
    }

    protected Index<P> openOrCreateSecondaryIndex(String collectionName, String indexName, List<IndexKey> keys, boolean sparse) {
        log.warn("adding secondary index with keys {} is not yet implemented. ignoring", keys);
        return new EmptyIndex(indexName, keys);
    }

    private static boolean isAscending(Object keyValue) {
        return Objects.equals(Utils.normalizeValue(keyValue), 1.0);
    }

    private Index<P> openOrCreateIdIndex(String collectionName, String indexName, boolean ascending) {
        return this.openOrCreateUniqueIndex(collectionName, indexName, Collections.singletonList(new IndexKey("_id", ascending)), false);
    }

    protected abstract Index<P> openOrCreateUniqueIndex(String var1, String var2, List<IndexKey> var3, boolean var4);

    private List<Document> insertDocuments(Channel channel, String collectionName, List<Document> documents, Oplog oplog, boolean isOrdered) {
        this.clearLastStatus(channel);
        try {
            if (AbstractMongoDatabase.isSystemCollection(collectionName)) {
                throw new MongoServerError(16459, "attempt to insert in system namespace");
            }
            MongoCollection<P> collection = this.resolveOrCreateCollection(collectionName);
            List<Document> writeErrors = collection.insertDocuments(documents, isOrdered);
            oplog.handleInsert(collection.getFullName(), documents);
            if (!writeErrors.isEmpty()) {
                Document writeError = new Document(writeErrors.get(0));
                writeError.put("err", writeError.remove("errmsg"));
                this.putLastResult(channel, writeError);
            } else {
                Document result = new Document("n", 0);
                result.put("err", (Object)null);
                this.putLastResult(channel, result);
            }
            return writeErrors;
        }
        catch (MongoServerError e) {
            this.putLastError(channel, e);
            return Collections.singletonList(this.toWriteError(0, e));
        }
    }

    private Document deleteDocuments(Channel channel, String collectionName, Document selector, int limit, Oplog oplog) {
        this.clearLastStatus(channel);
        try {
            if (AbstractMongoDatabase.isSystemCollection(collectionName)) {
                throw new MongoServerError(73, "InvalidNamespace", "cannot write to '" + this.getFullCollectionNamespace(collectionName) + "'");
            }
            MongoCollection<P> collection = this.resolveCollection(collectionName, false);
            int n = collection == null ? 0 : collection.deleteDocuments(selector, limit, oplog);
            Document result = new Document("n", n);
            this.putLastResult(channel, result);
            return result;
        }
        catch (MongoServerError e) {
            this.putLastError(channel, e);
            throw e;
        }
    }

    private Document updateDocuments(String collectionName, Document selector, Document update, ArrayFilters arrayFilters, boolean multi, boolean upsert, Oplog oplog) {
        if (AbstractMongoDatabase.isSystemCollection(collectionName)) {
            throw new MongoServerError(10156, "cannot update system collection");
        }
        MongoCollection<P> collection = this.resolveOrCreateCollection(collectionName);
        return collection.updateDocuments(selector, update, arrayFilters, multi, upsert, oplog);
    }

    private void putLastError(Channel channel, MongoServerException ex) {
        Document error = this.toError(channel, ex);
        this.putLastResult(channel, error);
    }

    private Document toWriteError(int index, MongoServerException e) {
        Document error = new Document();
        error.put("index", (Object)index);
        error.put("errmsg", (Object)e.getMessageWithoutErrorCode());
        if (e instanceof MongoServerError) {
            MongoServerError err = (MongoServerError)e;
            error.put("code", (Object)err.getCode());
            error.putIfNotNull("codeName", err.getCodeName());
        }
        return error;
    }

    private Document toError(Channel channel, MongoServerException ex) {
        Document error = new Document();
        error.put("err", (Object)ex.getMessageWithoutErrorCode());
        if (ex instanceof MongoServerError) {
            MongoServerError err = (MongoServerError)ex;
            error.put("code", (Object)err.getCode());
            error.putIfNotNull("codeName", err.getCodeName());
        }
        error.put("connectionId", (Object)channel.id().asShortText());
        return error;
    }

    protected void putLastResult(Channel channel, Document result) {
        List<Document> results = this.lastResults.get(channel);
        Document last = results.get(results.size() - 1);
        Assert.isNull(last, () -> "last result already set: " + last);
        results.set(results.size() - 1, result);
    }

    private MongoCollection<P> createCollection(String collectionName, CollectionOptions options) {
        this.checkCollectionName(collectionName);
        if (collectionName.contains("$")) {
            throw new MongoServerError(10093, "cannot insert into reserved $ collection");
        }
        MongoCollection<P> collection = this.openOrCreateCollection(collectionName, options);
        this.addNamespace(collection);
        this.addIndex(AbstractMongoDatabase.getPrimaryKeyIndexDescription(collection.getFullName()));
        log.info("created collection {}", (Object)collection.getFullName());
        return collection;
    }

    protected abstract MongoCollection<P> openOrCreateCollection(String var1, CollectionOptions var2);

    @Override
    public void drop(Oplog oplog) {
        log.debug("dropping {}", (Object)this);
        for (String collectionName : this.collections.keySet()) {
            if (AbstractMongoDatabase.isSystemCollection(collectionName)) continue;
            this.dropCollection(collectionName, oplog);
        }
        this.dropCollectionIfExists(INDEXES_COLLECTION_NAME, oplog);
        this.dropCollectionIfExists(NAMESPACES_COLLECTION_NAME, oplog);
    }

    private void dropCollectionIfExists(String collectionName, Oplog oplog) {
        if (this.collections.containsKey(collectionName)) {
            this.dropCollection(collectionName, oplog);
        }
    }

    @Override
    public void dropCollection(String collectionName, Oplog oplog) {
        MongoCollection<P> collection = this.resolveCollection(collectionName, true);
        this.dropAllIndexes(collection);
        collection.drop();
        this.unregisterCollection(collectionName);
        oplog.handleDropCollection(String.format("%s.%s", this.databaseName, collectionName));
    }

    private void dropAllIndexes(MongoCollection<P> collection) {
        MongoCollection<P> indexCollection = this.indexes.get();
        if (indexCollection == null) {
            return;
        }
        ArrayList<Document> indexesToDrop = new ArrayList<Document>();
        for (Document index : indexCollection.handleQuery(new Document("ns", collection.getFullName()))) {
            indexesToDrop.add(index);
        }
        for (Document indexToDrop : indexesToDrop) {
            this.dropIndex(collection, indexToDrop);
        }
    }

    @Override
    public void unregisterCollection(String collectionName) {
        MongoCollection<P> removedCollection = this.collections.remove(collectionName);
        this.namespaces.deleteDocuments(new Document("name", removedCollection.getFullName()), 1);
    }

    @Override
    public void moveCollection(MongoDatabase oldDatabase, MongoCollection<?> collection, String newCollectionName) {
        String oldFullName = collection.getFullName();
        oldDatabase.unregisterCollection(collection.getCollectionName());
        collection.renameTo(this, newCollectionName);
        MongoCollection<?> newCollection = collection;
        MongoCollection<?> oldCollection = this.collections.put(newCollectionName, newCollection);
        Assert.isNull(oldCollection, () -> "Failed to register renamed collection. Another collection still existed: " + oldCollection);
        ArrayList<Document> newDocuments = new ArrayList<Document>();
        newDocuments.add(new Document("name", collection.getFullName()));
        this.indexes.get().updateDocuments(new Document("ns", oldFullName), new Document("$set", new Document("ns", newCollection.getFullName())), ArrayFilters.empty(), true, false, NoopOplog.get());
        this.namespaces.insertDocuments(newDocuments, true);
    }

    protected String getFullCollectionNamespace(String collectionName) {
        return this.getDatabaseName() + "." + collectionName;
    }

    static boolean isSystemCollection(String collectionName) {
        return collectionName.startsWith("system.");
    }
}

