/*
 * Decompiled with CFR 0.152.
 */
package de.bwaldvogel.mongo.backend;

import de.bwaldvogel.mongo.MongoBackend;
import de.bwaldvogel.mongo.MongoCollection;
import de.bwaldvogel.mongo.MongoDatabase;
import de.bwaldvogel.mongo.MongoVersion;
import de.bwaldvogel.mongo.ServerVersion;
import de.bwaldvogel.mongo.backend.Cursor;
import de.bwaldvogel.mongo.backend.CursorRegistry;
import de.bwaldvogel.mongo.backend.EmptyCursor;
import de.bwaldvogel.mongo.backend.QueryResult;
import de.bwaldvogel.mongo.backend.Utils;
import de.bwaldvogel.mongo.bson.Document;
import de.bwaldvogel.mongo.exception.MongoServerException;
import de.bwaldvogel.mongo.exception.MongoSilentServerException;
import de.bwaldvogel.mongo.exception.NamespaceExistsException;
import de.bwaldvogel.mongo.exception.NoReplicationEnabledException;
import de.bwaldvogel.mongo.exception.NoSuchCommandException;
import de.bwaldvogel.mongo.oplog.CollectionBackedOplog;
import de.bwaldvogel.mongo.oplog.NoopOplog;
import de.bwaldvogel.mongo.oplog.Oplog;
import de.bwaldvogel.mongo.wire.message.Message;
import de.bwaldvogel.mongo.wire.message.MongoMessage;
import de.bwaldvogel.mongo.wire.message.MongoQuery;
import io.netty.channel.Channel;
import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractMongoBackend
implements MongoBackend {
    private static final Logger log = LoggerFactory.getLogger(AbstractMongoBackend.class);
    protected static final String OPLOG_COLLECTION_NAME = "oplog.rs";
    static final String ADMIN_DB_NAME = "admin";
    private final Map<String, MongoDatabase> databases = new ConcurrentHashMap<String, MongoDatabase>();
    private MongoVersion version = ServerVersion.MONGO_4_0;
    private final Clock clock;
    private final Instant started;
    private final CursorRegistry cursorRegistry = new CursorRegistry();
    protected Oplog oplog = NoopOplog.get();

    protected AbstractMongoBackend() {
        this(AbstractMongoBackend.defaultClock());
    }

    protected AbstractMongoBackend(Clock clock) {
        this.started = Instant.now(clock);
        this.clock = clock;
    }

    protected static Clock defaultClock() {
        return Clock.systemDefaultZone();
    }

    private MongoDatabase resolveDatabase(Message message) {
        return this.resolveDatabase(message.getDatabaseName());
    }

    @Override
    public MongoDatabase resolveDatabase(String databaseName) {
        return this.databases.computeIfAbsent(databaseName, name -> {
            MongoDatabase database = this.openOrCreateDatabase(databaseName);
            log.info("created database {}", (Object)database.getDatabaseName());
            return database;
        });
    }

    @Override
    public Document getServerStatus() {
        Document serverStatus = new Document();
        try {
            serverStatus.put("host", (Object)InetAddress.getLocalHost().getHostName());
        }
        catch (UnknownHostException e) {
            throw new MongoServerException("failed to get hostname", e);
        }
        serverStatus.put("version", (Object)this.version.toVersionString());
        serverStatus.put("process", (Object)"java");
        serverStatus.put("pid", (Object)this.getProcessId());
        Duration uptime = Duration.between(this.started, Instant.now(this.clock));
        serverStatus.put("uptime", (Object)Math.toIntExact(uptime.getSeconds()));
        serverStatus.put("uptimeMillis", (Object)uptime.toMillis());
        serverStatus.put("localTime", (Object)Instant.now(this.getClock()));
        Document connections = new Document();
        connections.put("current", (Object)1);
        serverStatus.put("connections", (Object)connections);
        Document metrics = new Document();
        Document cursorMetrics = new Document();
        cursorMetrics.put("timedOut", (Object)0L);
        Document openCursors = new Document();
        openCursors.put("noTimeout", (Object)0L);
        openCursors.put("pinned", (Object)0L);
        openCursors.put("total", (Object)this.cursorRegistry.size());
        cursorMetrics.put("open", (Object)openCursors);
        metrics.put("cursor", (Object)cursorMetrics);
        serverStatus.put("metrics", (Object)metrics);
        Utils.markOkay(serverStatus);
        return serverStatus;
    }

    private Integer getProcessId() {
        String runtimeName = ManagementFactory.getRuntimeMXBean().getName();
        if (runtimeName.contains("@")) {
            return Integer.valueOf(runtimeName.substring(0, runtimeName.indexOf(64)));
        }
        return 0;
    }

    private Document getLog(String argument) {
        log.debug("getLog: {}", (Object)argument);
        Document response = new Document();
        switch (argument) {
            case "*": {
                response.put("names", (Object)List.of("startupWarnings"));
                Utils.markOkay(response);
                break;
            }
            case "startupWarnings": {
                response.put("totalLinesWritten", (Object)0);
                response.put("log", (Object)new ArrayList());
                Utils.markOkay(response);
                break;
            }
            default: {
                throw new MongoSilentServerException("no RamLog named: " + argument);
            }
        }
        return response;
    }

    private Document handleAdminCommand(String command, Document query) {
        if (command.equalsIgnoreCase("listdatabases")) {
            List databases = this.listDatabaseNames().stream().sorted().map(databaseName -> {
                MongoDatabase database = this.openOrCreateDatabase((String)databaseName);
                Document dbObj = new Document("name", database.getDatabaseName());
                dbObj.put("empty", (Object)database.isEmpty());
                return dbObj;
            }).collect(Collectors.toList());
            Document response = new Document();
            response.put("databases", (Object)databases);
            Utils.markOkay(response);
            return response;
        }
        if (command.equalsIgnoreCase("find")) {
            String collectionName = (String)query.get(command);
            if (collectionName.equals("$cmd.sys.inprog")) {
                return Utils.firstBatchCursorResponse(collectionName, new Document("inprog", Collections.emptyList()));
            }
            throw new NoSuchCommandException(new Document(command, collectionName).toString());
        }
        if (command.equalsIgnoreCase("replSetGetStatus")) {
            throw new NoReplicationEnabledException();
        }
        if (command.equalsIgnoreCase("getLog")) {
            Object argument = query.get(command);
            return this.getLog(argument == null ? null : argument.toString());
        }
        if (command.equalsIgnoreCase("renameCollection")) {
            return this.handleRenameCollection(command, query);
        }
        if (command.equalsIgnoreCase("getLastError")) {
            log.debug("getLastError on admin database");
            return AbstractMongoBackend.successResponse();
        }
        if (command.equalsIgnoreCase("connectionStatus")) {
            Document response = new Document();
            response.append("authInfo", new Document().append("authenticatedUsers", Collections.emptyList()).append("authenticatedUserRoles", Collections.emptyList()));
            Utils.markOkay(response);
            return response;
        }
        if (command.equalsIgnoreCase("hostInfo")) {
            return this.handleHostInfo();
        }
        if (command.equalsIgnoreCase("getCmdLineOpts")) {
            return this.handleGetCmdLineOpts();
        }
        if (command.equalsIgnoreCase("getFreeMonitoringStatus")) {
            return this.handleGetFreeMonitoringStatus();
        }
        if (command.equalsIgnoreCase("endSessions")) {
            log.debug("endSessions on admin database");
            return AbstractMongoBackend.successResponse();
        }
        throw new NoSuchCommandException(command);
    }

    private static Document successResponse() {
        Document response = new Document();
        Utils.markOkay(response);
        return response;
    }

    private Document handleHostInfo() {
        Document response = new Document();
        String osName = System.getProperty("os.name");
        String osVersion = System.getProperty("os.version");
        response.append("os", new Document().append("type", osName).append("version", osVersion));
        response.append("system", new Document().append("currentTime", Instant.now()).append("hostname", Utils.getHostName()));
        response.append("extra", new Document().append("versionString", osName + " " + osVersion).append("kernelVersion", osVersion));
        Utils.markOkay(response);
        return response;
    }

    private Document handleGetCmdLineOpts() {
        Document response = new Document();
        response.append("argv", Collections.emptyList());
        response.append("parsed", new Document());
        Utils.markOkay(response);
        return response;
    }

    private Document handleGetFreeMonitoringStatus() {
        Document response = new Document();
        response.append("state", "disabled");
        response.append("debug", Map.of("state", "undecided"));
        response.append("message", "Free monitoring is deprecated, refer to 'debug' field for actual status");
        Utils.markOkay(response);
        return response;
    }

    protected Set<String> listDatabaseNames() {
        return this.databases.keySet();
    }

    private Document handleRenameCollection(String command, Document query) {
        String oldNamespace = query.get(command).toString();
        String newNamespace = query.get("to").toString();
        boolean dropTarget = Utils.isTrue(query.get("dropTarget"));
        Document response = new Document();
        if (!oldNamespace.equals(newNamespace)) {
            MongoCollection<?> oldCollection = this.resolveCollection(oldNamespace);
            if (oldCollection == null) {
                throw new MongoServerException("source namespace does not exist");
            }
            String newDatabaseName = Utils.getDatabaseNameFromFullName(newNamespace);
            String newCollectionName = Utils.getCollectionNameFromFullName(newNamespace);
            MongoDatabase oldDatabase = this.resolveDatabase(oldCollection.getDatabaseName());
            MongoDatabase newDatabase = this.resolveDatabase(newDatabaseName);
            MongoCollection<?> newCollection = newDatabase.resolveCollection(newCollectionName, false);
            if (newCollection != null) {
                if (dropTarget) {
                    newDatabase.dropCollection(newCollectionName, this.oplog);
                } else {
                    throw new NamespaceExistsException("target namespace exists");
                }
            }
            newDatabase.moveCollection(oldDatabase, oldCollection, newCollectionName);
        }
        Utils.markOkay(response);
        return response;
    }

    private MongoCollection<?> resolveCollection(String namespace) {
        String databaseName = Utils.getDatabaseNameFromFullName(namespace);
        String collectionName = Utils.getCollectionNameFromFullName(namespace);
        MongoDatabase database = this.databases.get(databaseName);
        if (database == null) {
            return null;
        }
        return database.resolveCollection(collectionName, false);
    }

    protected abstract MongoDatabase openOrCreateDatabase(String var1);

    @Override
    public Document handleCommand(Channel channel, String databaseName, String command, Document query) {
        if (command.equalsIgnoreCase("whatsmyuri")) {
            Document response = new Document();
            InetSocketAddress remoteAddress = (InetSocketAddress)channel.remoteAddress();
            response.put("you", (Object)(remoteAddress.getAddress().getHostAddress() + ":" + remoteAddress.getPort()));
            Utils.markOkay(response);
            return response;
        }
        if (command.equalsIgnoreCase("ismaster")) {
            Document response = new Document("ismaster", Boolean.TRUE);
            response.put("maxBsonObjectSize", (Object)0x1000000);
            response.put("maxWriteBatchSize", (Object)1000);
            response.put("maxMessageSizeBytes", (Object)48000000);
            response.put("maxWireVersion", (Object)this.version.getWireVersion());
            response.put("minWireVersion", (Object)0);
            response.put("localTime", (Object)Instant.now(this.clock));
            Utils.markOkay(response);
            return response;
        }
        if (command.equalsIgnoreCase("buildinfo")) {
            Document response = new Document("version", this.version.toVersionString());
            response.put("versionArray", (Object)this.version.getVersionArray());
            response.put("maxBsonObjectSize", (Object)0x1000000);
            Utils.markOkay(response);
            return response;
        }
        if (command.equalsIgnoreCase("dropDatabase")) {
            return this.handleDropDatabase(databaseName);
        }
        if (command.equalsIgnoreCase("getMore")) {
            return this.handleGetMore(databaseName, command, query);
        }
        if (command.equalsIgnoreCase("killCursors")) {
            return this.handleKillCursors(query);
        }
        if (command.equalsIgnoreCase("ping")) {
            return AbstractMongoBackend.successResponse();
        }
        if (command.equalsIgnoreCase("serverStatus")) {
            return this.getServerStatus();
        }
        if (databaseName.equals(ADMIN_DB_NAME)) {
            return this.handleAdminCommand(command, query);
        }
        MongoDatabase mongoDatabase = this.resolveDatabase(databaseName);
        return mongoDatabase.handleCommand(channel, command, query, this::resolveDatabase, this.oplog);
    }

    @Override
    public Collection<Document> getCurrentOperations(MongoQuery query) {
        return Collections.emptyList();
    }

    @Override
    public QueryResult handleQuery(MongoQuery query) {
        return this.resolveDatabase(query).handleQuery(query);
    }

    @Override
    public void closeCursors(List<Long> cursorIds) {
        cursorIds.forEach(this.cursorRegistry::remove);
    }

    protected Document handleKillCursors(Document query) {
        List cursorIds = (List)query.get("cursors");
        ArrayList<Long> cursorsKilled = new ArrayList<Long>();
        ArrayList<Long> cursorsNotFound = new ArrayList<Long>();
        for (Long cursorId : cursorIds) {
            if (this.cursorRegistry.remove(cursorId)) {
                log.info("Killed cursor {}", (Object)cursorId);
                cursorsKilled.add(cursorId);
                continue;
            }
            log.info("Cursor {} not found", (Object)cursorId);
            cursorsNotFound.add(cursorId);
        }
        Document response = new Document();
        response.put("cursorsKilled", (Object)cursorsKilled);
        response.put("cursorsNotFound", (Object)cursorsNotFound);
        Utils.markOkay(response);
        return response;
    }

    protected Document handleGetMore(String databaseName, String command, Document query) {
        MongoDatabase mongoDatabase = this.resolveDatabase(databaseName);
        String collectionName = (String)query.get("collection");
        long cursorId = ((Number)query.get(command)).longValue();
        int batchSize = ((Number)query.getOrDefault("batchSize", 0)).intValue();
        QueryResult queryResult = this.handleGetMore(cursorId, batchSize);
        List<Document> nextBatch = queryResult.collectDocuments();
        String fullCollectionName = databaseName + "." + collectionName;
        return Utils.nextBatchCursorResponse(fullCollectionName, nextBatch, queryResult.getCursorId());
    }

    private QueryResult handleGetMore(long cursorId, int numberToReturn) {
        Cursor cursor = this.cursorRegistry.getCursor(cursorId);
        List<Document> documents = cursor.takeDocuments(numberToReturn);
        if (cursor.isEmpty()) {
            log.debug("Removing empty {}", (Object)cursor);
            this.cursorRegistry.remove(cursor);
        }
        return new QueryResult(documents, cursor.isEmpty() ? EmptyCursor.get().getId() : cursorId);
    }

    protected Document handleDropDatabase(String databaseName) {
        this.dropDatabase(databaseName);
        Document response = new Document("dropped", databaseName);
        Utils.markOkay(response);
        return response;
    }

    @Override
    public Document handleMessage(MongoMessage message) {
        Channel channel = message.getChannel();
        String databaseName = message.getDatabaseName();
        Document query = message.getDocument();
        String command = query.keySet().iterator().next();
        return this.handleCommand(channel, databaseName, command, query);
    }

    @Override
    public void dropDatabase(String databaseName) {
        MongoDatabase removedDatabase = this.databases.remove(databaseName);
        if (removedDatabase != null) {
            removedDatabase.drop(this.oplog);
        }
    }

    @Override
    public void handleClose(Channel channel) {
        for (MongoDatabase db : this.databases.values()) {
            db.handleClose(channel);
        }
    }

    @Override
    public void close() {
        log.info("closing {}", (Object)this);
        this.databases.clear();
    }

    @Override
    public MongoBackend version(MongoVersion version) {
        this.version = version;
        return this;
    }

    @Override
    public Clock getClock() {
        return this.clock;
    }

    @Override
    public void disableOplog() {
        this.oplog = NoopOplog.get();
    }

    @Override
    public void enableOplog() {
        this.oplog = this.createOplog();
    }

    protected Oplog createOplog() {
        MongoDatabase localDatabase = this.resolveDatabase("local");
        MongoCollection<Document> collection = localDatabase.resolveCollection(OPLOG_COLLECTION_NAME, false);
        if (collection == null) {
            collection = localDatabase.createCollectionOrThrowIfExists(OPLOG_COLLECTION_NAME);
        }
        return new CollectionBackedOplog(this, collection, this.cursorRegistry);
    }

    protected CursorRegistry getCursorRegistry() {
        return this.cursorRegistry;
    }
}

