/*
 * Decompiled with CFR 0.152.
 */
package de.bwaldvogel.mongo.backend.memory;

import com.mongodb.BasicDBObject;
import de.bwaldvogel.mongo.backend.LimitedList;
import de.bwaldvogel.mongo.backend.MongoCollection;
import de.bwaldvogel.mongo.backend.Utils;
import de.bwaldvogel.mongo.backend.memory.CommonDatabase;
import de.bwaldvogel.mongo.backend.memory.MemoryBackend;
import de.bwaldvogel.mongo.backend.memory.MemoryCollection;
import de.bwaldvogel.mongo.backend.memory.MemoryIndexesCollection;
import de.bwaldvogel.mongo.backend.memory.MemoryNamespacesCollection;
import de.bwaldvogel.mongo.backend.memory.index.UniqueIndex;
import de.bwaldvogel.mongo.exception.MongoServerError;
import de.bwaldvogel.mongo.exception.MongoServerException;
import de.bwaldvogel.mongo.exception.MongoSilentServerException;
import de.bwaldvogel.mongo.exception.NoSuchCommandException;
import de.bwaldvogel.mongo.wire.message.MongoDelete;
import de.bwaldvogel.mongo.wire.message.MongoInsert;
import de.bwaldvogel.mongo.wire.message.MongoQuery;
import de.bwaldvogel.mongo.wire.message.MongoUpdate;
import io.netty.channel.Channel;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.bson.BSONObject;
import org.bson.BasicBSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MemoryDatabase
extends CommonDatabase {
    private static final Logger log = LoggerFactory.getLogger(MemoryDatabase.class);
    private Map<String, MongoCollection> collections = new HashMap<String, MongoCollection>();
    private Map<Channel, List<BSONObject>> lastResults = new HashMap<Channel, List<BSONObject>>();
    private MemoryBackend backend;
    private MongoCollection namespaces;
    private MongoCollection indexes;

    public MemoryDatabase(MemoryBackend backend, String databaseName) throws MongoServerException {
        super(databaseName);
        this.backend = backend;
        this.namespaces = new MemoryNamespacesCollection(this.getDatabaseName());
        this.collections.put(this.namespaces.getCollectionName(), this.namespaces);
        this.indexes = new MemoryIndexesCollection(this.getDatabaseName());
        this.addNamespace(this.indexes);
    }

    private synchronized MongoCollection resolveCollection(String collectionName, boolean throwIfNotFound) throws MongoServerException {
        this.checkCollectionName(collectionName);
        MongoCollection collection = this.collections.get(collectionName);
        if (collection == null && throwIfNotFound) {
            throw new MongoServerException("ns not found");
        }
        return collection;
    }

    private void checkCollectionName(String collectionName) throws MongoServerException {
        if (collectionName.length() > 128) {
            throw new MongoServerError(10080, "ns name too long, max size is 128");
        }
        if (collectionName.isEmpty()) {
            throw new MongoServerError(16256, "Invalid ns [" + collectionName + "]");
        }
    }

    @Override
    public boolean isEmpty() {
        return this.collections.isEmpty();
    }

    @Override
    public Iterable<BSONObject> handleQuery(MongoQuery query) throws MongoServerException {
        this.clearLastStatus(query.getChannel());
        String collectionName = query.getCollectionName();
        MongoCollection collection = this.resolveCollection(collectionName, false);
        if (collection == null) {
            return Collections.emptyList();
        }
        return collection.handleQuery(query.getQuery(), query.getNumberToSkip(), query.getNumberToReturn(), query.getReturnFieldSelector());
    }

    @Override
    public void handleClose(Channel channel) {
        this.lastResults.remove(channel);
    }

    protected synchronized void clearLastStatus(Channel channel) {
        List<BSONObject> results = this.lastResults.get(channel);
        if (results == null) {
            results = new LimitedList<BSONObject>(10);
            this.lastResults.put(channel, results);
        }
        results.add(null);
    }

    @Override
    public void handleInsert(MongoInsert insert) throws MongoServerException {
        Channel channel = insert.getChannel();
        this.clearLastStatus(channel);
        try {
            String collectionName = insert.getCollectionName();
            if (collectionName.equals(this.indexes.getCollectionName())) {
                for (BSONObject indexDescription : insert.getDocuments()) {
                    this.addIndex(indexDescription);
                }
                return;
            }
            if (collectionName.startsWith("system.")) {
                throw new MongoServerError(16459, "attempt to insert in system namespace");
            }
            MongoCollection collection = this.resolveCollection(collectionName, false);
            if (collection == null) {
                collection = this.createCollection(collectionName);
            }
            collection.handleInsert(insert);
            this.putLastResult(channel, (BSONObject)new BasicBSONObject("n", (Object)0));
        }
        catch (MongoServerError e) {
            log.error("failed to insert {}", (Object)insert, (Object)e);
            this.putLastError(channel, e);
        }
    }

    private void addIndex(BSONObject indexDescription) throws MongoServerException {
        int index;
        String ns = indexDescription.get("ns").toString();
        String collectionName = ns.substring((index = ns.indexOf(46)) + 1);
        MongoCollection collection = this.resolveCollection(collectionName, false);
        if (collection == null) {
            collection = this.createCollection(collectionName);
        }
        this.indexes.addDocument(indexDescription);
        BSONObject key = (BSONObject)indexDescription.get("key");
        if (key.keySet().equals(Collections.singleton("_id"))) {
            boolean ascending = Utils.normalizeValue(key.get("_id")).equals(1.0);
            collection.addIndex(new UniqueIndex("_id", ascending));
            log.info("adding unique _id index for collection {}", (Object)collectionName);
        } else if (Utils.isTrue(indexDescription.get("unique"))) {
            if (key.keySet().size() != 1) {
                throw new MongoServerException("Compound unique indices are not yet implemented");
            }
            log.info("adding unique index {} for collection {}", (Object)key.keySet(), (Object)collectionName);
            String field = (String)key.keySet().iterator().next();
            boolean ascending = Utils.normalizeValue(key.get(field)).equals(1.0);
            collection.addIndex(new UniqueIndex(field, ascending));
        } else {
            log.warn("adding non-unique non-id index with key {} is not yet implemented", (Object)key);
        }
    }

    private MongoCollection createCollection(String collectionName) throws MongoServerException {
        this.checkCollectionName(collectionName);
        if (collectionName.contains("$")) {
            throw new MongoServerError(10093, "cannot insert into reserved $ collection");
        }
        MemoryCollection collection = new MemoryCollection(this.getDatabaseName(), collectionName, "_id");
        this.addNamespace(collection);
        BasicBSONObject indexDescription = new BasicBSONObject();
        indexDescription.put("name", (Object)"_id_");
        indexDescription.put("ns", (Object)collection.getFullName());
        indexDescription.put("key", (Object)new BasicBSONObject("_id", (Object)1));
        this.addIndex((BSONObject)indexDescription);
        log.info("created collection {}", (Object)collection.getFullName());
        return collection;
    }

    protected void addNamespace(MongoCollection collection) throws MongoServerException {
        this.collections.put(collection.getCollectionName(), collection);
        this.namespaces.addDocument((BSONObject)new BasicDBObject("name", (Object)collection.getFullName()));
    }

    @Override
    public void handleDelete(MongoDelete delete) throws MongoServerException {
        Channel channel = delete.getChannel();
        this.clearLastStatus(channel);
        try {
            String collectionName = delete.getCollectionName();
            if (collectionName.startsWith("system.")) {
                throw new MongoServerError(12050, "cannot delete from system namespace");
            }
            MongoCollection collection = this.resolveCollection(collectionName, false);
            int n = collection == null ? 0 : collection.handleDelete(delete);
            this.putLastResult(channel, (BSONObject)new BasicBSONObject("n", (Object)n));
        }
        catch (MongoServerError e) {
            log.error("failed to delete {}", (Object)delete, (Object)e);
            this.putLastError(channel, e);
        }
    }

    @Override
    public void handleUpdate(MongoUpdate update) throws MongoServerException {
        Channel channel = update.getChannel();
        this.clearLastStatus(channel);
        try {
            String collectionName = update.getCollectionName();
            if (collectionName.startsWith("system.")) {
                throw new MongoServerError(10156, "cannot update system collection");
            }
            MongoCollection collection = this.resolveCollection(collectionName, false);
            if (collection == null) {
                collection = this.createCollection(collectionName);
            }
            BSONObject result = collection.handleUpdate(update);
            this.putLastResult(channel, result);
        }
        catch (MongoServerException e) {
            log.error("failed to update {}", (Object)update, (Object)e);
            this.putLastError(channel, e);
        }
    }

    @Override
    public BSONObject handleCommand(Channel channel, String command, BSONObject query) throws MongoServerException {
        if (command.equalsIgnoreCase("getlasterror")) {
            return this.commandGetLastError(channel, command, query);
        }
        if (command.equalsIgnoreCase("getpreverror")) {
            return this.commandGetPrevError(channel, command, query);
        }
        if (command.equalsIgnoreCase("reseterror")) {
            return this.commandResetError(channel, command, query);
        }
        this.clearLastStatus(channel);
        if (command.equalsIgnoreCase("count")) {
            return this.commandCount(command, query);
        }
        if (command.equalsIgnoreCase("distinct")) {
            String collectionName = query.get(command).toString();
            MongoCollection collection = this.resolveCollection(collectionName, true);
            return collection.handleDistinct(query);
        }
        if (command.equalsIgnoreCase("drop")) {
            return this.commandDrop(query);
        }
        if (command.equalsIgnoreCase("dropDatabase")) {
            return this.commandDropDatabase();
        }
        if (command.equalsIgnoreCase("dbstats")) {
            return this.commandDatabaseStats();
        }
        if (command.equalsIgnoreCase("collstats")) {
            String collectionName = query.get("collstats").toString();
            MongoCollection collection = this.resolveCollection(collectionName, true);
            return collection.getStats();
        }
        if (command.equalsIgnoreCase("validate")) {
            String collectionName = query.get("validate").toString();
            MongoCollection collection = this.resolveCollection(collectionName, true);
            return collection.validate();
        }
        if (command.equalsIgnoreCase("findAndModify")) {
            String collectionName = query.get(command).toString();
            MongoCollection collection = this.resolveCollection(collectionName, false);
            if (collection == null) {
                collection = this.createCollection(collectionName);
            }
            return collection.findAndModify(query);
        }
        log.error("unknown query: {}", (Object)query);
        throw new NoSuchCommandException(command);
    }

    private BSONObject commandDatabaseStats() throws MongoServerException {
        BasicBSONObject response = new BasicBSONObject("db", (Object)this.getDatabaseName());
        response.put("collections", (Object)this.namespaces.count());
        long indexSize = 0L;
        long objects = 0L;
        long dataSize = 0L;
        double averageObjectSize = 0.0;
        for (MongoCollection collection : this.collections.values()) {
            BSONObject stats = collection.getStats();
            objects += ((Number)stats.get("count")).longValue();
            dataSize += ((Number)stats.get("size")).longValue();
            BSONObject indexSizes = (BSONObject)stats.get("indexSize");
            for (String indexName : indexSizes.keySet()) {
                indexSize += ((Number)indexSizes.get(indexName)).longValue();
            }
        }
        if (objects > 0L) {
            averageObjectSize = (double)dataSize / (double)objects;
        }
        response.put("objects", (Object)objects);
        response.put("avgObjSize", (Object)averageObjectSize);
        response.put("dataSize", (Object)dataSize);
        response.put("storageSize", (Object)0L);
        response.put("numExtents", (Object)0);
        response.put("indexes", (Object)this.indexes.count());
        response.put("indexSize", (Object)indexSize);
        response.put("fileSize", (Object)0);
        response.put("nsSizeMB", (Object)0);
        Utils.markOkay((BSONObject)response);
        return response;
    }

    private BSONObject commandDropDatabase() {
        this.backend.dropDatabase(this);
        BasicBSONObject response = new BasicBSONObject("dropped", (Object)this.getDatabaseName());
        Utils.markOkay((BSONObject)response);
        return response;
    }

    private BSONObject commandDrop(BSONObject query) throws MongoServerException {
        String collectionName = query.get("drop").toString();
        MongoCollection collection = this.collections.remove(collectionName);
        if (collection == null) {
            throw new MongoSilentServerException("ns not found");
        }
        BasicBSONObject response = new BasicBSONObject();
        this.namespaces.removeDocument((BSONObject)new BasicBSONObject("name", (Object)collection.getFullName()));
        response.put("nIndexesWas", (Object)collection.getNumIndexes());
        response.put("ns", (Object)collection.getFullName());
        Utils.markOkay((BSONObject)response);
        return response;
    }

    private void putLastError(Channel channel, MongoServerException ex) {
        BasicBSONObject error = new BasicBSONObject();
        if (ex instanceof MongoServerError) {
            MongoServerError err = (MongoServerError)ex;
            error.put("err", (Object)err.getMessage());
            error.put("code", (Object)err.getCode());
        } else {
            error.put("err", (Object)ex.getMessage());
        }
        error.put("connectionId", (Object)channel.hashCode());
        this.putLastResult(channel, (BSONObject)error);
    }

    private synchronized void putLastResult(Channel channel, BSONObject result) {
        List<BSONObject> results = this.lastResults.get(channel);
        BSONObject last = results.get(results.size() - 1);
        if (last != null) {
            throw new IllegalStateException();
        }
        results.set(results.size() - 1, result);
    }

    private BSONObject commandGetLastError(Channel channel, String command, BSONObject query) throws MongoServerException {
        String subCommand;
        Iterator it = query.keySet().iterator();
        String cmd = (String)it.next();
        if (!cmd.equals(command)) {
            throw new IllegalStateException();
        }
        if (it.hasNext() && !(subCommand = (String)it.next()).equals("w") && !subCommand.equals("fsync")) {
            throw new MongoServerException("unknown subcommand: " + subCommand);
        }
        List<BSONObject> results = this.lastResults.get(channel);
        BasicBSONObject result = null;
        if (results != null && !results.isEmpty()) {
            result = results.get(results.size() - 1);
            if (result == null) {
                result = new BasicBSONObject();
            }
        } else {
            result = new BasicBSONObject();
            result.put("err", null);
        }
        Utils.markOkay((BSONObject)result);
        return result;
    }

    private BSONObject commandGetPrevError(Channel channel, String command, BSONObject query) {
        List<BSONObject> results = this.lastResults.get(channel);
        if (results != null) {
            for (int i = 1; i < results.size(); ++i) {
                BSONObject result = results.get(results.size() - i);
                if (result == null) continue;
                boolean isRelevant = false;
                if (result.get("err") != null) {
                    isRelevant = true;
                } else if (((Number)result.get("n")).intValue() > 0) {
                    isRelevant = true;
                }
                if (!isRelevant) continue;
                result.put("nPrev", (Object)i);
                return result;
            }
        }
        BasicBSONObject result = new BasicBSONObject();
        result.put("nPrev", (Object)-1);
        Utils.markOkay((BSONObject)result);
        return result;
    }

    private BSONObject commandResetError(Channel channel, String command, BSONObject query) {
        List<BSONObject> results = this.lastResults.get(channel);
        if (results != null) {
            results.clear();
        }
        BasicBSONObject result = new BasicBSONObject();
        Utils.markOkay((BSONObject)result);
        return result;
    }

    private BSONObject commandCount(String command, BSONObject query) throws MongoServerException {
        String collection = query.get(command).toString();
        BasicBSONObject response = new BasicBSONObject();
        MongoCollection coll = this.collections.get(collection);
        if (coll == null) {
            response.put("missing", (Object)Boolean.TRUE);
            response.put("n", (Object)0);
        } else {
            response.put("n", (Object)coll.count((BSONObject)query.get("query")));
        }
        Utils.markOkay((BSONObject)response);
        return response;
    }
}

