package de.codecamp.vaadin.security.spring.access;

import java.io.Serializable;
import java.util.Objects;

import com.vaadin.flow.router.Route;

import de.codecamp.vaadin.security.spring.access.SecuredAccess.NotSetAccessEvaluator;



/**
 * Represents the rule determining access. If both an expression and an {@link AccessEvaluator} are
 * specified, both must be satisfied to gain access. At least one of them must be specified.
 */
public class AccessRule
  implements
    Serializable
{

  private final String expression;

  private final Class<? extends AccessEvaluator> evaluator;

  private final boolean checkLayout;


  private AccessRule(SecuredAccess securedAccess)
  {
    this.expression =
        !securedAccess.value().equals(SecuredAccess.EXPRESSION_NOT_SET) ? securedAccess.value()
            : null;
    this.evaluator =
        securedAccess.evaluator() != NotSetAccessEvaluator.class ? securedAccess.evaluator() : null;

    if (expression == null && evaluator == null)
    {
      throw new AccessRuleException(
          "Either a security expression or an AccessEvaluator must be specified.");
    }

    this.checkLayout = securedAccess.checkLayout();
  }

  private AccessRule(String expression, boolean checkLayout)
  {
    Objects.requireNonNull(expression, "expression must not be null");

    this.expression = expression;
    this.evaluator = null;
    this.checkLayout = checkLayout;
  }

  private AccessRule(Class<? extends AccessEvaluator> evaluator, boolean checkLayout)
  {
    Objects.requireNonNull(evaluator, "evaluator must not be null");

    this.expression = null;
    this.evaluator = evaluator;
    this.checkLayout = checkLayout;
  }


  /**
   * A Spring Security expression to control access; may be null.
   * <p>
   * If both an expression and an evaluator are set, both must be satisfied to grant access.
   *
   * @return a Spring Security expression; may be null
   */
  public String expression()
  {
    return expression;
  }

  /**
   * A {@link AccessEvaluator programmatic evaluator} to control access; may be null.
   * <p>
   * If both an expression and an evaluator are set, both must be satisfied to grant access.
   *
   * @return a {@link AccessEvaluator}; may be null
   */
  public Class<? extends AccessEvaluator> evaluator()
  {
    return evaluator;
  }

  /**
   * Returns whether {@link Route#layout() parent layouts} should also be examined for access rules.
   * Only applicable to access to routes.
   *
   * @return whether {@link Route#layout() parent layouts} should also be examined for access rules
   */
  public boolean checkLayout()
  {
    return checkLayout;
  }


  /**
   * Creates a new access rule as copy from a {@link SecuredAccess} annotation.
   *
   * @param securedAccess
   *          the {@link SecuredAccess} annotation
   * @return the access rule
   * @see SecuredAccess
   */
  public static AccessRule asCopyOf(SecuredAccess securedAccess)
  {
    return new AccessRule(securedAccess);
  }

  /**
   * Creates a new access rule based on an expression, checking parent layouts for access rules.
   *
   * @param expression
   *          a Spring Security expression
   * @return the access rule
   * @see SecuredAccess#value()
   */
  public static AccessRule of(String expression)
  {
    return of(expression, true);
  }

  /**
   * Creates a new access rule based on an expression, optionally checking parent layouts for access
   * rules.
   *
   * @param expression
   *          a Spring Security expression
   * @param checkLayout
   *          whether parent layouts should also be checked for access rules
   * @return the access rule
   * @see SecuredAccess#value()
   */
  public static AccessRule of(String expression, boolean checkLayout)
  {
    return new AccessRule(expression, checkLayout);
  }

  /**
   * Creates a new access rule based on an evaluator, checking parent layouts for access rules.
   *
   * @param evaluator
   *          a {@link AccessEvaluator}
   * @return the access rule
   * @see SecuredAccess#evaluator()
   */
  public static AccessRule of(Class<? extends AccessEvaluator> evaluator)
  {
    return of(evaluator, true);
  }

  /**
   * Creates a new access rule based on an evaluator, optionally checking parent layouts for access
   * rules.
   *
   * @param evaluator
   *          a {@link AccessEvaluator}
   * @param checkLayout
   *          whether parent layouts should also be checked for access rules
   * @return the access rule
   * @see SecuredAccess#evaluator()
   */
  public static AccessRule of(Class<? extends AccessEvaluator> evaluator, boolean checkLayout)
  {
    return new AccessRule(evaluator, checkLayout);
  }

}
